/* #Specification: FIELD_PASSTHROUGH / principle
	Linuxconf supports multiple user interface. From a generic DIALOG
	interface, it builds HTML, Text and GUI dialogs. These dialogs are
	uniform, but sometime, one wishes to make the GUI version more
	appealing. The FIELD_PASSTHROUGH is used to record GUI commands.
	See <htmlurl
		url=http://www.solucorp.qc.ca/linuxcong/tech/guiapi
		>

	When a single FIELD_PASSTHROUGH is defined in a dialog, linuxconf
	assume the dialog is layed out manually, so the programmer is
	free to implement about any disposition he wants.

	FIELD_PASSTHROUGH has no effect for the HTML and Text interface.
*/
#include "dialog.h"
#include "../diajava/proto.h"
#include "internal.h"

// Function needed to force the linkage of pass.o
void pass_required(){}

class FIELD_DUMMY: public FIELD{
	/*~PROTOBEG~ FIELD_DUMMY */
public:
	FIELD_DUMMY (void);
	MENU_STATUS dokey (WINDOW *,
		 int ,
		 FIELD_MSG&,
		 bool&);
	void drawtxt (WINDOW *, int);
	const char *get_registry_value (void);
	char getidprefix (void);
	void gui_draw (int , SSTRINGS&);
	MENU_STATUS gui_get (int ,
		 const char *,
		 const char *);
	void html_draw (int);
	int html_validate (int);
	void reload (const char *, int);
	void restore (void);
	void save (void);
	void set_registry_value (const char *);
	/*~PROTOEND~ FIELD_DUMMY */
};

PUBLIC FIELD_DUMMY::FIELD_DUMMY()
	: FIELD ("")
{
	vsize = 0;
	readonly = 1;
}

PUBLIC void FIELD_DUMMY::set_registry_value(const char *){}
PUBLIC const char *FIELD_DUMMY::get_registry_value(){return NULL;}
PUBLIC void FIELD_DUMMY::drawtxt(WINDOW *, int){}
PUBLIC MENU_STATUS FIELD_DUMMY::dokey(WINDOW *, int , FIELD_MSG &, bool &){	return MENU_NULL;}
PUBLIC void FIELD_DUMMY::save(){}
PUBLIC void FIELD_DUMMY::restore(){}
PUBLIC void FIELD_DUMMY::reload(const char *, int ){}
PUBLIC void FIELD_DUMMY::html_draw(int){}
PUBLIC MENU_STATUS FIELD_DUMMY::gui_get(int, const char *, const char * ){return MENU_NULL;}
PUBLIC char FIELD_DUMMY::getidprefix (){return ' '; }
PUBLIC int  FIELD_DUMMY::html_validate(int ){ return 0;}
PUBLIC void FIELD_DUMMY::gui_draw(int, SSTRINGS &){}

class FIELD_PASSTHROUGH: public FIELD_DUMMY{
	int command;
	SSTRING args;
	/*~PROTOBEG~ FIELD_PASSTHROUGH */
public:
	FIELD_PASSTHROUGH (int _command,
		 const char *_args);
	void gui_draw (int , SSTRINGS&tb);
	bool is_passthrough (void);
	/*~PROTOEND~ FIELD_PASSTHROUGH */
};


PUBLIC FIELD_PASSTHROUGH::FIELD_PASSTHROUGH(int _command, const char *_args)
{
	command = _command;
	args.setfrom (_args);
}

PUBLIC void FIELD_PASSTHROUGH::gui_draw(int, SSTRINGS &tb)
{
	if (command == P_End){
		tb.remove_del(tb.getnb()-1);
	}else if (command == P_Form
		|| command == P_Group
		|| command == P_Groupfit
		|| command == P_Book
		|| command == P_Page
		|| command == P_Treemenu){
		SSTRING *s = new SSTRING;
		s->copyword (args.get());
		tb.add (s);
	}else if (command == P_Treesub){
		tb.add (new SSTRING("sub"));
	}
	diagui_sendcmd (command,"%s\n",args.get());
}

PUBLIC bool FIELD_PASSTHROUGH::is_passthrough()
{
	return true;
}


/*
	Record a GUI command. This only affect the GUI.
	Using this override the normal disposition done for the GUI. You
	are on your own...
*/
PUBLIC void DIALOG::gui_passthrough(int command, const char *args, ...)
{
	va_list list;
	va_start (list,args);
	char buf[1000];
	vsnprintf (buf,sizeof(buf)-1,args,list);
	add (new FIELD_PASSTHROUGH(command,buf));
	va_end (list);
}


/*
	Record a GUI command. This only affect the GUI.
	Using this override the normal disposition done for the GUI. You
	are on your own...
*/
PUBLIC void DIALOG::gui_passthroughv(int command, const char *args, va_list list)
{
	char buf[1000];
	vsnprintf (buf,sizeof(buf)-1,args,list);
	add (new FIELD_PASSTHROUGH(command,buf));
}

/*
	Record a line change command. This only affect the GUI.
	Using this override the normal disposition done for the GUI. You
	are on your own...
*/
PUBLIC void DIALOG::newline()
{
	gui_passthrough (P_Newline,"");
}

/*
	Record a label command. This only affect the GUI.
	Accept a va_list.
*/
PUBLIC void DIALOG::gui_labelv(const char *args, va_list list)
{
	char buf[1000];
	vsnprintf (buf,sizeof(buf)-1,args,list);
	add (new FIELD_PASSTHROUGH(P_Label,buf));
}
/*
	Record a label command. This only affect the GUI.
	Using this override the normal disposition done for the GUI. You
	are on your own...
*/
PUBLIC void DIALOG::gui_label(const char *ctl, ...)
{
	va_list list;
	va_start (list,ctl);
	gui_labelv (ctl,list);
	va_end (list);
}
/*
	Record a end of block command. This only affect the GUI.
	Using this override the normal disposition done for the GUI. You
	are on your own...
*/
PUBLIC void DIALOG::gui_end()
{
	gui_passthrough (P_End,"");
}
/*
	Record a grouline change command. This only affect the GUI.
	Using this override the normal disposition done for the GUI. You
	are on your own...
*/
PUBLIC void DIALOG::gui_group(const char *title)
{
	gui_passthrough (P_Group,"gr%d \"%s\"",getnb(),title);
}
PUBLIC void DIALOG::gui_form()
{
	gui_passthrough (P_Form,"f%d",getnb());
}

PUBLIC void DIALOG::gui_groupfit(const char *title)
{
	gui_passthrough (P_Groupfit,"gr%d \"%s\"",getnb(),title);
}

/*
	Control the disposition of the previous object. The object
	may occupy several cells horizontally and vertically and may
	be positionned inside this space.
*/
PUBLIC void DIALOG::gui_dispolast(
	GUI_H_DISPO dispoh,
	int nbcellh,
	GUI_V_DISPO dispov,
	int nbcellv)
{
	static char tbh[]={'l','c','r'};
	static char tbv[]={'t','c','b'};
	gui_passthrough (P_Dispolast,"%c %d %c %d"
		,tbh[dispoh],nbcellh,tbv[dispov],nbcellv);
}

/*
	This class records various flags used by the layout manager and other
	The FIELD::getflag() normally returns false, but for this FIELD_FLAGS
	it reports true, meaning it delivers some flags
*/
class FIELD_FLAGS: public FIELD_DUMMY{
	unsigned flags;
	/*~PROTOBEG~ FIELD_FLAGS */
public:
	FIELD_FLAGS (unsigned _flags);
	bool getflags (unsigned &_flags);
	/*~PROTOEND~ FIELD_FLAGS */
};

PUBLIC FIELD_FLAGS::FIELD_FLAGS (unsigned _flags)
{
	flags = _flags;
}

PUBLIC VIRTUAL bool FIELD::getflags (unsigned &)
{
	return false;
}
PUBLIC bool FIELD_FLAGS::getflags (unsigned &_flags)
{
	_flags = flags;
	return true;
}
/*
	This function only affects the GUI mode.
	Normally, after every field, a DIALOG::newline() is appended, placing
	every fields in a single column.

	When enhancing a complex dialog, one may want to place several
	fields on a single line. To do this, it issues several fields
	and then a DIALOG::newline(). This tells the layout manager
	to turn off auto-newline.

	On small dialogs, this is ok. If there is a single DIALOG::newline
	(or DIALOG::gui_passthrough in fact)
	then auto-newline is off. On more complex dialog, it becomes
	cumbersome. Just for one area of the dialog, you are forced to
	issue DIALOG::newline() all over the place.

	This function record the operation mode for the next fields. You can
	set auto-newline on or off and you can call this function several
	time.
*/
PUBLIC void DIALOG::auto_newline(bool mode)
{
	add (new FIELD_FLAGS (mode ? DIAFLAGS_AUTONEWLINE : DIAFLAGS_NOAUTONEWLINE));
}

