/*
 * $Id: bitmap.c,v 1.5 2001/06/01 02:07:25 antona Exp $
 *
 * bitmap.c - Bitmap handling code. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "types.h"
#include "bitmap.h"

/*
 * NOTES:
 *
 * - Operations are 8-bit only to ensure the functions work both on little
 *   and big endian machines! So don't make them 32-bit ops!
 * - bitmap starts at bit = 0 and ends at bit = bitmap size - 1.
 * - _Caller_ has to make sure that the bit to operate on is less than the
 *   size of the bitmap.
 */

__inline__ void ntfs_set_bit(__u8 *bitmap, const __u64 bit,
			     const __u8 new_value)
{
//	printf("bitmap %p, bit 0x%Lx, new_value %i\n", bitmap, bit, new_value);
	if (!bitmap || new_value > 1)
		return;
	if (!new_value)
		bitmap[bit >> 3] &= ~(1 << (bit & 7));
	else
		bitmap[bit >> 3] |= (1 << (bit & 7));
}

__inline__ char ntfs_get_bit(const __u8 *bitmap, const __u64 bit)
{
	if (!bitmap)
		return -1;
	return (bitmap[bit >> 3] >> (bit & 7)) & 1;
}

__inline__ void ntfs_change_bit(__u8 *bitmap, const __u64 bit)
{
	if (!bitmap)
		return;
	bitmap[bit >> 3] ^= 1 << (bit & 7);
}

__inline__ char ntfs_get_and_set_bit(__u8 *bitmap, const __u64 bit,
			    const __u8 new_value)
{
	register __u8 old_bit, shift;
	
	if (!bitmap || new_value > 1)
		return -1;
	shift = bit & 7;
	old_bit = (bitmap[bit >> 3] >> shift) & 1;
	if (new_value != old_bit)
		bitmap[bit >> 3] ^= 1 << shift; 
	return old_bit;
}

