/*
 * $Id: mft.h,v 1.24 2001/12/15 05:13:08 antona Exp $
 *
 * mft.h - Exports for MFT record handling. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef MFT_H
#define MFT_H

#include "types.h"
#include "list.h"
#include "support.h"

/**
 * get_mft_record_data_size - return number of bytes used in mft record @b
 * @b:		mft record to get the data size of
 *
 * Takes the mft record @b and returns the number of bytes used in the record
 * or 0 on error (i.e. b is not a valid mft record). Zero is not a valid size
 * for an mft record as it at least has to have the MFT_RECORD, thus making the
 * minimum size:
 * 	(sizeof(MFT_RECORD) + 7) & ~7 + sizeof(ATTR_TYPES) = 52 bytes
 * Aside: The 8-byte alignment and the 4 bytes for the attribute type are needed
 * as each mft record has to have a list of attributes even if it only contains
 * the attribute $END which doesn't contain anything else apart from it's type.
 * Also, you would expect every mft record to contain an update sequence array
 * as well but that could in theory be non-existent (don't know if Windows NTFS
 * wouldn't view this as corruption in itself though!).
 */
extern __inline__ __u32 get_mft_record_data_size(const MFT_RECORD *b);

/* Mft entry flag bit values */
#define ME_mapped		0
#define ME_dirty		1
#define ME_error		2
#define ME_is_base_record	3
#define ME_attr_list_present	4
#define ME_attr_list_mapped	5
				/* bits 6-31 reserved for future use */

#define MftEntryMapped(me)		test_bit(ME_mapped, (me)->m_flags)
#define SetMftEntryMapped(me)		set_bit(ME_mapped, (me)->m_flags)
#define ClearMftEntryMapped(me)		clear_bit(ME_mapped, (me)->m_flags)

#define MftEntryDirty(me)		test_bit(ME_dirty, (me)->m_flags)
#define SetMftEntryDirty(me)		set_bit(ME_dirty, (me)->m_flags)
#define ClearMftEntryDirty(me)		clear_bit(ME_dirty, (me)->m_flags)

#define MftEntryError(me)		test_bit(ME_error, (me)->m_flags)
#define SetMftEntryError(me)		set_bit(ME_error, (me)->m_flags)
#define ClearMftEntryError(me)		clear_bit(ME_error, (me)->m_flags)

#define MftEntryIsBaseRecord(me)	test_bit(ME_is_base_record, \
								(me)->m_flags)
#define SetMftEntryIsBaseRecord(me)	set_bit(ME_is_base_record, \
								(me)->m_flags)
#define ClearMftEntryIsBaseRecord(me)	clear_bit(ME_is_base_record, \
								(me)->m_flags)

#define MftEntryAttrListPresent(me)	test_bit(ME_attr_list_present, \
								(me)->m_flags)
#define SetMftEntryAttrListPresent(me)	set_bit(ME_attr_list_present, \
								(me)->m_flags)
#define ClearMftEntryAttrListPresent(me) clear_bit(ME_attr_list_present, \
								(me)->m_flags)

#define MftEntryAttrListMapped(me)	test_bit(ME_attr_list_mapped, \
								(me)->m_flags)
#define SetMftEntryAttrListMapped(me)	set_bit(ME_attr_list_mapped, \
								(me)->m_flags)
#define ClearMftEntryAttrListMapped(me)	clear_bit(ME_attr_list_mapped, \
								(me)->m_flags)

extern int ntfs_flush_mft_entry(mft_entry *m);

/* Internal use only. Use ntfs_set_mft_entry_dirty() instead. */
extern void __ntfs_set_mft_entry_dirty(mft_entry *m);

/**
 * ntfs_set_mft_entry_dirty - set an mft entry dirty
 * @m:		mft entry to set dirty
 *
 * If the mft entry @m is not dirty it is added to the dirty_mft_entries list
 * of the volume to which @m belongs and the dirty flag of the entry is set.
 */
static __inline__ void ntfs_set_mft_entry_dirty(mft_entry *m)
{
	if (!test_and_set_bit(ME_dirty, m->m_flags))
		__ntfs_set_mft_entry_dirty(m);
}

/* Internal use only. Use ntfs_map_mft_entry() instead. */
extern int __ntfs_map_mft_entry(mft_entry *m);

/**
 * ntfs_map_mft_entry - map an mft entry into memory
 * @m:		mft entry to map
 *
 * Increase the use count of the entry @m and map it into memory (if it is not
 * mapped). This function operates on an already existing mft_entry which has
 * been inserted in a volume already. It doesn't need to be associated with a
 * file though. Note that no setting up of the mft_entry is performed or stuff
 * like that. If you need this kind of stuff chances are you should be using
 * ntfs_map_file_entry() instead and your probably don't have the mft_entry
 * yet.
 *
 * Return 0 on success and -ERRNO on error. See __map_mft_entry() for more
 * details on the returned error codes.
 */
static __inline__ int ntfs_map_mft_entry(mft_entry *m)
{
	int err;
	
	m->m_count++;
	if (MftEntryMapped(m))
		return 0;
	if ((err = __ntfs_map_mft_entry(m)) < 0)
		m->m_count--;
	return err;
}

extern mft_entry *ntfs_map_file_entry(ntfs_volume *v, const MFT_REF mref);

/**
 * ntfs_unmap_mft_entry - unmap an mft entry
 * @m:		mft entry to unmap
 *
 * Decrease the use count of the mft entry (only if it is above 1). Do not
 * actually unmap it. We always keep it cached in memory. However, when we
 * have unmapped it and the count has reached zero, the entry can be safely
 * unmapped if it is necessary, but that happens elsewhere.
 */
static __inline__ void ntfs_unmap_mft_entry(mft_entry *m)
{
	if (MftEntryMapped(m) && m->m_count > 1)
		m->m_count--;
}

/**
 * ntfs_unmap_file_entry - unmap a file mft entry
 * @m:		file mft entry to unmap
 *
 * Same as ntfs_unmap_mft_entry(), so we just wrap it.
 */
static __inline__ void ntfs_unmap_file_entry(mft_entry *m)
{
	ntfs_unmap_mft_entry(m);
}

extern ntfs_file *ntfs_open_by_mref(ntfs_volume *vol, const MFT_REF mref);

extern int ntfs_close(ntfs_file *f);

#endif /* defined MFT_H */

