/* lightlab, Copyright (c) 2002 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include "lightlab.h"
#include "scene.h"
#include "teapot.h"
#include "sphere.h"

struct scene_data {
  Display *dpy;
  Window window;
  XWindowAttributes xgwa;
  GLXContext glx_context;
  GLfloat rotx, roty, rotz;
  GLfloat dx, dy, dz;
};


Visual *
get_gl_visual (Display *dpy, int screen_num)
{
# define R GLX_RED_SIZE
# define G GLX_GREEN_SIZE
# define B GLX_BLUE_SIZE
# define D GLX_DEPTH_SIZE
# define I GLX_BUFFER_SIZE
# define DB GLX_DOUBLEBUFFER

  int attrs[][20] = {
    { GLX_RGBA, R, 8, G, 8, B, 8, D, 8, DB, 0 }, /* rgb double */
    { GLX_RGBA, R, 4, G, 4, B, 4, D, 4, DB, 0 },
    { GLX_RGBA, R, 2, G, 2, B, 2, D, 2, DB, 0 },
    { GLX_RGBA, R, 8, G, 8, B, 8, D, 8,     0 }, /* rgb single */
    { GLX_RGBA, R, 4, G, 4, B, 4, D, 4,     0 },
    { GLX_RGBA, R, 2, G, 2, B, 2, D, 2,     0 },
    { I, 8,                       D, 8, DB, 0 }, /* cmap double */
    { I, 4,                       D, 4, DB, 0 },
    { I, 8,                       D, 8,     0 }, /* cmap single */
    { I, 4,                       D, 4,     0 },
    { GLX_RGBA, R, 1, G, 1, B, 1, D, 1,     0 }  /* monochrome */
  };

  int i;
  for (i = 0; i < sizeof(attrs)/sizeof(*attrs); i++)
    {
      XVisualInfo *vi = glXChooseVisual (dpy, screen_num, attrs[i]);
      if (vi)
        {
          Visual *v = vi->visual;
          XFree (vi);
          return v;
        }
    }
  return 0;
}


scene_data *
init_scene (Display *dpy, Window window)
{
  scene_data *sd = (scene_data *) calloc (sizeof(*sd), 1);
  int screen_num = DefaultScreen (dpy);
  XVisualInfo vi_in, *vi_out;
  int out_count;

  sd->dpy = dpy;
  sd->window = window;

  XGetWindowAttributes (dpy, window, &sd->xgwa);

  vi_in.screen = screen_num;
  vi_in.visualid = XVisualIDFromVisual (sd->xgwa.visual);
  vi_out = XGetVisualInfo (dpy, VisualScreenMask|VisualIDMask,
			   &vi_in, &out_count);
  if (! vi_out) abort ();

  sd->glx_context = glXCreateContext (dpy, vi_out, 0, GL_TRUE);
  XSync (sd->dpy, False);

  if (!sd->glx_context)
    {
      fprintf(stderr, "%s: couldn't create GL context for visual 0x%x.\n",
	      progname, (unsigned int) XVisualIDFromVisual (sd->xgwa.visual));
      exit(1);
    }

  glXMakeCurrent (dpy, window, sd->glx_context);

  {
    GLboolean rgba_mode = 0;
    glGetBooleanv(GL_RGBA_MODE, &rgba_mode);
    if (!rgba_mode)
      {
	glIndexi (WhitePixel (dpy, screen_num));
	glClearIndex (BlackPixel (dpy, screen_num));
      }
  }

  sd->rotx = 30;
  sd->roty = 30;
  sd->rotz = 0;

  sd->dy = -2;

  glDrawBuffer(GL_BACK);

  reshape_scene (sd);
  return sd;
}


void
reshape_scene (scene_data *sd)
{
  int width, height;
  GLfloat h;

  XGetWindowAttributes (sd->dpy, sd->window, &sd->xgwa);

  width = sd->xgwa.width;
  height = sd->xgwa.height;
  h = (GLfloat) height / (GLfloat) width;
  glViewport (0, 0, (GLint) width, (GLint) height);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  gluPerspective( 30.0, 1/h, 1.0, 100.0 );
  gluLookAt( 0.0, 0.0, 15.0,
             0.0, 0.0, 0.0,
             0.0, 1.0, 0.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glTranslatef(0.0, 0.0, -15.0);

  glClear(GL_COLOR_BUFFER_BIT);
}


void
init_texture (scene_data *sd, int width, int height, unsigned char *data)
{
  glPixelStorei (GL_UNPACK_ALIGNMENT, 1);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexGeni (GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexGeni (GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0,
                GL_RGBA, GL_UNSIGNED_BYTE, data);
}


static void
make_floor (int wire_p)
{
  GLfloat z = -1.0;
  GLfloat outer = 3;
  int steps = 7;
  GLfloat size = outer/steps;
  int xx, yy;

  glFrontFace(GL_CCW);
  glNormal3f (0, 1, 0);
  for (yy = 0; yy < steps; yy++)
    for (xx = 0; xx < steps; xx++)
      {
        GLfloat x = xx * size - outer/2;
        GLfloat y = yy * size - outer/2;

        GLfloat tx = (GLfloat)xx / steps;
        GLfloat ty = (GLfloat)yy / steps;
        GLfloat tx2 = tx + size/outer;
        GLfloat ty2 = ty + size/outer;

        if ((xx&1) ^ (yy&1)) continue;  /* xor checkerboard */

        glBegin (wire_p ? GL_LINE_LOOP : GL_QUADS);
        glTexCoord2f(tx,  ty);  glVertex3f (x,      z, y);
        glTexCoord2f(tx2, ty);  glVertex3f (x+size, z, y);
        glTexCoord2f(tx2, ty2); glVertex3f (x+size, z, y+size);
        glTexCoord2f(tx,  ty2); glVertex3f (x,      z, y+size);
        glEnd();
      }
}


static void
generate_scene (scene_data *sd, int object, int wire_p)
{
  make_floor (wire_p);

  switch (object)
    {
    case 0:
      glFrontFace(GL_CW);
      if (wire_p)
        wireTeapot(1);
      else
        solidTeapot(1);
      break;

    case 1:
      glFrontFace(GL_CCW);
      glPushMatrix();
      glScalef(1.4, 1.4, 1.4);
      unit_cube (wire_p);
      glPopMatrix();
      break;

    case 2:
      glFrontFace(GL_CCW);
      unit_sphere (32, 32, wire_p);
      break;

    default:
      abort();
      break;
    }
}


void
update_scene (scene_data *sd)
{
  int i;
  int object;
  int wire_p = 0;

  glEnable(GL_DEPTH_TEST);
  glEnable(GL_NORMALIZE);
/*  glEnable(GL_CULL_FACE);*/

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glPushMatrix ();

  for (i = 0; i < 3; i++)
    {
      int light_number = GL_LIGHT0 + i;
      light_info L;
      static GLfloat a[4];

      get_light_info (i, &L);

      if (L.enabled_p)
        glEnable (light_number);
      else
        glDisable (light_number);

      a[0] = L.x; a[1] = L.y; a[2] = L.z; a[3] = L.w;
      glLightfv (light_number, GL_POSITION, a);

      a[3] = 1.0;

      a[0] = L.ambient.r; a[1] = L.ambient.g; a[2] = L.ambient.b;
      glLightfv (light_number, GL_AMBIENT, a);

      a[0] = L.diffuse.r; a[1] = L.diffuse.g; a[2] = L.diffuse.b;
      glLightfv (light_number, GL_DIFFUSE, a);

      a[0] = L.specular.r; a[1] = L.specular.g; a[2] = L.specular.b;
      glLightfv (light_number, GL_SPECULAR, a);
    }


  for (i = 0; i < 1; i++)
    {
      object_info L;
      static GLfloat a[4];

      get_object_info (i, &L);

      switch (L.render)
        {
        case 0: /* smooth */
          glEnable (GL_LIGHTING);
          glShadeModel (GL_SMOOTH);
          break;
        case 1: /* faceted */
          glEnable (GL_LIGHTING);
          glShadeModel (GL_FLAT);
          break;
        case 2: /* wireframe */
          glDisable (GL_LIGHTING);
          glShadeModel (GL_FLAT);
          wire_p = 1;
          break;
        default:
          abort();
          break;
        }

      switch (L.solid_p)
        {
        case 0: glDisable(GL_TEXTURE_2D); break;
        case 1: glEnable (GL_TEXTURE_2D); break;
        default: abort();                 break;
        }

      object = L.type;

      a[3] = 1.0;

      a[0] = L.ambient.r; a[1] = L.ambient.g; a[2] = L.ambient.b;
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, a);

      a[0] = L.diffuse.r; a[1] = L.diffuse.g; a[2] = L.diffuse.b;
      glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, a);

      a[0] = L.specular.r; a[1] = L.specular.g; a[2] = L.specular.b;
      glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, a);

      glMaterialf (GL_FRONT_AND_BACK, GL_SHININESS, L.shininess);
    }


# define INC(a,b) a += b; if (a < 0) a += 360; else if (a >= 360) a -= 360
  INC (sd->rotx, sd->dx);
  INC (sd->roty, sd->dy);
  INC (sd->rotz, sd->dz);
# undef INC

  glRotatef (sd->rotx, 1, 0, 0);
  glRotatef (sd->roty, 0, 1, 0);
  glRotatef (sd->rotz, 0, 0, 1);

  glScalef(3, 3, 3);

  generate_scene (sd, object, wire_p);

  glPopMatrix();

  glFinish();
  glXSwapBuffers(sd->dpy, sd->window);
  glFlush();
}
