// LabPlot : PlotPolar.cc

#include <iostream>
#include <klocale.h>
#include <kdebug.h>
#include "PlotPolar.h"

using namespace std;

//! general polar Plot class
PlotPolar::PlotPolar(Worksheet *p)
	: Plot(p)
{
	QFont font = p->getMainWin()->defaultFont();
	title = new Label(i18n("Polar Plot"),font,QColor(Qt::black));
	title->setPosition(0.4,0.02);

	// set default axis settings for all axes
	for(int i=0;i<2;i++)
		readAxisSettings(&axis[i],PPOLAR,i);

	font.setPointSize((int)(0.7*font.pointSize()));	// for tic label
	axis[0].setTickLabelFont(font);
	axis[1].setTickLabelFont(font);

	axis[0].enableMajorGrid();
	axis[1].enableMajorGrid();

	axis[0].setTickLabelFormat(DEGREE);
	axis[0].setTickLabelPrecision(0);
}

PlotPolar::~PlotPolar() {
	delete graphlist;

	delete title;
	delete markx;
	delete marky;

	delete region;
}

//! Infos for project explorer
QStringList PlotPolar::Info() {
	QStringList s;
	s<<"Polar";

	s<<QString::number(position.X())+QString(" , ")+QString::number(position.Y());
	s<<QString::number(size.X())+QString(" X ")+QString::number(size.Y());
	if (transparent)
		s<<QString("yes");
	else
		s<<QString("no");
	s<<bgcolor.color().name();
	s<<gbgcolor.color().name();

	return s;
}

void PlotPolar::draw(QPainter *p, int w, int h) {
	kdDebug()<<"PlotPolar::draw() w/h : "<<w<<' '<<h<<endl;

	if(aspect_enabled) {	// set aspect ratio to 1
		int wsize = (int) fmin(w,h);
		w=h=wsize;
	}

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	kdDebug()<<"XMIN/MXAX/YMIN/YMAX = "<<xmin<<','<<xmax<<','<<ymin<<','<<ymax<<endl;
	kdDebug()<<"p1 = "<<p1.X()<<'/'<<p1.Y()<<" p2 = "<<p2.X()<<'/'<<p2.Y()<<endl;

	if (!transparent) {
		// background color
		p->setBrush(bgcolor);
		p->setPen(Qt::NoPen);
		p->drawRect((int)(w*position.X()),(int)(h*position.Y()),(int)(w*size.X()),(int)(h*size.Y()));

		// graph background color
		p->setBrush(gbgcolor);
		p->setPen(Qt::NoPen);
	}

	kdDebug()<<"PlotPolar : title->draw() pos:"<<position.X()<<' '<<position.Y()<<endl;
	kdDebug()<<" 			size:"<<size.X()<<' '<<size.Y()<<endl;
	title->draw(worksheet,p,position,size,w,h,0);

	drawCurves(p, w, h);

	if(legend.Enabled())
		legend.draw(p,type,graphlist,position,size,w,h);
	p->setPen(Qt::NoPen);
}

void PlotPolar::drawCurves(QPainter *p, int w, int h) {
	kdDebug()<<"PlotPolar::drawCurves()"<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	// used as miny=0
	//double miny = actrange[1].rMin();
	double maxy = actrange[1].rMax();

	kdDebug()<<"xmin/xmax ymin/ymax : "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;
	kdDebug()<<"width/height : "<<w<<'/'<<h<<endl;

	// TODO : tic color ???
	p->setPen(Qt::black);

	int major_tics = (int) axis[1].MajorTicks();
	int minor_tics = axis[1].MinorTicks();
	if(major_tics==-1)
		major_tics=autoTicks(0,maxy);

	for (int i=0;i<major_tics;i++) {
		int x = xmin+(int)(i*(xmax-xmin)/2.0/major_tics);
		int xw = (xmax-xmin)-(int)(i*(xmax-xmin)/major_tics);
		int y = ymin+(int)(i*(ymax-ymin)/2.0/major_tics);
		int yh =(ymax-ymin)-(int)(i*(ymax-ymin)/major_tics);

		// major tics
		if (axis[1].MajorTicksEnabled()) {
			p->drawLine(x,y+yh/2-5,x,y+yh/2+5);
			p->drawLine(x+xw,y+yh/2-5,x+xw,y+yh/2+5);
			p->drawLine(x+xw/2-5,y,x+xw/2+5,y);
			p->drawLine(x+xw/2-5,y+yh,x+xw/2+5,y+yh);
		}

		// major grid r
		if (axis[1].MajorGridEnabled())
			p->drawEllipse(x,y,xw,yh);

		for (int j=0;j<minor_tics;j++) {
			// TODO : minor tics
			// TODO ; minor grid r
			// if (gridenabled[3])
		}
	}

	// r values
	int xw = xmax-xmin, yh = ymax-ymin;
	int prec = axis[1].TickLabelPrecision();
	TFormat atlf = axis[1].TickLabelFormat();
	if (axis[1].Enabled() && axis[1].tickLabelEnabled())  {
		QFont f = axis[1].TickLabelFont();
		f.setPointSize((int)(f.pointSize()*size.X())); //  scale font size with plot
		QFontMetrics fm(f);
		QColor c = axis[1].TickLabelColor();
		double offset = axis[1].Shift();
		double scaling = axis[1].Scaling();
		double pos = axis[1].TickLabelPosition();

		for (int i=0;i<major_tics+1;i++) {
			double value = i/(double)major_tics*maxy;

			QString label = TicLabel(atlf,prec,axis[1].DateTimeFormat(),scaling*value+offset);

			// apply prefix & suffix
			label.prepend(axis[1].TickLabelPrefix());
			label.append(axis[1].TickLabelSuffix());

			p->save();
			p->translate(xmin+(int)(xw/2+i/(double)major_tics*xw/16)-10+pos,ymin+(int)(yh/2-i/(double)major_tics*yh/2)-15+pos);
			p->rotate(axis[1].TickLabelRotation());
			if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
				p->setPen(c);
				p->setFont(f);

				p->drawText(-fm.width(label)/2,fm.ascent()/2,label);
			}
			else {  // rich text label
				QSimpleRichText *richtext = new QSimpleRichText(label,f);
				QColorGroup cg;
				cg.setColor(QColorGroup::Text,c);

				richtext->draw(p,-richtext->width()/4, -fm.ascent()/2,QRect(),cg);
				delete richtext;
			}
			p->restore();
		}
	}

	// phi values
	if (axis[0].Enabled() && axis[0].tickLabelEnabled()) {
		prec = axis[0].TickLabelPrecision();
		atlf = axis[0].TickLabelFormat();

		QFont f = axis[0].TickLabelFont();
		f.setPointSize((int)(f.pointSize()*size.X()));	//  scale font size with plot
		QFontMetrics fm(f);
		QColor c = axis[0].TickLabelColor();
		double pos = 2.0*axis[0].TickLabelPosition();
		double offset = axis[0].Shift();
		double scaling = axis[0].Scaling();
		double direction = 1.0;
		if(scaling < 0.0) {
			scaling *= -1.0;
			direction = -1.0;
		}
		double rx = xw/2+pos, ry = yh/2+pos;
		for (int i=0;i<12;i++) {
			double phi = 30.0*i*M_PI/180.0;

			QString label = TicLabel(atlf,prec,axis[0].DateTimeFormat(),phi*scaling);

			// apply prefix & suffix
			label.prepend(axis[0].TickLabelPrefix());
			label.append(axis[0].TickLabelSuffix());

			phi += offset*M_PI/180.0;
			phi *= direction;

			p->save();
			p->translate((int)(xmin+xw/2+rx*cos(phi)-5),(int)(ymin+yh/2-ry*sin(phi)));
			p->rotate(axis[0].TickLabelRotation());
			if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
				p->setPen(c);
				p->setFont(f);

				p->drawText(-fm.width(label)/2,fm.ascent()/2,label);
			}
			else {  // rich text label
				QSimpleRichText *richtext = new QSimpleRichText(label,f);
				QColorGroup cg;
				cg.setColor(QColorGroup::Text,c);

				richtext->draw(p,-richtext->width()/4, -fm.ascent()/2,QRect(),cg);
				delete richtext;
			}
			p->restore();
		}
	}

	// axes lines
	p->drawLine(xmin,(int)((ymax+ymin)/2.0),xmax,(int)((ymax+ymin)/2.0));
	p->drawLine((int)((xmax+xmin)/2.0),ymin,(int)((xmax+xmin)/2.0),ymax);

	if (axis[0].MajorGridEnabled()) {	// phi major grid
		double rx = xw/2, ry = yh/2;
		double phi =30.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
		phi =60.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
		phi =120.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
		phi =150.0*M_PI/180.0;
		p->drawLine((int)(xmin+xw/2+rx*cos(phi)),(int)(ymin+yh/2-ry*sin(phi)),
			(int)(xmin+xw/2+rx*cos(phi+M_PI)),(int)(ymin+yh/2-ry*sin(phi+M_PI)));
	}

	// clipping rect with some space (clipoffset)
	p->setClipRect(xmin-clipoffset,ymin-clipoffset,xmax-xmin+2*clipoffset,ymax-ymin+2*clipoffset);

	// draw curves
	for (unsigned int nr=0; nr < graphlist->Number() ; nr++) {
		if(graphlist->getGraph(nr)->isShown() == false)
			continue;

		GRAPHType s = graphlist->getType(nr);
		if (s == GRAPH2D) {
			Graph2D *g = graphlist->getGraph2D(nr);

			kdDebug()<<"GRAPH2D / 2D number="<<g->Number()<<endl;

			Point *d = g->Data();
			int N = g->Number();
			Style *style =  g->getStyle();
			Symbol *symbol = g->getSymbol();

			double oldx=0.0, oldy=0.0;
			QPointArray pa(N);
			int pointindex=0;
			for (int i=0;i<N;i++) {
				if(d[i].Masked())
					continue;

				if(worksheet->getMainWin()->speedMode()) {
					int mod = (int) (N/worksheet->getMainWin()->speedModeValue());
					if(mod==0) mod=1;
					if(i%mod != 0) continue;	// speed mode
				}

				double phi = d[i].X(), r =  d[i].Y();

				int x = xmin+xw/2+(int)(r/maxy*xw/2*cos(phi));
				int y = ymin+yh/2-(int)(r/maxy*yh/2*sin(phi));

//				kdDebug()<<"phi/r = "<<phi<<' '<<r<<endl;
//				kdDebug()<<"x/y = "<<x<<' '<<y<<endl;
				pa[pointindex++]=QPoint(x,y);
				oldx=x;oldy=y;
			}
			pa.resize(pointindex);	// resize to clear all unneeded points (from draw outside border)
			drawStyle(p,style,symbol,pa,xmin,xmax,ymin,ymax);
		}
	}
	p->setClipping(false);
}

void PlotPolar::saveAxes(QTextStream *t) {
	for (int i = 0; i < 2; i++)
		saveAxis(t,&axis[i]);
}

void PlotPolar::openAxes(QTextStream *t, int version) {
	for(int i = 0;i<2;i++)
		openAxis(t,version, &axis[i]);
}

void PlotPolar::saveXML(QDomDocument doc,QDomElement plottag) {
	QDomElement tag;

	for (int i = 0; i < 2; i++) {
		tag = axis[i].saveXML(doc,i);
   		plottag.appendChild( tag );
	}
}

void PlotPolar::openXML(QDomElement e) {
	if(e.tagName() == "Axis")
		axis[e.attribute("id").toInt()].openXML(e.firstChild());
}
