/***************************************************************************
                          wcard.cpp  -  description
                             -------------------
    begin                : Mon Nov 11 2002
    copyright            : (C) 2002 by Matthias Reif
    email                : matthias.reif@informatik.tu-chemnitz.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "wcard.h"

/*!
	\fn WCard::WCard( const char *device )
 */
WCard::WCard( const char *device )
{
	ethFD = socket(AF_INET, SOCK_DGRAM, 0);

	this->device = new char[ 20 ];

	range = 0L;
	stats = 0L;
	stats = new iwstat();
	range = new iwrange();
	
	rangeBuffer = new char[ sizeof(iwrange) * 2 ];
	procBuffer = new char[ PROC_BUFFER_SIZE ];
	
	setDevice( device );
}

/*!
	\fn WCard::~WCard()
 */
WCard::~WCard()
{
	delete device;
	delete procBuffer;
	delete stats;
	delete range;
	delete rangeBuffer;
}

/*!
	\fn WCard::setDevice(const char* name)
 */
void WCard::setDevice(const char* name)
{
	strcpy( device, name );
	setHas();
}


/*!
	\fn WCard::getDevice()
 */
char* WCard::getDevice()
{
	return device;
}


/*!
	\fn WCard::getLinkQual(int &link, int &avg, int &signal, int &noise, int &maxQual, int &maxSignal, int &maxNoise)
 */
int WCard::getLinkQual(int &link, int &avg, int &signal, int &noise, int &maxQual, int &maxSignal, int &maxNoise)
{
	wrq.u.data.flags = 1;
	wrq.u.data.pointer = (caddr_t) stats;
	wrq.u.data.length = sizeof(iwstat);
	if( iwIoctl( SIOCGIWSTATS ) < 0 )
		return -1;

	link = (int)stats->qual.qual;
	signal = (int)stats->qual.level;
	noise = (int)stats->qual.noise;

	if( has.range )
	{
		maxQual = (int)range->max_qual.qual;
		maxSignal = (int)range->max_qual.level;
		maxNoise = (int)range->max_qual.noise;
		avg = (int)range->avg_qual.qual;
	}
	return 0;
}


/*!
	\fn WCard::getNetID()
 */
int WCard::getNetID()
{
	if( iwIoctl( SIOCGIWNWID ) < 0 )
		return -1;
	return wrq.u.nwid.value;
}


/*!
	\fn WCard::getIP()
 */
char* WCard::getIP()
{
	struct ifreq ifrq;
	memset(&ifrq, 0, sizeof(struct ifreq));
	strcpy(ifrq.ifr_name, device);
	if( ioctl(ethFD, SIOCGIFADDR, &ifrq) < 0 )
		return NULL;
	struct sockaddr_in *addr;
	addr = (sockaddr_in*)(&ifrq.ifr_addr);
	return inet_ntoa(addr->sin_addr);
}


/*!
	\fn WCard::setIP( const char *ip )
 */
char* WCard::setIP( const char *ip )
{
	struct ifreq ifrq;
	memset( &ifrq, 0, sizeof( struct ifreq ) );
	
	strcpy( ifrq.ifr_name, device );
	
	if( ioctl( ethFD, SIOCGIFADDR, &ifrq ) < 0 )
	{
		perror("SIOCGIFADDR");
	}
	
	struct sockaddr_in *addr = new struct sockaddr_in;
	addr->sin_family = AF_INET;
	inet_aton( ip, &(addr->sin_addr) );
	
	memcpy( &(ifrq.ifr_addr), addr, sizeof(struct sockaddr) );
	
	strcpy(ifrq.ifr_name, device);
	if( ioctl(ethFD, SIOCSIFADDR, &ifrq) < 0 )
		return NULL;
	
	return getIP();
}


/*!
	\fn WCard::getProtocol(char *protocol)
 */
int WCard::getProtocol(char *protocol)
{
	if( iwIoctl( SIOCGIWNAME ) < 0)
	{
		return -1;
	}
	strncpy(protocol , wrq.u.name, strlen(wrq.u.name)+1 );
	return 0;
}


/*!
	\fn WCard::getMode()
 */
int WCard::getMode()
{
	if( iwIoctl( SIOCGIWMODE ) < 0 )
		return -1;
	return wrq.u.mode;  
}


/*!
	\fn WCard::setMode(int mode)
 */
int WCard::setMode(int mode)
{
	wrq.u.mode = mode;
	if( iwIoctl( SIOCSIWMODE ) < 0 )
		return -1;
	return getMode();
}

/**
	@deprecated
*/
int WCard::getFreq()
{
	if( iwIoctl( SIOCGIWFREQ ) < 0)
		return -1;
	return wrq.u.freq.m;
}


/*!
	\fn WCard::getBitRates( )
 */
vector<int> WCard::getBitRates( )
{
	vector<int> bitrates;
	if( has.range )
	{
		int i;
		for( i = 0; i < range->num_bitrates; i++ )
		{
			bitrates.push_back( range->bitrate[ i ] );
		}
	}
	return bitrates;
}


/*!
	\fn WCard::getBitRate( )
 */
int WCard::getBitRate( )
{
	if( iwIoctl( SIOCGIWRATE ) < 0)
		return -1;

	return wrq.u.bitrate.value;
}



/*!
	\fn WCard::getAPMac(char *s)
 */
int WCard::getAPMac(char *s)
{
	if( iwIoctl( SIOCGIWAP ) < 0 )
		return -1;
	
	struct ether_addr *addr;
	addr = (ether_addr*)(&wrq.u.ap_addr.sa_data);

	sprintf(s, "%02X:%02X:%02X:%02X:%02X:%02X" , addr->ether_addr_octet[0], addr->ether_addr_octet[1], addr->ether_addr_octet[2], addr->ether_addr_octet[3], addr->ether_addr_octet[4], addr->ether_addr_octet[5]);

	return 0;
}



/*!
	\fn WCard::isWave()
 */
bool WCard::isWave()
{
	return ( iwIoctl( SIOCGIWNAME ) >= 0 );
}


/*!
	\fn WCard::getPower( bool& enabled, int& flags )
 */
int WCard::getPower( bool& enabled, int& flags )
{
	wrq.u.power.flags = 0;
	if( iwIoctl( SIOCGIWPOWER ) < 0 )
	{
		return -1;
	}
	enabled = !wrq.u.power.disabled;
	flags = wrq.u.power.flags;
	return 0;
}


/*!
	\fn WCard::getTransmitPower( bool& disabled, bool& fixed, int& unit )
 */
int WCard::getTransmitPower( bool& disabled, bool& fixed, int& unit )
{
	if( iwIoctl( SIOCGIWTXPOW ) < 0 )
	{
		return -1;
	}
	else
	{
		if( wrq.u.txpower.flags & IW_TXPOW_RELATIVE )
			unit = IW_TXPOW_RELATIVE;
		else if( wrq.u.txpower.flags & IW_TXPOW_MWATT )
			unit = IW_TXPOW_MWATT;
		else
			unit = IW_TXPOW_DBM;
			
		disabled = wrq.u.txpower.disabled;
		fixed = wrq.u.txpower.fixed;
		return wrq.u.txpower.value;
	}
}


/*!
	\fn WCard::setTransmitPower( bool disabled, bool fixed, int value, int unit )
 */
int WCard::setTransmitPower( bool disabled, bool fixed, int value, int unit )
{
	if( disabled )
		wrq.u.txpower.disabled = 1;
	else
		wrq.u.txpower.disabled = 0;
	if( fixed )
		wrq.u.txpower.fixed = 1;
	else
		wrq.u.txpower.fixed = 0;
	wrq.u.txpower.value = value;
	wrq.u.txpower.flags = unit;
	return iwIoctl( SIOCSIWTXPOW );
}


/*!
	\fn WCard::getEncode(char* key)
 */
bool WCard::getEncode(char* key)
{
	wrq.u.encoding.pointer = (caddr_t)key;
	wrq.u.encoding.length  = IW_ENCODING_TOKEN_MAX;
	wrq.u.encoding.flags   = 0;

	if( iwIoctl( SIOCGIWENCODE ) >= 0 )
	{
		if( (wrq.u.encoding.flags & IW_ENCODE_DISABLED) == IW_ENCODE_DISABLED)
		{
			return false;
		}
		else
		{
			return true;
		}
	}
	return false;
}


/*!
	\fn WCard::getSensitivity( int &value, int &range )
 */
int WCard::getSensitivity( int &value, int &range )
{
	if( iwIoctl( SIOCGIWSENS ) < 0 )
	{
		return -1;
	}

	value = wrq.u.sens.value;

	if( has.range )
	{
		range = this->range->sensitivity;
	}
	else
	{
		range = -1;
	}
	return 0;
}


/*!
	\fn WCard::setSensitivity( int value )
 */
int WCard::setSensitivity( int value )
{
	wrq.u.sens.value = value;
	return iwIoctl( SIOCSIWSENS );
}


/*!
	\fn WCard::getRTS( iw_param& rts )
 */
int WCard::getRTS( iw_param& rts )
{
	int result = iwIoctl( SIOCGIWRTS );
	rts = wrq.u.rts;
	return result;
}


/*!
	\fn WCard::getFrag( iw_param& frag )
 */
int WCard::getFrag( iw_param& frag )
{
	int result = iwIoctl( SIOCGIWFRAG );
	frag = wrq.u.frag;
	return result;
}


/*!
	\fn WCard::setRTS( iw_param rts )
 */
int WCard::setRTS( iw_param rts )
{
	wrq.u.rts = rts;
	return iwIoctl( SIOCSIWRTS );
}


/*!
	\fn WCard::setFrag( iw_param frag )
 */
int WCard::setFrag( iw_param frag )
{
	wrq.u.frag = frag;
	return iwIoctl( SIOCSIWFRAG );
}


/*!
	\fn WCard::hasLink()
 */
bool WCard::hasLink()
{
	wrq.u.data.length = 0;
	wrq.u.data.flags = 1;
	wrq.u.data.pointer = (caddr_t) stats;
	wrq.u.data.length = sizeof(iwstat);
	if( iwIoctl( SIOCGIWSTATS ) < 0)
		return false;
	else
		return ( (int)stats->qual.qual > 0 );
}


/*!
	\fn WCard::setBitRate( int value )
 */
int WCard::setBitRate( int value )
{
	wrq.u.bitrate.value = value;
	wrq.u.bitrate.fixed = 0;
	return iwIoctl( SIOCSIWRATE );
}


/*!
	\fn WCard::getESSID(char *value)
 */
int WCard::getESSID(char *value)
{
	wrq.u.essid.pointer = (caddr_t)value;
	wrq.u.essid.length  = IW_ENCODING_TOKEN_MAX;
	wrq.u.essid.flags   = 0;
	
	if( iwIoctl( SIOCGIWESSID ) >= 0 )
	{
		value[ wrq.u.essid.length ] = '\0';
		return 0;
	}
	else
	{
		return -1;
	}
}


/*!
	\fn WCard::setESSID(const char *value)
 */
int WCard::setESSID(const char *value)
{
	char *buffer = (char*)malloc(strlen(value)+1);
	strcpy( buffer, value );
	if( strlen(buffer) == 0 || strncmp( buffer, "any", strlen(buffer) ) == 0 )
	{
		buffer[0] = '\0';
	}

	wrq.u.essid.pointer = (caddr_t)buffer;
	wrq.u.essid.length  = strlen(buffer)+1;
	wrq.u.essid.flags   = 1;

	int returnValue = iwIoctl( SIOCSIWESSID );
	free(buffer);
	return returnValue;
}


/*!
	\fn WCard::getTraffic( double &sent, double &recv )
 */
bool WCard::getTraffic( double &sent, double &recv )
{
	ifstream procFile( "/proc/net/dev" );
	if( ! procFile.good() )
		return false;
	
	char* device = new char[ strlen( this->device ) + 3 ];
	strcpy( device, this->device );
	strcat( device, ":\0" );
	
	bool found = false;
	char* foundStart = NULL;
	while( procFile.good() && !found)
	{
		procFile.getline( procBuffer, PROC_BUFFER_SIZE );
		found = (foundStart = strstr( procBuffer, device ) ) != NULL;
	}
	
	procFile.close();
	
	if( found )
	{
		foundStart += strlen( device );
		stringstream deviceStream;
		deviceStream << foundStart;
		
		deviceStream >> recv;
		int i;
		for( i = 0; i < 8; i++ )
			deviceStream >> sent;
	}

	delete device;
	return found;
}


/*!
	\fn WCard::getNetworkID()
 */
int WCard::getNetworkID()
{
	if( iwIoctl( SIOCGIWNWID ) < 0)
		return -1;
	else
		return wrq.u.nwid.value;
}


/*!
	\fn WCard::getNickName( char *name )
 */
int WCard::getNickName( char *name )
{
	wrq.u.data.pointer = (caddr_t)name;
	wrq.u.data.length = 99;
	wrq.u.data.flags = 0;

	int result;
	result = iwIoctl( SIOCGIWNICKN );
	if( result >= 0 )
	{
		name[ wrq.u.data.length ] = '\0';
	}
	return result;
}


/*!
	\fn WCard::setNickName( const char *name )
 */
int WCard::setNickName( const char *name )
{
	wrq.u.data.pointer = (caddr_t)name;
	wrq.u.data.length = strlen( name ) + 1;
	wrq.u.data.flags = 0;
	return iwIoctl( SIOCSIWNICKN );
}


/*!
	\fn WCard::getRetryLimit(int &value)
 */
int WCard::getRetryLimit(int &value)
{
	if( iwIoctl( SIOCGIWRETRY ) >= 0 )
	{
		value = wrq.u.retry.value;
		return 0;
	}
	else
	{
		return -1;
	}
}

/*!
	\fn WCard::getRange()
 */
int WCard::getRange()
{
	memset( rangeBuffer, 0, sizeof(iwrange) * 2 );
	wrq.u.data.pointer = (caddr_t) rangeBuffer;
	wrq.u.data.length = sizeof(iwrange) * 2;
	wrq.u.data.flags = 0;
	if( iwIoctl( SIOCGIWRANGE ) < 0)
	{
		return -1;
	}
	memcpy( (char *) range, rangeBuffer, sizeof(iwrange) );
	return 0;
}


/*!
    \fn WCard::setHas()
 */
void WCard::setHas()
{
	strncpy( wrq.ifr_name, device, IFNAMSIZ );
	has.range = ( getRange() >= 0 );
	
	has.sens = (ioctl(ethFD , SIOCGIWSENS, &wrq) >= 0);
	has.ap_addr = (ioctl(ethFD , SIOCGIWAP, &wrq) >= 0);
	char* nickname = new char[ IW_ESSID_MAX_SIZE + 1 ];
	wrq.u.data.pointer = (caddr_t)nickname;
	wrq.u.data.length = IW_ESSID_MAX_SIZE + 1;
	wrq.u.data.flags = 0;
	has.nickname = (ioctl(ethFD , SIOCGIWNICKN, &wrq) >= 0) && ( wrq.u.data.length > 1 );
	has.bitrate = (ioctl(ethFD , SIOCGIWRATE, &wrq) >= 0);
	has.rts = (ioctl(ethFD , SIOCGIWRTS, &wrq) >= 0);
	has.frag = (ioctl(ethFD , SIOCGIWFRAG, &wrq) >= 0);
	has.power = (ioctl(ethFD , SIOCGIWPOWER, &wrq) >= 0);
	has.txPower = (ioctl(ethFD , SIOCGIWTXPOW, &wrq) >= 0);
	has.retry = (ioctl(ethFD , SIOCGIWRETRY, &wrq) >= 0);
	wrq.u.data.flags = 1;
	wrq.u.data.pointer = (caddr_t) stats;
	wrq.u.data.length = sizeof(iwstat);
	has.stats = (ioctl(ethFD , SIOCGIWSTATS, &wrq) >= 0);
	delete [] nickname;
	
//	printf("version: %d\n", range->we_version_compiled);
}


/*!
    \fn WCard::getFreguencyAndChannel( double &frequency, int &channel, int &numChannel )
 */
int WCard::getFreguencyAndChannel( double &frequency, int &channel, int &numChannel )
{
	if( iwIoctl( SIOCGIWFREQ ) < 0 )
		return -1;
	frequency = frequency2Float(wrq.u.freq);
	if( has.range )
	{
		channel = frequency2Channel( frequency, range );
	}
	else
	{
		channel = -1;
	}
	numChannel = range->num_frequency;
	return 0;
}


/*!
    \fn WCard::channel2Frequency( int channel, double &frequency, const struct iw_range *range )
 */
int WCard::channel2Frequency( int channel, double &frequency, const struct iw_range *range )
{
	bool hasFrequency = false;
	int i;
	for( i = 0; i < range->num_frequency; i++)
	{
		if( ( range->freq[i].e != 0) || (range->freq[i].m > 1000 ) )
			hasFrequency = true;
	}
	if( ! hasFrequency )
	{
		return -1;
	}
	for( i = 0; i < range->num_frequency; i++)
	{
		if( range->freq[i].i == channel )
		{
			frequency = ( (double) range->freq[i].m ) * pow( (double)10.0, range->freq[i].e );
			return channel;
		}
	}
	return -2;
}

/*!
	\fn WCard::float2Frequency( double in, iw_freq* out )
*/
void WCard::float2Frequency( double in, iw_freq* out )
{
	out->e = (short)(floor(log10(in)));
	if(out->e > 8)
	{
		out->m = ((long) (floor(in / powf(10,out->e - 6)))) * 100;
		out->e -= 8;
	}
	else
	{
		out->m = (long) in;
		out->e = 0;
	}
}

/*!
	\fn WCard::frequency2Float( iw_freq freq )
*/
double WCard::frequency2Float( iw_freq freq )
{
	return ((double)freq.m)*pow( (double)10.0,freq.e);
}

/*!
    \fn WCard::scan()
 */
APInfos WCard::scan( int & error, int & errorNo )
{
	unsigned char* buffer = NULL;
	int bufferLength = IW_SCAN_MAX_DATA;
	struct timeval tv;
	int timeout = 5000000;
	
	APInfos results;
	
	if( ( ! has.range) || ( range->we_version_compiled < 14 ) )
	{
		error = 1;
		errorNo = errno;
		return results;
	}
	
	tv.tv_sec = 0;
	tv.tv_usec = 250000;
	
	wrq.u.data.pointer = NULL;
	wrq.u.data.flags = 0;
	wrq.u.data.length = 0;

	if( iwIoctl( SIOCSIWSCAN ) < 0 )
	{
		if(errno != EPERM)
		{
			error = 2;
		}
		else
		{
			error = 3;
		}
		errorNo = errno;
		return results;
	}

	timeout -= tv.tv_usec;
	
	bool ok = false;
	do
	{
		fd_set fdSet;
		int maxFd;
		int returnValue;
	
		FD_ZERO(&fdSet);
		maxFd = -1;
		
		returnValue = select( maxFd + 1, &fdSet, NULL, NULL, &tv );
		
		if( returnValue < 0 )
		{
			if( ( errno != EAGAIN ) && ( errno != EINTR ) )
			{
				error = 4;
				errorNo = errno;
				return results;
			}
		}
		else if( returnValue == 0 )
		{
			unsigned char* newBuffer;
			bool toSmall;
			do
			{
				toSmall = false;
				newBuffer = (unsigned char*)realloc( buffer, bufferLength );
				if( newBuffer == NULL )
				{
					if( buffer != NULL )
						free( buffer );
					error = 5;
					errorNo = errno;
					return results;
				}
				buffer = newBuffer;
				wrq.u.data.pointer = (char*)buffer;
				wrq.u.data.flags = 0;
				wrq.u.data.length = bufferLength;
				
				if( iwIoctl( SIOCGIWSCAN ) < 0 )
				{
					if( ( errno == E2BIG ) && (range->we_version_compiled > 16) )
					{
						if( wrq.u.data.length > bufferLength )
							bufferLength = wrq.u.data.length;
						else
							bufferLength *= 2;
						toSmall = true;
					}
					else if( errno == EAGAIN )
					{
						tv.tv_sec = 0;
						tv.tv_usec = 100000;
						timeout -= tv.tv_usec;
						if( timeout <= 0 )
						{
							free( buffer );
							error = 6;
							errorNo = errno;
							return results;
						}
					}
					else
					{
						free( buffer );
						error = 6;
						errorNo = errno;
						return results;
					}
				}
				else
				{
					ok = true;
				}
				
			}while( toSmall );
		}
	
	}while( ! ok );
	
	if( wrq.u.data.length )
	{
		APInfo aPInfo;
		struct iw_event event;
		struct stream_descr stream;
		int returnValue;

		memset( (char *)&stream, '\0', sizeof(struct stream_descr) );
		stream.current = (char*)buffer;
		stream.end = (char*)buffer + wrq.u.data.length;
		
		do
		{
			returnValue = extractEventStream( &stream, &event, range->we_version_compiled );
			if( returnValue > 0 )
			{
/*				if( event.cmd == SIOCGIWAP )
				{
					results.push_back( aPInfo );
					aPInfo.clear();
				}*/
				
				// copy essid 'cause pointer seems to be temporary
				if( event.cmd == SIOCGIWESSID )
				{
					if( ( event.u.essid.pointer ) && ( event.u.essid.length ) )
					{
						char* essid_c = new char[ IW_ESSID_MAX_SIZE + 1 ];
						memcpy( essid_c, event.u.essid.pointer, event.u.essid.length );
						essid_c[ event.u.essid.length ] = '\0';
						event.u.essid.pointer = essid_c;
					}
				}
				
				if ( ( event.cmd == IWEVCUSTOM ) || ( event.cmd == SIOCGIWENCODE ) )
				{
					if( ( event.u.data.pointer ) && ( event.u.data.length ) )
					{
						char* dataBuffer = new char[ IW_CUSTOM_MAX + 1 ];
						memcpy( dataBuffer, event.u.data.pointer, event.u.data.length );
						if( event.cmd == IWEVCUSTOM )
							dataBuffer[ event.u.data.length ] = '\0';
						event.u.data.pointer = dataBuffer;
					}
				}
				
				aPInfo.push_back( event );
			}
		}while( returnValue > 0 );
		
		results.push_back( aPInfo );
	}
	
	free( buffer );
	error = 0;
	return results;
}


/*!
    \fn WCard::iwIoctl( int code )
 */
int WCard::iwIoctl( int code )
{
	strncpy( wrq.ifr_name, device, IFNAMSIZ );
	return ioctl(ethFD, code, &wrq);
}


/*!
    \fn WCard::extractEventStream(struct stream_descr* stream, struct iw_event* event, int we_version )
 */
int WCard::extractEventStream(struct stream_descr* stream, struct iw_event* event, int we_version )
{
	/* =================
		from iwlist.c
	================== */
	int event_type = 0;
	unsigned int event_len = 1;		/* Invalid */
	char* pointer;
	/* Don't "optimise" the following variable, it will crash */
	unsigned cmd_index;		/* *MUST* be unsigned */
	
	/* Check for end of stream */
	if((stream->current + IW_EV_LCP_LEN) > stream->end)
		return(0);
	
	/* Extract the event header (to get the event id).
	* Note : the event may be unaligned, therefore copy... */
	memcpy((char *) event, stream->current, IW_EV_LCP_LEN);
	
	/* Check invalid events */
	if(event->len <= IW_EV_LCP_LEN)
		return(-1);
	
	/* Get the type and length of that event */
	if(event->cmd <= SIOCIWLAST)
	{
		cmd_index = event->cmd - SIOCIWFIRST;
		if(cmd_index < standard_ioctl_num)
			event_type = standard_ioctl_hdr[cmd_index];
	}
	else
	{
		cmd_index = event->cmd - IWEVFIRST;
		if(cmd_index < standard_event_num)
			event_type = standard_event_hdr[cmd_index];
	}
	/* Unknown events -> event_type=0 => IW_EV_LCP_LEN */
	event_len = event_type_size[event_type];
	/* Fixup for later version of WE */
	if((we_version > 18) && (event_type == IW_HEADER_TYPE_POINT))
		event_len -= IW_EV_POINT_OFF;
	
	/* Check if we know about this event */
	if(event_len <= IW_EV_LCP_LEN)
	{
		/* Skip to next event */
		stream->current += event->len;
		return(2);
	}
	event_len -= IW_EV_LCP_LEN;
	
	/* Set pointer on data */
	if(stream->value != NULL)
		pointer = stream->value;			/* Next value in event */
	else
		pointer = stream->current + IW_EV_LCP_LEN;	/* First value in event */
	
	/* Copy the rest of the event (at least, fixed part) */
	if((pointer + event_len) > stream->end)
	{
		/* Go to next event */
		stream->current += event->len;
		return(-2);
	}
	/* Fixup for later version of WE */
	if((we_version > 18) && (event_type == IW_HEADER_TYPE_POINT))
		memcpy((char *) event + IW_EV_LCP_LEN + IW_EV_POINT_OFF, pointer, event_len);
	else
		memcpy((char *) event + IW_EV_LCP_LEN, pointer, event_len);
	
	/* Skip event in the stream */
	pointer += event_len;
	
	/* Special processing for iw_point events */
	if(event_type == IW_HEADER_TYPE_POINT)
	{
		/* Check the length of the payload */
		if((event->len - (event_len + IW_EV_LCP_LEN)) > 0)
			/* Set pointer on variable part (warning : non aligned) */
			event->u.data.pointer = pointer;
		else
			/* No data */
			event->u.data.pointer = NULL;
			/* Go to next event */
		stream->current += event->len;
	}
	else
	{
		/* Is there more value in the event ? */
		if((pointer + event_len) <= (stream->current + event->len))
			/* Go to next value */
			stream->value = pointer;
		else
		{
			/* Go to next event */
			stream->value = NULL;
			stream->current += event->len;
		}
	}
	return(1);
}


/*!
    \fn WCard::getChannel( double frequency )
 */
int WCard::getChannel( double frequency )
{
	if( frequency < KILO )
		return -1;
	double refFreq;
	int i;
	if( !has.range )
		return -3;
	for( i = 0; i < range->num_frequency; i++ )
	{
		refFreq = ( (double) range->freq[ i ].m ) * pow( (double)10.0, range->freq[ i ].e );
		if( frequency == refFreq )
			return( range->freq[ i ].i );
	}
	/* Not found */
	return -2;
}


/*!
    \fn WCard::setChannel( double channel, bool fixed )
 */
int WCard::setChannel( double channel, bool fixed )
{
	if( fixed )
		wrq.u.freq.flags = IW_FREQ_FIXED;
	else
		wrq.u.freq.flags = 0;
	float2Frequency( channel, &(wrq.u.freq) );
	return iwIoctl( SIOCSIWFREQ );
}


/*!
    \fn WCard::setPower( bool enable, int mode )
 */
int WCard::setPower( bool enable, int mode )
{
	if( !enable )
	{
		wrq.u.power.disabled = 1;
	}
	else
	{
		wrq.u.power.disabled = 0;
		wrq.u.power.flags = IW_POWER_ON;
		wrq.u.power.flags |= mode;
	}
	return iwIoctl( SIOCSIWPOWER );
}


/*!
    \fn WCard::frequency2Channel( double frequency, const struct iw_range* range )
 */
int WCard::frequency2Channel( double frequency, const struct iw_range* range )
{
	if( frequency < KILO )
	{
		return (int)frequency;
	}
	int i;
	for( i = 0; i < range->num_frequency; i++ )
	{
		if( frequency == frequency2Float( range->freq[ i ] ) )
			return( range->freq[i].i );
	}
	return(-2);
}


/*!
    \fn WCard::getStats( iw_discarded& discarded, int& missedBeacon )
 */
bool WCard::getStats( iw_discarded& discarded, int& missedBeacon )
{
	discarded.nwid = 10;
	discarded.code = 10;
	discarded.fragment = 10;
	discarded.retries = 10;
	discarded.misc = 10;
	missedBeacon = 10;


	ifstream procFile( "/proc/net/wireless" );
	if( ! procFile.good() )
		return false;
	
	char* device = new char[ strlen( this->device ) + 3 ];
	strcpy( device, this->device );
	strcat( device, ":\0" );
	
	bool found = false;
	char* foundStart = NULL;
	while( procFile.good() && !found)
	{
		procFile.getline( procBuffer, PROC_BUFFER_SIZE );
		found = (foundStart = strstr( procBuffer, device ) ) != NULL;
	}
	
	procFile.close();
	
	if( found )
	{
		foundStart += strlen( device );
		stringstream deviceStream;
		deviceStream << foundStart;
		
		char buffer[ 32 ];
		int dummy;
		
		// status
		deviceStream >> buffer;
		// 3 times quality
		deviceStream >> dummy;
		deviceStream >> dummy;
		deviceStream >> dummy;
		
		deviceStream >> discarded.nwid;
		deviceStream >> discarded.code;
		deviceStream >> discarded.fragment;
		deviceStream >> discarded.retries;
		deviceStream >> discarded.misc;
		deviceStream >> missedBeacon;
	}
	
	delete device;
	return found;
}


/*!
    \fn WCard::getWeVersion()
 */
int WCard::getWeVersion()
{
	FILE* file;
	char* p;
	int  version;

	/* Check if /proc/net/wireless is available */
	file = fopen( "/proc/net/wireless", "r" );

	if( file == NULL )
	{
		return -1;
	}

	int PROC_BUFFER_SIZE = 256;
	char* procBuffer = new char[ PROC_BUFFER_SIZE ];

	/* Read the first line of buffer */
	fgets( procBuffer, PROC_BUFFER_SIZE, file );

	if( strstr( procBuffer, "| WE" ) == NULL )
	{
		/* Prior to WE16, so explicit version not present */

		/* Black magic */
		if( strstr( procBuffer, "| Missed" ) == NULL )
		{
			version = 11;
		}
		else
		{
			version = 15;
		}
		fclose( file );
		delete procBuffer;
		return version;
	}

	/* Read the second line of buffer */
	fgets( procBuffer, PROC_BUFFER_SIZE, file );

	/* Get to the last separator, to get the version */
	p = strrchr( procBuffer, '|' );
	if( ( p == NULL ) || ( sscanf( p + 1, "%d", &version ) != 1 ) )
	{
		fclose( file );
		delete procBuffer;
		return -1;
	}

	fclose( file );
	delete procBuffer;
	return version;
}


/*!
    \fn WCard::getGateway()
 */
char* WCard::getGateway()
{
	ifstream file( "/proc/net/route" );
	char* gateway = NULL;
	while( file.good() && ( gateway == NULL ) )
	{
		file.getline( procBuffer, PROC_BUFFER_SIZE );
		file >> procBuffer;
		if( strcmp( procBuffer, device ) == 0 )
		{
			file >> procBuffer;
			if( strcmp( procBuffer, "00000000" ) == 0 )
			{
				procBuffer[0] = '0';
				procBuffer[1] = 'x';
				file >> (procBuffer+2);
				ip.s_addr = strtol( procBuffer, NULL, 16 );
				gateway = inet_ntoa( ip );
			}
		}
	}
	file.close();
	return gateway;
}


/*!
    \fn WCard::setGateway( const char* ip )
 */
int WCard::setGateway( const char* ip )
{
	struct rtentry entry = createGatewayStruct( ip );
	return ioctl( ethFD, SIOCADDRT, &entry );
}


/*!
    \fn WCard::deleteGateway( const char* ip )
 */
int WCard::deleteGateway( const char* ip )
{
	struct rtentry entry = createGatewayStruct( ip );
	return ioctl( ethFD, SIOCDELRT, &entry );
}


/*!
    \fn WCard::createGatewayStruct( const char* ip )
 */
struct rtentry WCard::createGatewayStruct( const char* ip )
{
	struct rtentry entry;
	struct sockaddr_in address;
	address.sin_family = PF_INET;
	address.sin_port = 0;
	inet_aton( ip, &address.sin_addr );
	memcpy( &entry.rt_gateway, &address, sizeof( struct sockaddr_in ) );
	inet_aton( "0.0.0.0\0", &address.sin_addr );
	memcpy( &entry.rt_dst, &address, sizeof( struct sockaddr_in ) );
	memcpy( &entry.rt_genmask, &address, sizeof( struct sockaddr_in ) );
	entry.rt_flags = RTF_UP | RTF_GATEWAY;
	entry.rt_metric = 1;
	entry.rt_dev = device;
	return entry;
}


/*!
    \fn WCard::getKeys()
 */
vector< EKey > WCard::getKeys()
{
	vector< EKey > keys;
	int i;
	for( i = 1; i <= range->max_encoding_tokens; i++ )
	{
		EKey key;
		key.key = new unsigned char[ IW_ENCODING_TOKEN_MAX ];
		
		wrq.u.data.pointer = (caddr_t) key.key;
		wrq.u.data.length = IW_ENCODING_TOKEN_MAX;
		wrq.u.data.flags = i;
		if( iwIoctl( SIOCGIWENCODE ) >= 0 )
		{
			if( ( wrq.u.data.flags & IW_ENCODE_DISABLED ) || ( wrq.u.data.length == 0 ) )
			{
				key.length = 0;
				delete [] key.key;
			}
			else
			{
				key.length = wrq.u.data.length;
				key.formatedKey = new char[ key.length * 3 + 1 ];
				formatKey( &key, wrq.u.data.flags );
			}
			keys.push_back( key );
		}
	}
	return keys;
}


/*!
    \fn WCard::formatKey( EKey* key, int keyFlags )
 */
void WCard::formatKey( EKey* key, int keyFlags )
{
	char* buffer = key->formatedKey;
	if( keyFlags & IW_ENCODE_NOKEY )
	{
		if( key->length <= 0 )
		{
			strcpy( key->formatedKey, "on" );
		}
		else
		{
			strcpy( buffer, "**" );
			buffer +=2;
			int i;
			for( i = 1; i < key->length; i++ )
			{
				if( ( i & 0x1 ) == 0 )
				{
					strcpy( buffer++, "-" );
				}
				strcpy( buffer, "**" );
				buffer += 2;
			}
		}
	}
	else
	{
		sprintf( buffer, "%.2X", key->key[0] );
		buffer +=2;
		int i;
		for( i = 1; i < key->length; i++ )
		{
			if( ( i & 0x1 ) == 0 )
			{
				strcpy( buffer++, "-" );
			}
			sprintf( buffer, "%.2X", key->key[ i ] );
			buffer += 2;
		}
	}
}


/*!
    \fn WCard::getEncoding()
 */
Encoding WCard::getEncoding()
{
	Encoding encoding;
	unsigned char key[IW_ENCODING_TOKEN_MAX];
	wrq.u.data.pointer = (caddr_t) key;
	wrq.u.data.length = IW_ENCODING_TOKEN_MAX;
	wrq.u.data.flags = 0;
	if( iwIoctl( SIOCGIWENCODE ) < 0 )
	{
		encoding.key = -1;
	}
	else
	{
		encoding.key = wrq.u.data.flags & IW_ENCODE_INDEX;
		if( ( wrq.u.data.flags & IW_ENCODE_DISABLED ) || ( wrq.u.data.length == 0 ) )
			encoding.mode = IW_ENCODE_DISABLED;
		else if( wrq.u.data.flags & IW_ENCODE_RESTRICTED )
			encoding.mode = IW_ENCODE_RESTRICTED;
		else
			encoding.mode = IW_ENCODE_OPEN;
	}
	return encoding;
}


/*!
    \fn WCard::setEncoding( int mode, int index, unsigned char* key, int length )
 */
int WCard::setEncoding( int mode, int index, unsigned char* key, int length )
{
	wrq.u.data.length = length;
	wrq.u.data.pointer = (caddr_t) key;
	wrq.u.data.flags = mode;
	wrq.u.data.flags |= index & IW_ENCODE_INDEX;
	if(wrq.u.data.pointer == NULL)
	{
		wrq.u.data.flags |= IW_ENCODE_NOKEY;
	}
	return iwIoctl( SIOCSIWENCODE );
}


/*!
    \fn WCard::setEncoding( int index )
 */
int WCard::setEncoding( int index )
{
	wrq.u.data.pointer = (caddr_t) NULL;
	wrq.u.data.flags = 0;
	wrq.u.data.length = 0;
	wrq.u.data.flags |= index & IW_ENCODE_INDEX;
	wrq.u.data.flags |= IW_ENCODE_NOKEY;
	return iwIoctl( SIOCSIWENCODE );
}

/*!
    \fn WCard::setEncodingMode( int mode )
 */
int WCard::setEncodingMode( int mode )
{
	wrq.u.data.pointer = (caddr_t) NULL;
	wrq.u.data.flags = 0;
	wrq.u.data.length = 0;
	wrq.u.data.flags |= mode;
	return iwIoctl( SIOCSIWENCODE );
}

