/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: progress.c
 */

#include <unistd.h>
#include <sys/types.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "progress.h"
#include "commit.h"
#include "main.h"
#include "logging.h"

/*
 * Track the number of progress callbacks we have that
 * are still with us.
 */
static guint progress_callbacks_started = 0;

guint get_progress_callbacks_started(void)
{
	return progress_callbacks_started;
}

/*
 *
 *   inline void reset_progress_bar (GtkProgressBar *)
 *   
 *   Description:
 *      This routine resets/clears the progress bar.
 * 
 *   Entry:
 *      progress_bar - the id of progress bar to reset
 *
 *   Exit:
 *      See description.
 *
 */
inline void reset_progress_bar(GtkProgressBar * progress_bar)
{
	gtk_progress_set_show_text(GTK_PROGRESS(progress_bar), FALSE);
	gtk_progress_set_activity_mode(GTK_PROGRESS(progress_bar), FALSE);
	gtk_progress_bar_update(progress_bar, 0.0);
}

/*
 *
 *   inline void update_indeterminate_progress (GtkProgressBar *progress_bar, gfloat *, gint *)
 *   
 *   Description:
 *      This routine updates a progress bar so that it incrementally
 *      move the progress bar back and forth.
 * 
 *   Entry:
 *      progress_bar - the id of the main progress bar
 *      pos          - the address of the progress bar position to update
 *      orientation  - the address of the progress bar direction to update
 *
 *   Exit:
 *      The pos and orientation are updated as needed
 *
 */
inline void update_indeterminate_progress(GtkProgressBar * progress_bar, gfloat * pos,
					  gint * orientation)
{
	gfloat position = *pos;

	/*
	 * Make updates in increments of 5%. Once we get to 100%,
	 * reverse direction and start progress at 5%.
	 */

	position += 0.05;

	if (position >= 0.99) {
		if (*orientation == 0)
			gtk_progress_bar_set_orientation(progress_bar, GTK_PROGRESS_RIGHT_TO_LEFT);
		else
			gtk_progress_bar_set_orientation(progress_bar, GTK_PROGRESS_LEFT_TO_RIGHT);

		*orientation = !(*orientation);
		position = 0.05;
	}

	gtk_progress_bar_update(progress_bar, position);

	*pos = position;
}

/*
 *
 *   gboolean update_status_bar_text_from_main_thread (gchar *)
 *
 *   Description:
 *      This routine is a one-time execution idle function that 
 *      updates the status bar while running under the control of
 *      the main event loop. 
 *
 *   Entry:
 *      text - text to place in the status bar and then free
 *
 *   Exit:
 *      The current message in the status bar is removed and the
 *      given text is added.
 *
 */
gboolean update_status_bar_text_from_main_thread(gchar * text)
{
	gdk_threads_enter();

	remove_status_bar_message();
	set_status_bar_message(g_strstrip(text));

	gdk_threads_leave();

	g_free(text);

	return FALSE;
}

/*
 *
 *   inline void adjust_progress_bar_properties_as_necessary (GtkProgress *, progress_t *)
 *   
 *   Description:
 *      This routine modifies, as necessary, attributes of the progress bar
 *      in order to support the type of progress updates we need to make.
 * 
 *   Entry:
 *      gtk_progress - the id of the GTK progress bar (cast as superclass)
 *      progress     - the address of the structure that contains all
 *                     we need to about the operation progress
 *
 *   Exit:
 *      Any progress bar properties are changed as needed.
 *
 */
inline void adjust_progress_bar_properties_as_necessary(GtkProgress * gtk_progress,
							progress_t * progress)
{
	if (progress->type == INDETERMINATE) {
		if (!gtk_progress->activity_mode)
			gtk_progress_set_activity_mode(gtk_progress, TRUE);

		if (gtk_progress->show_text)
			gtk_progress_set_show_text(gtk_progress, FALSE);
	} else {
		gchar *format;

		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(gtk_progress),
						 GTK_PROGRESS_LEFT_TO_RIGHT);

		/*
		 * Make sure that we are not running in activity mode but in percentage
		 * mode and that we are showing text and in the proper format.
		 */

		if (gtk_progress->activity_mode)
			gtk_progress_set_activity_mode(gtk_progress, FALSE);

		if (!gtk_progress->show_text) {
			gtk_progress_set_show_text(gtk_progress, TRUE);
			gtk_progress_set_text_alignment(gtk_progress, 0.5, 0.5);
		}

		if (progress->type == DISPLAY_PERCENT)
			format = "%p%%";
		else
			format = "%v of %u";

		gtk_progress_set_format_string(gtk_progress, format);
	}
}

/*
 *
 *   static gboolean indeterminate_progress_timer (progress_t *)
 *   
 *   Description:
 *      This routine is called periodically to update the
 *      main progress bar mainly for lengthy operations.
 * 
 *   Entry:
 *      progress_bar - the id of the progress bar
 *
 *   Exit:
 *      Update the progress bar if we are the current
 *      progress callback.
 *
 */
static gboolean indeterminate_progress_timer(progress_t * progress)
{
	GtkProgressBar *progress_bar;
	static gfloat pos = 0;
	static gint orientation = 0;

	progress_bar = GTK_PROGRESS_BAR(get_progress_bar_id());

	gdk_threads_enter();

	/*
	 * Update the progress bar as long as our id is equal to the
	 * progress callbacks that are running/started. In other words,
	 * if there has been another progress callback started, we skip
	 * our updates until we are back at our level.
	 */

	if (progress->id == progress_callbacks_started) {
		adjust_progress_bar_properties_as_necessary(GTK_PROGRESS(progress_bar), progress);
		update_indeterminate_progress(progress_bar, &pos, &orientation);
	}

	gdk_threads_leave();

	return TRUE;
}

/*
 *
 *   inline void process_progress_update (GtkProgressBar *, progress_t *)
 *   
 *   Description:
 *      This routine updates the progress of some operation.
 * 
 *   Entry:
 *      progress_bar - the id of the progress bar
 *      progress     - the address of the structure that contains all
 *                     we need to about the operation progress
 *
 *   Exit:
 *      The progress is updated.
 *
 */
inline void process_progress_update(GtkProgressBar * progress_bar, progress_t * progress)
{
	if (progress->title) {
		gchar *text;

		if (progress->remaining_seconds != 0) {
			gchar *title;
			guint seconds = progress->remaining_seconds;
			guint minutes;
			guint hours;

			hours = seconds / 3600;
			seconds %= 3600;
			minutes = seconds / 60;
			seconds %= 60;

			title = g_strdup(progress->title);

			if (hours != 0) {
				text =
				    g_strdup_printf(_("%s  (Time remaining:  %02u:%02u:%02u)"),
						    g_strstrip(title), hours, minutes, seconds);
			} else {
				text =
				    g_strdup_printf(_("%s  (Time remaining:  %02u:%02u)"),
						    g_strstrip(title), minutes, seconds);
			}
			g_free(title);
		} else {
			text = g_strdup(progress->title);
		}
		remove_status_bar_message();
		set_status_bar_message(g_strstrip(text));

		g_free(text);
	}

	if (progress->type != INDETERMINATE) {
		adjust_progress_bar_properties_as_necessary(GTK_PROGRESS(progress_bar), progress);

		if (progress->type == DISPLAY_PERCENT) {
			gfloat percentage;

			if (progress->total_count == 0)
				percentage = 0.0;
			else
				percentage =
				    (gfloat) progress->count / (gfloat) progress->total_count;

			gtk_progress_set_percentage(GTK_PROGRESS(progress_bar), percentage);
		} else {
			gtk_progress_set_value(GTK_PROGRESS(progress_bar),
					       (gfloat) progress->count);
		}
	}
}

/*
 *
 *   inline void process_progress_start (GtkProgressBar *, progress_t *)
 *   
 *   Description:
 *      This routine starts the reporting of the progress of some
 *      operation.
 * 
 *   Entry:
 *      progress_bar - the id of the progress bar
 *      progress     - the address of the structure that contains all
 *                     we need to about the operation progress
 *
 *   Exit:
 *      The progress reporting is started.
 *
 */
inline void process_progress_start(GtkProgressBar * progress_bar, progress_t * progress)
{
	progress_callbacks_started++;

	progress->id = progress_callbacks_started;

	if (progress->type == DISPLAY_COUNT)
		gtk_progress_configure(GTK_PROGRESS(progress_bar),
				       (gfloat) progress->count,
				       0.0, (gfloat) progress->total_count);
	else if (progress->type == DISPLAY_PERCENT)
		gtk_progress_configure(GTK_PROGRESS(progress_bar),
				       (gfloat) progress->count, 0.0, 1.0);
	else {
		guint handler_id;

		/*
		 * Start a timer to move the progress bar back and forth. If the callback is on
		 * a thread other than the main event loop then it will look cool. Otherwise,
		 * we are at the mercy of the plug-in and how often it calls us back for updates as
		 * to allow the timeout function to get to run.
		 */

		handler_id =
		    gtk_timeout_add(50, (GtkFunction) indeterminate_progress_timer, progress);

		progress->ui_private_data = GUINT_TO_POINTER(handler_id);
	}

	process_progress_update(progress_bar, progress);
}

/*
 *
 *   inline void process_progress_stop (GtkProgressBar *, progress_t *)
 *   
 *   Description:
 *      This routine stops the reporting of the progress of some
 *      operation.
 * 
 *   Entry:
 *      progress_bar - the id of the progress bar
 *      progress     - the address of the structure that contains all
 *                     we need to about the operation progress
 *
 *   Exit:
 *      The last progress update is done and the progress bar is
 *      is reset.
 *
 */
inline void process_progress_stop(GtkProgressBar * progress_bar, progress_t * progress)
{
	process_progress_update(progress_bar, progress);

	if (progress->type == INDETERMINATE)
		gtk_timeout_remove(GPOINTER_TO_UINT(progress->ui_private_data));

	if (progress_callbacks_started == 1) {
		if (is_commit_in_progress()) {
			gtk_progress_set_activity_mode(GTK_PROGRESS(progress_bar), TRUE);
			gtk_progress_set_show_text(GTK_PROGRESS(progress_bar), FALSE);
		} else
			reset_progress_bar(progress_bar);
	}

	progress_callbacks_started--;
}

/*
 *
 *   gint progress_callback (progress_t *)
 *   
 *   Description:
 *      This routine is called by the engine to start, update,
 *      or stop the reporting of the progress of some operation.
 * 
 *   Entry:
 *      progress - the address of the structure that contains all
 *                 we need to about the operation progress
 *
 *   Exit:
 *      The progress is initiated, updated or stopped.
 *
 */
gint progress_callback(progress_t * progress)
{
	gboolean is_main_thread;
	GtkProgressBar *progress_bar = GTK_PROGRESS_BAR(get_progress_bar_id());

	is_main_thread = is_main_event_loop_thread();

	if (!is_main_thread)
		gdk_threads_enter();

	if (progress->id == 0)
		process_progress_start(progress_bar, progress);
	else if (progress->count < progress->total_count)
		process_progress_update(progress_bar, progress);
	else
		process_progress_stop(progress_bar, progress);

	if (is_main_thread)
		while (gtk_events_pending())
			gtk_main_iteration_do(FALSE);
	else
		gdk_threads_leave();

	return 0;
}
