/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: handlemmgr.c
 *
 */

#include <stdlib.h>
#include <errno.h>

#include "fullengine.h"
#include "handlemgr.h"
#include "engine.h"

/* Structure to hold information about a handle */
typedef struct handle_entry_s {
	object_handle_t         handle;
	void                  * thing;
	object_type_t           type;
	struct handle_entry_s * next;
} handle_entry_t;


/* Structure for an entry in the hash table */
typedef struct hash_table_entry_s {
	u_int32_t        serial;
	handle_entry_t * head;
} hash_table_entry_t;


/*
 * The hash table size should be prime number.  A prim number promotes
 * a more even distribution of entries in the hash table.
 * Other "prime" candidates that are near a power of 10 or power of 2):
 * 257, 499, 503, 509, 521, 997, 1009, 1021, 1031
 */
#define HASH_TABLE_SIZE 127

/* A handle consists of an index into the hash table and a serial number */
#define HANDLE_INDEX_BITS 8
#define HANDLE_INDEX_MASK ((1 << HANDLE_INDEX_BITS) - 1)
#define HANDLE_SERIAL_BITS (32 - INDEX_BITS)

#define MAKE_HANDLE(index, serial) (index + 1 + (serial << HANDLE_INDEX_BITS))
#define GET_HASH_INDEX(handle) ((handle & HANDLE_INDEX_MASK) - 1)


static hash_table_entry_t * hash_table = NULL;

/*
 * Initialize the handle manager.
 * The main task here is to allocate the hash table.
 *
 * Return TRUE if the allocation as successful, else return FALSE.
 */
boolean initialize_handle_manager(void) {

	boolean result = TRUE;

	LOG_PROC_ENTRY();

	/* Has the Handle Manager already been initialized? */
	if (hash_table == NULL) {
		hash_table = (hash_table_entry_t *) calloc(HASH_TABLE_SIZE, sizeof(hash_table_entry_t));

		if (hash_table == NULL)
			result =  FALSE;
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


/*
 * Hash value computer.  Modeled after ElfHash().
 */
static u_int32_t hash_value_for_address(void * address) {

	u_int32_t h = 0;
	u_int32_t g;
	int i;
	u_char * pByte = (u_char *) &address;

	for (i = 0; i < sizeof(address); i++) {
		h = (h << 4) + pByte[i];
		g = h & 0xF0000000;

		if (g != 0) {
			h ^= g >> 24;
		}

		h &= ~g;
	}

	return h;
}


/*
 * Create a user handle for a thing.  The object type is
 * associated with the handle.
 */
int create_handle(void * thing, object_type_t type, object_handle_t * handle) {
	int              rc = HANDLE_MANAGER_NO_ERROR;
	handle_entry_t * handle_entry;
	u_int32_t        hash_index;

	LOG_PROC_ENTRY();

	/* Initialize return handle in case we bail out. */

	*handle = 0;

	/* Has the hash_table been created yet? */
	if (hash_table != NULL) {

		/* Allocate a new handle entry. */
		handle_entry = malloc(sizeof(handle_entry_t));

		if (handle_entry != NULL) {

			/* Get the hash index for this new handle entry. */
			hash_index = hash_value_for_address(handle_entry) % HASH_TABLE_SIZE;

			handle_entry->thing = thing;
			handle_entry->type  = type;
			handle_entry->next  = NULL;

			/*
			 * The handle is a combination of the hash index and the current
			 * serial number at that index.  Increment the serial number for
			 * the next handle entry that will go in the same hash slot.
			 */
			handle_entry->handle = MAKE_HANDLE(hash_index, hash_table[hash_index].serial++);

			/* Put the handle entry into the hash table. */
			handle_entry->next = hash_table[hash_index].head;
			hash_table[hash_index].head = handle_entry;

			/* Return the new handle. */
			*handle = handle_entry->handle;

		} else {
			rc = ENOMEM;
		}

	} else {
		/*
		 * There is no hash_table, which means that this module has not been
		 * initialized yet!
		 */
		rc = HANDLE_MANAGER_NOT_INITIALIZED;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Destroy a user handle.  Remove it from the hash table so it will never be
 * found again.
 */
int destroy_handle(object_handle_t handle) {
	int               rc = HANDLE_MANAGER_NO_ERROR;
	u_int32_t         hash_index;
	handle_entry_t ** pp_handle_entry;

	LOG_PROC_ENTRY();

	/* Has the hash_table been created yet? */
	if (hash_table != NULL) {

		/* Get the index of the hash table entry that should have this handle.*/
		hash_index = GET_HASH_INDEX(handle);

		if (hash_index < HASH_TABLE_SIZE) {

			/*
			 * Look for a handle_entry with a handle that matches the one
			 * given.
			 */
			pp_handle_entry = &hash_table[hash_index].head;
			while ((*pp_handle_entry != NULL) && ((*pp_handle_entry)->handle != handle)) {
				pp_handle_entry = &((*pp_handle_entry)->next);
			}

			/*
			 * If we found a matching handle_entry, remove it from the hash table,
			 * and free the handle_entry.
			 */
			if (*pp_handle_entry != NULL) {
				handle_entry_t * p_handle_entry = *pp_handle_entry;

				*pp_handle_entry = (*pp_handle_entry)->next;
				free(p_handle_entry);

			} else {
				/* Bad handle.  It was not found in the table. */
				rc = HANDLE_MANAGER_BAD_HANDLE;
			}

		} else {
			/*
			 * The handle is not valid.  Its index goes beyond the end of the
			 * hash table.
			 */
			rc = HANDLE_MANAGER_BAD_HANDLE;
		}

	} else {
		/*
		 * There is no hash_table, which means that this module has not been
		 * initialized yet!
		 */
		rc = HANDLE_MANAGER_NOT_INITIALIZED;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Destroy all the handle entries in the hash table.
 */
int destroy_all_handles(void) {
	int rc = HANDLE_MANAGER_NO_ERROR;
	int i;
	handle_entry_t * handle_entry;

	LOG_PROC_ENTRY();

	/* Has the hash_table list been created yet? */
	if (hash_table != NULL) {

		/*
		 * Loop through all the hash_table entries and free any handle entries
		 * that are found.  The serial numbers are left alone, just in case
		 * new handle entries will be added later.  Leaving the serial numbers
		 * will help ensure that future handles are different from past handles.
		 */
		for (i = 0; i < HASH_TABLE_SIZE; i ++) {
			while (hash_table[i].head != NULL) {
				handle_entry = hash_table[i].head;
				hash_table[i].head = handle_entry->next;
				free(handle_entry);
			}
		}

	} else {
		/*
		 * There is no hash_table, which means that this module has not been
		 * initialized yet!
		 */
		rc = HANDLE_MANAGER_NOT_INITIALIZED;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Given a handle, get the thing and its type.
 */
int translate_handle(object_handle_t handle, void * thing, object_type_t * type) {
	int rc = HANDLE_MANAGER_NO_ERROR;

	u_int32_t        hash_index;
	handle_entry_t * handle_entry;

	LOG_PROC_ENTRY();

	/* Has the hash_table list been created yet? */
	if (hash_table != NULL) {

		/* Get the index of the hash table entry that should have this handle.*/
		hash_index = GET_HASH_INDEX(handle);

		if (hash_index < HASH_TABLE_SIZE) {
			/*
			 * Look for a handle_entry with a handle that matches the one
			 * given. */
			handle_entry = hash_table[hash_index].head;
			while ((handle_entry != NULL) && (handle_entry->handle != handle)) {
				handle_entry = handle_entry->next;
			}

			/*
			 * If a matching handle entry was found, return the information
			 * stored in the handle entry.
			 */
			if (handle_entry != NULL) {
				*((char * *) thing) = (char *) handle_entry->thing;
				*type  = handle_entry->type;

			} else {
				/* Bad handle.  It was not found in the table. */
				rc = HANDLE_MANAGER_BAD_HANDLE;
			}

		} else {
			/*
			 * The handle is not valid.  Its index goes beyond the end of the
			 * hash table.
			 */
			rc = HANDLE_MANAGER_BAD_HANDLE;
		}

	} else {
		/*
		 * There is no hash_table, which means that this module has not been
		 * initialized yet!
		 */
		rc = HANDLE_MANAGER_NOT_INITIALIZED;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}
