/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: pixmap.c
 */

#include <frontend.h>

#include <gtk/gtk.h>

#include "pixmap.h"
#include "xpm/unchecked.xpm"
#include "xpm/checkmark.xpm"
#include "xpm/red_checkmark.xpm"
#include "xpm/disk.xpm"
#include "xpm/partition.xpm"
#include "xpm/container.xpm"
#include "xpm/region.xpm"
#include "xpm/object.xpm"
#include "xpm/volume.xpm"
#include "xpm/volume_open.xpm"
#include "xpm/plugin.xpm"
#include "xpm/refresh.xpm"
#include "xpm/info.xpm"
#include "xpm/error.xpm"
#include "xpm/warning.xpm"
#include "xpm/question.xpm"
#include "xpm/topmost.xpm"
#include "logging.h"

static GdkPixmap *unchecked_pixmap;
static GdkBitmap *unchecked_pixmap_mask;
static GdkPixmap *checkmark_pixmap;
static GdkBitmap *checkmark_pixmap_mask;
static GdkPixmap *red_checkmark_pixmap;
static GdkBitmap *red_checkmark_pixmap_mask;
static GdkPixmap *disk_pixmap;
static GdkBitmap *disk_pixmap_mask;
static GdkPixmap *partition_pixmap;
static GdkBitmap *partition_pixmap_mask;
static GdkPixmap *container_pixmap;
static GdkBitmap *container_pixmap_mask;
static GdkPixmap *region_pixmap;
static GdkBitmap *region_pixmap_mask;
static GdkPixmap *object_pixmap;
static GdkBitmap *object_pixmap_mask;
static GdkPixmap *volume_pixmap;
static GdkBitmap *volume_pixmap_mask;
static GdkPixmap *volume_open_pixmap;
static GdkBitmap *volume_open_pixmap_mask;
static GdkPixmap *plugin_pixmap;
static GdkBitmap *plugin_pixmap_mask;
static GdkPixmap *info_pixmap;
static GdkBitmap *info_pixmap_mask;
static GdkPixmap *error_pixmap;
static GdkBitmap *error_pixmap_mask;
static GdkPixmap *warning_pixmap;
static GdkBitmap *warning_pixmap_mask;
static GdkPixmap *question_pixmap;
static GdkBitmap *question_pixmap_mask;
static GdkPixmap *refresh_pixmap;
static GdkBitmap *refresh_pixmap_mask;
static GdkPixmap *topmost_pixmap;
static GdkBitmap *topmost_pixmap_mask;

/*
 *
 *   void create_standard_pixmaps (GtkWidget *)
 *
 *   Description:
 *      This routine creates some of the often seen and used
 *      GdkPixmaps. This routine should be called once from
 *      either main() or an init routine called by main().
 *      The GdkPixmap and GdkBitmap pointers should be accessed
 *      through one of the get functions from this file.
 * 
 *   Entry:
 *      widget - used to provide colormap to
 *               gdk_pixmap_colormap_create_from_xpm_d().
 *
 *   Exit:
 *      Creates the GdkPixmaps and saves the pointers in the file
 *      scope static variables defined above.
 *
 */
void create_standard_pixmaps (GtkWidget *widget)
{
    GdkColormap *colormap;
    
    colormap = gtk_widget_get_colormap (widget);

    unchecked_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, 
                                                              &unchecked_pixmap_mask,
                                                              NULL, unchecked_xpm);
    
    unchecked_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, 
                                                              &unchecked_pixmap_mask,
                                                              NULL, unchecked_xpm);
    
    checkmark_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                              &checkmark_pixmap_mask, 
                                                              NULL, checkmark_xpm);

    red_checkmark_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, 
                                                                  &red_checkmark_pixmap_mask, 
                                                                  NULL, red_checkmark_xpm);

    disk_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, 
                                                         &disk_pixmap_mask, 
                                                         NULL, disk_xpm);

    partition_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, 
                                                              &partition_pixmap_mask, 
                                                              NULL, partition_xpm);

    container_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                              &container_pixmap_mask, 
                                                              NULL, container_xpm);

    region_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                           &region_pixmap_mask, 
                                                           NULL, region_xpm);

    object_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                           &object_pixmap_mask, 
                                                           NULL, object_xpm);

    volume_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                           &volume_pixmap_mask, 
                                                           NULL, volume_xpm);

    volume_open_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                                &volume_open_pixmap_mask, 
                                                                NULL, volume_open_xpm);

    plugin_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                           &plugin_pixmap_mask, 
                                                           NULL, plugin_xpm);
                                                           
    refresh_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                            &refresh_pixmap_mask, 
                                                            NULL, refresh_xpm);
                                                            
    info_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                         &info_pixmap_mask, 
                                                         NULL, info_xpm);
                                                            
    error_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                          &error_pixmap_mask, 
                                                          NULL, error_xpm);
                                                            
    warning_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                            &warning_pixmap_mask,
                                                            NULL, warning_xpm);
                                                            
    question_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                             &question_pixmap_mask,
                                                             NULL, question_xpm);
                                                             
    topmost_pixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap,
                                                            &topmost_pixmap_mask,
                                                            NULL, topmost_xpm);
}

/*
 *
 *   void get_pixmap_for_object_type (object_type_t, GdkPixmap **, GdkBitmap **)
 *   
 *   Description:
 *      This routine returns the GdkPixmap * and GdkBitmap *
 *      corresponding to the pixmap and mask for the icon
 *      representing the given object type.
 * 
 *   Entry:
 *      type   - the object type, e.g. DISK, SEGMENT, or REGION
 *      pixmap - the address of where to store the pointer to the pixmap
 *      mask   - the address of where to store the pointer to the mask
 *
 *   Exit:
 *      The pointers are set with the proper pixmap and mask pointers
 *      allocated at init time.
 *
 */
void get_pixmap_for_object_type (object_type_t type, GdkPixmap **pixmap, GdkBitmap **mask)
{
    switch (type)
    {
        case VOLUME:
            *pixmap = volume_pixmap;
            *mask   = volume_pixmap_mask;
            break;

        case EVMS_OBJECT:
            *pixmap = object_pixmap; 
            *mask   = object_pixmap_mask;
            break;

        case REGION:
            *pixmap = region_pixmap;
            *mask   = region_pixmap_mask;
            break;

        case CONTAINER:
            *pixmap = container_pixmap; 
            *mask   = container_pixmap_mask;
            break;

        case SEGMENT:
            *pixmap = partition_pixmap;
            *mask   = partition_pixmap_mask;
            break;

        case DISK:
            *pixmap = disk_pixmap;
            *mask   = disk_pixmap_mask;
            break;

        case PLUGIN:
            *pixmap = plugin_pixmap;
            *mask   = plugin_pixmap_mask;
            break;

        default:
            *pixmap = NULL;
            *mask   = NULL;
            log_warning ("%s: Unknown type %d.\n", __FUNCTION__, type);
            break;
    }
}

/*
 *
 *   void get_dialog_pixmap (dialog_pixmap_type_t, GdkPixmap **, GdkBitmap **)
 *   
 *   Description:
 *      This routine returns the GdkPixmap * and GdkBitmap *
 *      for a given dialog pixmap type.
 * 
 *   Entry:
 *      type   - INFO_PIXMAP, ERROR_PIXMAP, WARNING_PIXMAP, or QUESTION_PIXMAP
 *      pixmap - the address of where to store the pointer to the pixmap
 *      mask   - the address of where to store the pointer to the mask
 *
 *   Exit:
 *      The pointers are set with the proper pixmap and mask pointers
 *      allocated at init time.
 *
 */
void get_dialog_pixmap (dialog_pixmap_type_t type, GdkPixmap **pixmap, GdkBitmap **mask)
{
    switch (type)
    {
        case INFO_PIXMAP:
            *pixmap = info_pixmap;
            *mask   = info_pixmap_mask;
            break;

        case WARNING_PIXMAP:
            *pixmap = warning_pixmap; 
            *mask   = warning_pixmap_mask;
            break;

        case ERROR_PIXMAP:
            *pixmap = error_pixmap;
            *mask   = error_pixmap_mask;
            break;

        case QUESTION_PIXMAP:
            *pixmap = question_pixmap; 
            *mask   = question_pixmap_mask;
            break;

        default:
            *pixmap = NULL;
            *mask   = NULL;
            log_warning ("%s: Unknown type %d.\n", __FUNCTION__, type);
            break;
    }
}

void get_checkmark_box_pixmap (gboolean critical, gboolean checked, GdkPixmap **pixmap, GdkBitmap **mask)
{
    if (checked)
    {
        if (critical)
        {
            *pixmap = red_checkmark_pixmap;
            *mask   = red_checkmark_pixmap_mask;
        }
        else
        {
            *pixmap = checkmark_pixmap;
            *mask   = checkmark_pixmap_mask;
        }
    }
    else
    {
        *pixmap = unchecked_pixmap;
        *mask   = unchecked_pixmap_mask;
    }
}

void get_open_volume_pixmap (GdkPixmap **pixmap, GdkBitmap **mask)
{
    *pixmap = volume_open_pixmap;
    *mask   = volume_open_pixmap_mask;
}

void get_refresh_pixmap (GdkPixmap **pixmap, GdkBitmap **mask)
{
    *pixmap = refresh_pixmap;
    *mask   = refresh_pixmap_mask;
}

void get_topmost_pixmap (GdkPixmap **pixmap, GdkBitmap **mask)
{
    *pixmap = topmost_pixmap;
    *mask   = topmost_pixmap_mask;
}

void set_clist_row_pixmap (GtkCList *clist, gint row, object_type_t type)
{
    GdkBitmap *mask;
    GdkPixmap *pixmap;

    get_pixmap_for_object_type (type, &pixmap, &mask);
    gtk_clist_set_pixmap (clist, row, 0, pixmap, mask);
}
