/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of raid1 MD physical volumes and logical
 *              volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#define MY_PLUGIN raid1_plugin
#include "md.h"
#include "raid1_discover.h"



int raid1_create_region(md_volume_t * volume, dlist_t output_list, boolean final_call){
	int rc = 0;
	storage_object_t * region;
	int found = 0;
	int i, j = -1;
	int active_disks = 0;

	LOG_ENTRY;

	if ((!volume->super_block || (volume->nr_disks !=  volume->super_block->nr_disks)) &&
	    !final_call) {
		LOG_DETAILS("Region %s. missing members, delaying discovery\n",volume->name);
		RETURN(0);
	}

	LOG_DETAILS("Discovered region %s.\n",volume->name);
	if ((rc = EngFncs->allocate_region(volume->name, &region))){
		for (j = MAX_MD_MINORS -1;(rc != 0) && (j >=0) ; j--) {
			sprintf(volume->name, "md/md%d",j);
			rc = EngFncs->allocate_region(volume->name, &region);
		}
		if (j<0) {
			LOG_ERROR("No more names for MD ");
			RETURN(ENOMEM);
		}
	}
	region->size = -1;  // initialize for min calculation
	for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
		// check for null object, if missing, skip and set corrupt flag
		if (volume->child_object[i]) {
			// if name registration failed and we changed the name, fix up all the minor numbers
			if (j >= 0) {
				volume->super_array[i]->md_minor = j+1;
			}
			md_append_region_to_object(region, volume->child_object[i]);
			LOG_DETAILS("Adding Object %s.to %s\n",volume->child_object[i]->name,volume->name);
			region->size = min(MD_NEW_SIZE_SECTORS(volume->child_object[i]->size),region->size);
			found++;
			if (volume->super_array[i]->this_disk.state && (1 << MD_DISK_ACTIVE)) {
				active_disks++;
			}
		}else {
			MESSAGE("Error building region %s. Missing member object %d\n",volume->name,i);
			volume->flags |= MD_DEGRADED; // set degrade for now.
		}
	}

	if (!active_disks) {
		volume->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;
	}
	region->data_type = DATA_TYPE;
	region->plugin = raid1_plugin;
	region->private_data = (void *)volume;
	if ((volume->flags & MD_DIRTY) && !(volume->flags & MD_CORRUPT)) {
		// mark region dirty
		region->flags |= SOFLAG_DIRTY;
		EngFncs->set_changes_pending();
	}
	volume->flags |= MD_DISCOVERED;
	volume->region = region;
	if (raid1_verify_and_fix_array(volume, 0)){
		int    answer = 0;
		char * choice_text[3] = { "Don't Fix", "Fix", NULL };
		EngFncs->user_message(my_plugin, &answer,choice_text,
			"MD region %s has inconsistent metadata.  Any missing objects "
			"will be *PERMANENTLY* removed from the region and all super "
			"blocks will be updated.  If you elect to not fix the region "
			"at this time, you may do so later by using \"modify properties\" "
			"for the region.  Changes will not be written to disk until Commit\n\n",
			volume->name);
		if (answer) {
			raid1_verify_and_fix_array(volume, 1);
		} else {
	  		volume->flags |= MD_DEGRADED;
	        }
	}

	md_add_object_to_list(region, output_list);
	RETURN(rc);
}




/* Function: discover_regions
 *
 *	run the global list of regions and pirce them together.
 */
int raid1_discover_regions( dlist_t output_list, int *count, boolean final_call)
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	LOG_ENTRY;

	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == RAID1)) {
			rc = raid1_create_region(volume, output_list, final_call);
			if (volume->flags & MD_DISCOVERED) {
				*count = *count + 1;
			}
		}
		volume = volume->next;
	}



	RETURN(rc);
}


// verify the raid 1 array.  If 'fix' is 0 then just perform validation, return 0 if
// array is OK, 1 if array needs to be fixed.
// If 'fix' is 1, then fix up the array on the fly and return 1 if array was modified,
// 0 if untouched.

int  raid1_verify_and_fix_array(md_volume_t * volume, int fix){

	int     i,k;
	int 	change = 0;
	int 	nr_disks = 0, raid_disks = 0, spare_disks = 0, working_disks=0, active_disks=0;
	int 	failed_disks=0;
	int 	major, minor;
	mdp_disk_t disk;
	mdp_disk_t disk2;

	LOG_ENTRY;

	for (i = 0; i < MAX_MD_DEVICES && nr_disks < volume->nr_disks; i++ ) {
		if (!volume->child_object[i]) {
			// ok, found a hole
			change= TRUE;
			if (!fix) {	  // not allowed to fix, quit now.
				MESSAGE("Region %s missing raid array object %d. "
				"Possible identifier of missing object is Major=%d Minor=%d\n",
				volume->name, i, volume->super_block->disks[i].major,volume->super_block->disks[i].minor);
			} else {
			       // collapse super array and object array
				for (k = i; k < (MAX_MD_DEVICES -1); k++) {
					volume->super_array[k]= volume->super_array[k+1];
					volume->child_object[k] = volume->child_object[k+1];
					volume->super_block->disks[k]= volume->super_block->disks[k+1];
				}
				memset(&volume->super_block->disks[k],0,sizeof(mdp_disk_t));  //NULL out now empty disk entry
				i--; // try again
			}
		} else {
			nr_disks++;
			if (volume->super_block->disks[i].number != i ||
			    volume->super_block->disks[i].raid_disk != i) {
				change= TRUE;
				if (!fix) {	  // not allowed to fix, quit now.
					MESSAGE("Region %s object index incorrect: is %d, should be %d\n",volume->name, volume->super_block->disks[i].number,i);
				} else {
					volume->super_block->disks[i].number = i;
					volume->super_block->disks[i].raid_disk = i;
				}
			}

			get_legacy_dev(volume, volume->child_object[i]->name, &major, &minor);
			if (volume->super_block->disks[i].major != major ||
			    volume->super_block->disks[i].minor != minor) {
				change = TRUE;
				if (!fix) {	  // not allowed to fix, quit now.
					MESSAGE("Region %s object index %d(%s) has incorrect major/minor\n",volume->name, i, volume->child_object[i]->name);
				} else {
					volume->super_block->disks[i].major = major;
					volume->super_block->disks[i].minor = minor;
				}
			}

			switch (volume->super_block->disks[i].state) {
			case (1<<MD_DISK_ACTIVE | 1<<MD_DISK_SYNC):
			case (1<<MD_DISK_ACTIVE | 1<<MD_DISK_SYNC | 1<<MD_DISK_PENDING_ACTIVE ):
			case (1<<MD_DISK_ACTIVE | 1<<MD_DISK_SYNC | 1<<MD_DISK_PENDING_ACTIVE | 1<<MD_DISK_NEW):
			case (1<<MD_DISK_ACTIVE | 1<<MD_DISK_SYNC | 1<<MD_DISK_NEW):
       				active_disks++;
       				raid_disks++;
       				working_disks++;
       				break;

				// active, but not sync, kernel just kind of ignores these drives
				// so make him a spare so that the kernel will re-sync if needed.
				// or sync, but not active, do the same.
			case (1<<MD_DISK_ACTIVE):
			case (1<<MD_DISK_SYNC):
				change= TRUE;
				if (!fix) {
					MESSAGE("Region %s object index %d is in invalid state.\n",volume->name, i);
				}else {
					volume->super_block->disks[i].state = 0;
				}
			case 0:	// 0 = spare
			case (1<<MD_DISK_NEW):	// new = spare
			case (1<<MD_DISK_PENDING_ACTIVE):	// new = spare
			case (1<<MD_DISK_PENDING_ACTIVE | 1<<MD_DISK_NEW):	// new = spare
				spare_disks++;
       				working_disks++;
				break;
			case (1<<MD_DISK_REMOVED):
			case (1<<MD_DISK_FAULTY):
			case (1<<MD_DISK_FAULTY | 1<<MD_DISK_REMOVED):
			default:
				if (!fix) {
					MESSAGE("Region %s object index %d(%s) is faulty. Array may be degraded\n",volume->name, i, volume->child_object[i]->name);
				}
				failed_disks++;
//				raid_disks++;
				break;
			}
		}
	}

	// check to be sure that all of the unused disks array entries are zeroed
	// If not, the boneheaded kernel MD code will use these even though
	// the count field indicates athat they are not valid.
	// To make matters worse, only raid4/5 and 1 work this way, so since we have
	// a common SB creation routine we can not always be right.  So just allow
	// these extras disks entries to have the sync bit on or off.
	memset(&disk, 0, sizeof(mdp_disk_t));
	memset(&disk2, 0, sizeof(mdp_disk_t));
	disk.state = (1<<MD_DISK_SYNC);
	for (i = volume->nr_disks; i < MAX_MD_DEVICES; i++) {
		if (memcmp(&disk, &volume->super_block->disks[i], sizeof(mdp_disk_t)) &&
		    memcmp(&disk2, &volume->super_block->disks[i], sizeof(mdp_disk_t))) {
			change = TRUE;
			if (!fix) {
				MESSAGE("Region %s disks array not zeroed  \n",volume->name);
			} else{
				memcpy(&volume->super_block->disks[i], &disk, sizeof(mdp_disk_t));
			}
		}
	}


	if (volume->super_block->active_disks != active_disks ||
	    volume->super_block->working_disks != working_disks ||
//	    volume->super_block->raid_disks != raid_disks || DON'T FIX RAID DISKS
	    volume->super_block->failed_disks != failed_disks ||
	    volume->super_block->spare_disks != spare_disks ||
	    volume->super_block->nr_disks != nr_disks ) {
		change = TRUE;
		if (!fix) {
			MESSAGE("Region %s disk counts incorrect \n",volume->name);
		} else {
			volume->super_block->active_disks = active_disks;
			volume->super_block->working_disks = working_disks;
//			volume->super_block->raid_disks = raid_disks;
			volume->super_block->failed_disks = failed_disks;
			volume->super_block->spare_disks = spare_disks;
			volume->super_block->nr_disks = nr_disks;
		}
	}

	if (fix) {
		volume->flags &= ~MD_CORRUPT;
		volume->flags &= ~MD_DEGRADED;
		volume->region->flags |= SOFLAG_DIRTY;
		md_rediscover_volumes_for_region(volume->region);
		EngFncs->set_changes_pending();
	}
	RETURN(change);
}

