/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_discover.c
 *
 * Description: This file contains all functions related to the initial
 *              discovery of LVM physical volumes, volume groups, and logical
 *              volumes.
 */ 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"


/* Function: lvm_discover_volume_groups
 *
 *	segments : Input list of storage objects
 *	regions  : Output list of storage objects
 *
 *	This function is the entry point into the first half of discovery,
 *	which examines all segments to find PVs and assigns them to the
 *	appropriate groups.
 *
 *	Traverse the input list. Read the PV metadata from each segment and
 *	determine if the segment is a PV. If so, find the group for that PV.
 *	Allocate a physical volume. Add that PV to the group. Read in the
 *	PE map for that PV.
 */
int lvm_discover_volume_groups( dlist_t segments,
				dlist_t	regions )
{
	storage_object_t	* segment;
	pv_disk_t		* pv;
	lvm_volume_group_t	* group;
	lvm_physical_volume_t	* pv_entry;
	TAG			tag;
	int			size, rc;

	LOG_ENTRY;
	LOG_DETAILS("Searching for PVs in the object list.\n");

	GoToStartOfList(segments);
	while ( ! BlindExtractObject(segments, &size, &tag, NULL, (void**)&segment) ) {

		if ( segment->data_type != DATA_TYPE ) {
			// The segment is not a DATA segment.
			LOG_EXTRA("Skipping object %s - not DATA_TYPE\n", segment->name);
			lvm_add_object_to_list(segment, regions);
			continue;
		}

		rc = lvm_read_pv(segment, &pv);
		if (rc) {
			// The segment is not an LVM PV, or an error occurred.
			lvm_add_object_to_list(segment, regions);
			continue;
		}

		rc = lvm_find_group_for_pv(segment, pv, &group);
		if (rc) {
			// Error getting the group for this PV.
			lvm_engine->engine_free(pv);
			lvm_add_object_to_list(segment, regions);
			continue;
		}

		pv_entry = lvm_allocate_physical_volume(segment, pv);
		if ( ! pv_entry ) {
			// Memory allocation error.
			lvm_add_object_to_list(segment, regions);
			continue;
		}

		rc = lvm_add_pv_to_group_list(pv_entry, group);
		if (rc) {
			lvm_deallocate_physical_volume(pv_entry);
			lvm_add_object_to_list(segment, regions);
			continue;
		}

		rc = lvm_read_pe_map(pv_entry);
		if (rc) {
			LOG_ERROR("Error reading PE maps for object %s\n", segment->name);
			LOG_ERROR("Any regions residing on this object will be incomplete!\n");
		}
	}

	LOG_DETAILS("Container discovery complete.\n");
	RETURN(0);
}


/* Function: lvm_discover_volumes_in_group
 *
 *	Helper function for lvm_discover_logical_volumes. Traverse the LV
 *	metadata array and create logical_volumes for each valid
 *	entry in the array.
 */
static int lvm_discover_volumes_in_group( lvm_volume_group_t * group )
{
	lv_disk_t		* lv_array = group->lv_array;
	lvm_logical_volume_t	* new_volume;
	int			i;

	LOG_ENTRY;

	// Translate all of the valid lv_array entries
	// into logical volumes
	for ( i = 0; i < MAX_LV; i++ ) {

		// An LV is only valid if it has a name and
		// a valid number
		if ( ! lv_array[i].lv_name[0] ||
		     lv_array[i].lv_number >= MAX_LV ) {
			continue;
		}

		// Make sure this volume isn't already in the list.
		// Don't forget about the screwy (+1) numbering scheme.
		if ( group->volume_list[lv_array[i].lv_number+1] ) {
			continue;
		}

		// Create the new volume and place it in the appropriate
		// spot in this group's list.
		new_volume = lvm_allocate_logical_volume(&lv_array[i], group);
		if ( ! new_volume ) {
			LOG_CRITICAL("Memory error creating region %s\n", lv_array[i].lv_name);
			continue;
		}
		group->volume_list[new_volume->number] = new_volume;
		group->volume_count++;
	}

	RETURN(0);
}


/* Function: lvm_discover_logical_volumes
 *
 *	After all volume groups have been discovered, revisit each group and
 *	gather any remaining information that is necessary. After the LV array
 *	has been read, the logical volumes can be constructed.
 */
int lvm_discover_logical_volumes( boolean final_call )
{
	lvm_volume_group_t	* group;
	int 			rc;

	LOG_ENTRY;

	FOR_EACH(group, lvm_group_list) {

		LOG_DETAILS("Searching for regions in container %s\n", group->container->name);

		// Read the LV array for this group.
		rc = lvm_read_lv_array(group);
		if (rc) {
			LOG_SERIOUS("Unable to read LV metadata for container %s\n", group->container->name);
			LOG_SERIOUS("No regions can be discovered for container %s\n", group->container->name);
			continue;
		}

		// Before creating the regular volumes, set up the freespace
		// volume so it is first in the container list. Only the engine
		// keeps track of the freespace in each container.
		if ( ! group->freespace ) {
			lvm_create_freespace_volume(group);
		}

		// Assemble each region in the group.
		lvm_discover_volumes_in_group(group);

		// Construct the LE maps for each volume in the group.
		lvm_build_le_maps(group);
		lvm_check_le_maps(group, final_call);

		// Set up pointers to link snapshot volumes to their originals.
		// Don't build the snapshot maps in the engine. I/O is not
		// allowed to snapshots in the engine, so it is unnecessary.
		lvm_link_snapshot_volumes(group);
	}

	RETURN(0);
}


/* Function: lvm_check_volume_groups
 *
 *	Perform any necessary consistency checks after all groups have been
 *	assembled.
 *
 *	In the kernel this function does a lot more work. But in the kernel
 *	we also have true re-discovers, which don't happen in the engine. Thus
 *	the kernel needs to check for missing metadata that needs to be
 *	re-read.
 */
int lvm_check_volume_groups( boolean final_call )
{
	lvm_volume_group_t	* group;
	lvm_volume_group_t	* groups[MAX_VG] = {NULL};
	int			i, rc;

	LOG_ENTRY;

	FOR_EACH(group, lvm_group_list) {

		// Display a warning if the number of PVs found for the group
		// does not match the number of PVs recorded in the metadata.
		if ( final_call && group->pv_count != group->vg->pv_cur ) {
			int vg_answer = 0;
			char * vg_choice_text[3] = { "Don't Fix", "Fix", NULL };
			LOG_ERROR("Container %s has incorrect number of objects!\n", group->container->name);
			LOG_ERROR("Looking for %d objects, found %d objects.\n", group->vg->pv_cur, group->pv_count);

			// Detect which PV(s) are missing. Prompt the user for 
			// if they want the missing PV removed from the VG
			// permanently.
			for ( i = 1; i <= MAX_PV; i++ ) {
				if ( group->uuid_list[i] && ! group->pv_list[i] ) {
					char * pv_choice_text[3] = { "Don't Remove", "Remove", NULL };
					int pv_answer = 0;
					LOG_ERROR("A UUID is recorded for PV %d, but PV %d was not found.\n", i, i);
					LOG_ERROR("\tUUID: %s\n", lvm_print_uuid(group->uuid_list[i]));
					lvm_engine->user_message(lvm_plugin, &pv_answer, pv_choice_text,
						"Container %s has a UUID recorded for PV %d, but PV %d was not found. "
						"Would you like to remove PV %d from container %s *PERMANENTLY*?\n\n"
						"You should only remove this PV if you know the PV will *NEVER* be "
						"available again. If you think it is just temporarily missing, do not "
						"remove it from the container.",
						group->container->name, i, i, i, group->container->name);
					if ( pv_answer ) {
						LOG_ERROR("PV %d is being removed from container %s\n", i, group->container->name);
						lvm_clear_uuid_list_entry(group, i);
					}
				}
			}

			// Fix up the VG metadata.
			lvm_engine->user_message(lvm_plugin, &vg_answer, vg_choice_text,
				"Would you like to fix the metadata for container %s?\n",
				group->container->name);
			if ( vg_answer ) {
				LOG_ERROR("Correcting metadata for container %s\n", group->container->name);
				lvm_fix_group_after_pv_removal(group);
				LOG_ERROR("Please perform a commit so these changes are recorded\n");
			}
		}

		// Mark off this group so we can fix vg_number collisions.
		if ( ! groups[group->vg->vg_number] ) {
			groups[group->vg->vg_number] = group;
		}
		else {
			LOG_ERROR("Containers %s and %s have conflicting vg_number %d.\n",
				group->container->name, groups[group->vg->vg_number]->container->name, group->vg->vg_number);
			rc = memcmp(group->vg->vg_uuid, groups[group->vg->vg_number]->vg->vg_uuid, UUID_LEN);
			if ( rc < 0 ) {
				groups[group->vg->vg_number]->flags |= LVM_VG_FLAG_INVALID_VG_NUMBER;
				groups[group->vg->vg_number] = group;
			}
			else {
				group->flags |= LVM_VG_FLAG_INVALID_VG_NUMBER;
			}
		}
	}

	// Go back through the group list and check for any conflicting VG
	// numbers. Find the next available VG number for each one.
	FOR_EACH(group, lvm_group_list) {
		if ( group->flags & LVM_VG_FLAG_INVALID_VG_NUMBER ) {
			for ( i = 0; i < MAX_VG; i++ ) {
				if ( ! groups[i] ) {
					LOG_ERROR("Changing container %s vg_number from %d to %d.\n",
						group->container->name, group->vg->vg_number, i);
					groups[i] = group;
					group->vg->vg_number = i;
					group->flags &= ~LVM_VG_FLAG_INVALID_VG_NUMBER;
					group->container->flags |= SCFLAG_DIRTY;
					lvm_engine->set_changes_pending();
					break;
				}
			}
			if ( group->flags & LVM_VG_FLAG_INVALID_VG_NUMBER ) {
				LOG_ERROR("All valid vg_numbers are in use.\n");
				LOG_ERROR("Cannot assign a new vg_number to container %s.\n", group->container->name);
			}
		}
	}

	RETURN(0);
}
