/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: defsegmgr.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "segs.h"
#include "checks.h"
#include "discovery.h"
#include "commit.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "segoptions.h"


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static plugin_record_t segmgr_plugin_record;

plugin_record_t                *Seg_My_PluginRecord_Ptr=&segmgr_plugin_record;

struct engine_functions_s      *SegEngFncs=NULL;

BOOLEAN                        Discovery_Completed = FALSE;




/*
 * Called to revert back to original drive geometry that was
 * reported by the kernel. This is done when we have been
 * using an alternate geometry and have deleted ALL data segs
 * off the disk and can now revert to the proper drive geometry.
 */
static int revert_drive_geometry( LOGICALDISK *ld )
{
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    int                rc=EINVAL;
    DISKSEG           *mbr;
    DISKSEG           *freespace;
    DISKSEG           *tseg;
    int                metadata_count=0;
    int                freespace_count=0;
    int                data_count=0;

    LOGENTRY();

    if (disk_pdata) {

        if ( ( disk_pdata->geometry.cylinders != ld->geometry.cylinders) ||
             ( disk_pdata->geometry.heads     != ld->geometry.heads) ||
             ( disk_pdata->geometry.sectors_per_track != ld->geometry.sectors_per_track) ) {

            rc = GoToStartOfList( ld->parent_objects );
            if (rc == DLIST_SUCCESS) {

                rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&tseg );
                while (rc == DLIST_SUCCESS) {

                    if ( tseg->data_type == FREE_SPACE_TYPE ) {
                        ++freespace_count;
                    }
                    else if (tseg->data_type == META_DATA_TYPE ) {
                        ++metadata_count;
                    }
                    else if (tseg->data_type == DATA_TYPE) {
                        ++data_count;
                    }

                    rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,(void **)&tseg );

                }

            }

            // test if we can revert the geometry
            if ( data_count > 0 ) {
                rc = EPERM;    // operation not permitted
                LOGEXITRC();
                return rc;
            }

            // ask the user if it is Ok
            {
                char * choices[] = {"Yes", "No", NULL};
                int answer = 0;     // Initialize to Yes ... revert to kernel reported geometry

                POPUP_MSG( &answer,
                           choices,
                    "\nQuestion: The only remaining segment on drive %s is being deleted. Since we used an alternate "
                    "geometry for this drive, in order to get through segment discovery, you now have an opportunity "
                    "to revert back to using the drive geometry that was reported by the kernel.  The recommendation is "
                    "that you reply YES and revert back to the kernel reported geometry.\n\n"
                    "Do you want to revert back to using the kernel reported geometry?\n",
                    ld->name);

               if (answer == 1) { // if user replied NO
                   rc = EPERM;    // then return ... operation not permitted
                   LOGEXITRC();
                   return rc;
               }

            }

            // revert old drive geometry
            memcpy(&disk_pdata->geometry, &ld->geometry, sizeof(geometry_t));

            // turn off LBA addressing flags
            disk_pdata->flags &= ~DISK_USES_LBA_ADDRESSING;
            disk_pdata->flags &= ~DISK_HAS_FORCED_LBA_ADDRESSING;

            // resize the mbr and freespace segments if needed
            mbr = get_mbr_from_seglist( ld->parent_objects );
            if ( ( mbr ) &&
                 ( mbr->size != disk_pdata->geometry.sectors_per_track ) &&
                 ( metadata_count == 1) &&
                 ( freespace_count == 1)) {

                freespace        = get_freespace_following_seg( mbr );

                mbr->size        = ld->geometry.sectors_per_track;

                freespace->start = mbr->size;
                freespace->size  = ld->size - mbr->size;

            }

            rc = 0;
        }
        else {
            rc = -1;  // non destructive error
        }

    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to return the Logical Disk that a segment belongs to.
 *
 *  Can be called with any object.
 *
 *  Returns NULL on failure.
 */
LOGICALDISK * get_logical_disk( storage_object_t *obj )
{
    LOGICALDISK  *ld=NULL;

    if (obj) {

        if ( obj->plugin != Seg_My_PluginRecord_Ptr ) {
            ld = (LOGICALDISK *) obj;
        }
        else if (obj->private_data) {

            if ( ((SEG_PRIVATE_DATA *)obj->private_data)->signature == DEFAULT_SEG_MGR_PDATA_SIGNATURE ) {

                ld = ((SEG_PRIVATE_DATA *)obj->private_data)->logical_disk;

            }

        }

    }

    return ld;
}


/*
 *  Called to test if we have private data for a disk.
 */
static BOOLEAN i_can_modify_disk( LOGICALDISK *ld )
{
    DISK_PRIVATE_DATA  *disk_pdata;

    if (ld) {

        disk_pdata = get_disk_private_data( ld );

        if (disk_pdata) {

            if (disk_pdata->signature == DEFAULT_SEG_MGR_PDATA_SIGNATURE) {
                return TRUE;
            }

        }
    }

    return FALSE;
}


/*
 *  Called to test if we own the specified disk segment and can
 *  work with the segment.
 */
static BOOLEAN i_can_modify_seg( DISKSEG *seg )
{
    if (seg) {

        if (seg->plugin == Seg_My_PluginRecord_Ptr ) {

            if (seg->private_data) {

                if ( ((SEG_PRIVATE_DATA *)seg->private_data)->signature == DEFAULT_SEG_MGR_PDATA_SIGNATURE ) {

                    if ( ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk ) {

                        return TRUE;

                    }

                }

            }

        }
    }

    return FALSE;
}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/
static int SEG_SetupEVMSPlugin( engine_mode_t        mode,
                                engine_functions_t * engine_functions)
{
    int rc = EINVAL;

    if (engine_functions) {

        engine_functions->write_log_entry(ENTRY_EXIT,Seg_My_PluginRecord_Ptr,"%s: entry\n",__FUNCTION__);

        SegEngFncs  = engine_functions;

        Disk_PrivateData_List = CreateList();

        if ( Disk_PrivateData_List == NULL ) {
            rc = ENOMEM;
        }
        else {
            SegEngFncs->register_name( "/dev/evms/os2" ); // prevent anyone from building objects
            SegEngFncs->register_name( "/dev/evms/OS2" ); // with a name of OS2
            rc = 0;
        }

        LOGEXITRC();
    }

    return rc;
}


static void SEG_Cleanup(void)
{
    dlist_t seglist=CreateList();
    int rc;
    DISKSEG *seg;


    LOGENTRY();

    if (seglist != NULL) {

        // cleanup all seg private data areas
        rc = SegEngFncs->get_object_list( SEGMENT,
                                          0,
                                          Seg_My_PluginRecord_Ptr,
                                          0,
                                          &seglist );

        rc = GoToStartOfList( seglist );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( seglist,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &seg );
            while (rc == DLIST_SUCCESS) {

                if ( seg->data_type == META_DATA_TYPE ) {

                    if ( ((SEG_PRIVATE_DATA *)seg->private_data)->dlat != NULL ) {
                        free ( ((SEG_PRIVATE_DATA *)seg->private_data)->dlat );
                    }

                }

                if (seg->private_data) free(seg->private_data);

                rc = GetNextObject( seglist,sizeof(DISKSEG),SEGMENT_TAG, (void **) &seg );

            }

        }

        DestroyList(&seglist, FALSE);
    }


    // cleanup all disk private data areas
    if ( Disk_PrivateData_List != NULL ) DestroyList( &Disk_PrivateData_List, TRUE );

    LOGEXIT();
}




/*
 *  I will allow the object to be made into a volume (or reverted) if ...
 *
 *  (1) I actually own the object
 *  (2) Which means it is a segment and has necessarey ctl blocks
 *
 */
static int SEG_can_set_volume(storage_object_t * seg, BOOLEAN flag )
{
    int rc = EINVAL;
    LOGICALDISK *ld;
    DISK_PRIVATE_DATA *disk_pdata;

    LOGENTRY();

    if ( ( seg ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type   == DATA_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE ) ) {

        ld = get_logical_disk(seg);

        if (ld) {

            if (flag == TRUE) {  // CREATE VOLUME

                disk_pdata = get_disk_private_data(ld);

                if (disk_pdata) {

                   rc = 0;

                }

            }
            else {      // REVERT VOLUME
                rc = 0;
            }
        }

    }

    LOGEXITRC();
    return rc;
}



/*
 *  I can delete this segment if ...
 *
 *  (1) I own the segment
 *  (2) It is either a data segment or else an mbr segment and
 *      there are no data segments.
 *  (3) It isnt the first logical drive in an ebr chain
 */
static int SEG_CanDestroy( DISKSEG * seg )
{
    int                rc = EINVAL;
    LOGICALDISK       *ld;
    DISKSEG           *tseg;
    DISK_PRIVATE_DATA *disk_pdata;
    SEG_PRIVATE_DATA  *pdata;
    BOOLEAN            delete_disk = FALSE;

    LOGENTRY();

    if ( ( seg ) &&
         ( seg->object_type == SEGMENT ) &&
         ( i_can_modify_seg( seg ) == TRUE )) {

        if ( seg->data_type == DATA_TYPE )  {

            rc = 0;

        }
        else if ( seg->data_type == META_DATA_TYPE ) {

            ld         = get_logical_disk(seg);
            disk_pdata = get_disk_private_data(ld);
            pdata      = (SEG_PRIVATE_DATA *) seg->private_data;

            if ((ld)&&(disk_pdata)&&(pdata)) {

                if (pdata->flags & SEG_IS_MBR ) {

                    delete_disk = TRUE;

                    rc = GoToStartOfList( ld->parent_objects );
                    if (rc == DLIST_SUCCESS) {

                        rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&tseg );

                        while ( (rc == DLIST_SUCCESS) && (delete_disk == TRUE) ) {

                            // if there is just 1 data segment left on the disk then
                            // we cant remove ourselves from the disk.
                            if (tseg->data_type == DATA_TYPE) {
                                    delete_disk = FALSE;
                            }

                            rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,(void **)&tseg );
                        }

                    }

                    if (delete_disk == TRUE) {
                        rc = 0;
                    }
                    else {
                        rc = EINVAL;
                    }
                }

            }
        }
    }


    LOGEXITRC();
    return rc;
}


/*
 *  I can expand the object if ...
 *
 *  (1) it is a data segment
 *  (2) I own the segment
 *  (3) the logical disk and segment info is Ok
 *  (4) a freespace segment immediately follows it
 *
 */
static int SEG_CanExpand( DISKSEG         *seg,               // data segment
                          sector_count_t  *expand_limit,      // ?
                          dlist_t          expansion_points ) // dlist to place expand object on
{
    DISKSEG              *freespace;
    LOGICALDISK          *ld;
    void                 *handle;
    int                   rc = EINVAL;
    expand_object_info_t *expand_object;
    SEG_PRIVATE_DATA     *pdata;

    LOGENTRY();

    if ( ( expansion_points ) &&
         ( seg ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type   == DATA_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE ) &&
         ( seg_is_volitile(seg)==TRUE) ) {

        freespace = get_freespace_following_seg( seg );
        ld        = get_logical_disk(seg);
        pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

        if ( freespace && ld ) {

            // we can only expand in cylinder size amounts
            if ( freespace->size >= get_cylinder_size(ld) ) {

                expand_object = (expand_object_info_t *) SegEngFncs->engine_alloc( sizeof(expand_object_info_t) );
                if (expand_object) {

                    expand_object->object          = seg;
                    expand_object->max_expand_size = freespace->size;

                    rc = InsertObject ( (dlist_t)          expansion_points,
                                        (int)              sizeof(expand_object_info_t),
                                        (void *)           expand_object,
                                        (TAG)              EXPAND_OBJECT_TAG,
                                        (void *)           NULL,
                                        (Insertion_Modes)  AppendToList,
                                        (BOOLEAN)          TRUE,
                                        (void **)         &handle );

                    if (rc) {
                        SegEngFncs->engine_free( expand_object );
                        rc = EPERM;  // operation was not permitted on DLIST
                    }

                }
                else {
                    LOG_ERROR("\nerror, alloc of expand object failed\n");
                    rc = ENOMEM;
                }
            }
        }
    }

    LOGEXITRC();
    return rc;

}


/*
 * I can expand the object by size sectors if:
 *
 *  (1) i own the object
 *  (2) the segment can be expanded
 *  (3) the freespace has at least a cylinder of space
 *
 * If I cannot expand because the freespace is too small
 * then I'll reduce the expand sector count to my maximum.
 *
 */
static int SEG_CanExpandBy(storage_object_t * seg, sector_count_t *size)
{
    int                        rc = EINVAL;
    LOGICALDISK               *ld;
    DISKSEG                   *freespace;
    sector_count_t             SectorsPerCylinder;
    sector_count_t             max_expand_sectors=0;
    lba_t                      freespace_end_lba=0;
    SEG_PRIVATE_DATA          *pdata;


    LOGENTRY();

    if ( ( i_can_modify_seg(seg)==TRUE ) &&
         ( seg_is_volitile(seg)==TRUE  )) {

        freespace = get_freespace_following_seg( seg );
        ld        = get_logical_disk(seg);
        pdata     = (SEG_PRIVATE_DATA *)seg->private_data;

        if ( freespace && ld ) {

            SectorsPerCylinder = get_cylinder_size(ld);

            // partitions end on a cylinder boundary. if freespace doesnt end on
            // a cylinder bdy then round it down to a cyl bdy before testing if
            // freespace can handle the ExpandBy.
            if (ends_on_cylinder_boundary(ld, freespace->start+freespace->size-1) == TRUE) {
                freespace_end_lba = freespace->start + freespace->size - 1;
            }
            else {
                freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
            }

            // calculate the max useable size of the freespace area, i.e. the
            // max area that ends on a cylinder boundary.
            if (freespace_end_lba > freespace->start) {
                max_expand_sectors = freespace_end_lba - freespace->start + 1;
            }
            else {
                max_expand_sectors = 0;
            }

            // we expand in cylinder size chunks ... if max useable freespace
            // size is less than 1 cylinder then we cant do any expand at all.
            if (max_expand_sectors >= SectorsPerCylinder) {

                if ( max_expand_sectors >= *size ) {

                    if ( max_expand_sectors == *size) {
                        rc = 0;
                    }
                    else {
                        freespace_end_lba = roundup_to_cylinder_boundary(ld, freespace->start + *size - 1 );
                        *size      = freespace_end_lba - freespace->start + 1;
                    }

                }
                else {
                    *size = max_expand_sectors;
                    rc = EINVAL;
                }

            }

        }

    }


    LOGEXITRC();
    return rc;
}




/*
 * I can shrink a seg if ...
 *
 *  (1) i own the object
 *  (2) it is a data segment
 *  (3) if I chop off a cylinder, the seg will still have
 *      a minimum of 1 cylinder of space
 *
 *  If not exact set new_size to closest higher value possible.
 */
static int SEG_CanShrink( storage_object_t * seg,               // object to shrink
                          sector_count_t   * shrink_limit,      // a delta size
                          dlist_t            shrink_points )    // of type shrink_object_info_t,
                                                                // tag = SHRINK_OBJECT_TAG
{
    int                   rc = EINVAL;
    void                 *handle;
    sector_count_t        SectorsPerCylinder;
    LOGICALDISK          *ld;
    shrink_object_info_t *shrink_object=NULL;
    SEG_PRIVATE_DATA     *pdata;


    LOGENTRY();

    if ( ( seg ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type == DATA_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE ) &&
         ( seg_is_volitile(seg)==TRUE )) {


        pdata = (SEG_PRIVATE_DATA *)seg->private_data;
        ld    = get_logical_disk(seg);

        if ( ld ) {

            SectorsPerCylinder = get_cylinder_size(ld);

            // see if caller is trying to shrink smaller than min partition size
            if ( seg->size > SectorsPerCylinder) {

                if (shrink_points) {

                    shrink_object = (shrink_object_info_t *) SegEngFncs->engine_alloc( sizeof(shrink_object_info_t) );
                    if (shrink_object) {

                        // we are suppose to return the max amount we can shrink. since the
                        // minimum partition size is 1 cylinder ... then anything more than
                        // a cylinder is the max we can shrink.
                        shrink_object->object          = seg;
                        shrink_object->max_shrink_size = seg->size - SectorsPerCylinder;

                        rc = InsertObject ( (dlist_t)          shrink_points,
                                            (int)              sizeof(shrink_object_info_t),
                                            (void *)           shrink_object,
                                            (TAG)              SHRINK_OBJECT_TAG,
                                            (void *)           NULL,
                                            (Insertion_Modes)  AppendToList,
                                            (BOOLEAN)          TRUE,
                                            (void **)         &handle );

                        if (rc) {
                            LOG_ERROR("\nerror, inserting object into shrink points failed\n");
                            rc = EPERM; // operation not permitted on dlist.
                        }

                    }
                    else {
                        LOG_ERROR("\nerror, failed to alloc shrink_object\n");
                        rc = ENOMEM;
                    }
                }
            }

        }
    }


    LOGEXITRC();
    return rc;
}


/*
 *  I can allow the storage object to shrink by the specified amount if ...
 *
 *  (1) the shrink point is a segment and I own it
 *  (2) the segment is large enough to allow a shrink
 *  (3) i can shrink it and end the segment on a cylinder boundary
 *
 *  If there are any problems then I'll return an error and
 *  report the amount that we could shrink by.
 *
 */
static int SEG_CanShrinkBy( storage_object_t * seg,
                            sector_count_t   * size )
{
    int               rc = EINVAL;
    sector_count_t    SectorsPerCylinder=0;
    LOGICALDISK      *ld=NULL;
    sector_count_t    delta=0;
    SEG_PRIVATE_DATA *pdata;

    LOGENTRY();

    if ( ( seg ) &&
         ( size ) &&
         (*size > 0 ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type == DATA_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE ) &&
         ( seg_is_volitile(seg)==TRUE )) {


        pdata = (SEG_PRIVATE_DATA *)seg->private_data;
        ld    = get_logical_disk(seg);

        if ( ld ) {

            SectorsPerCylinder = get_cylinder_size(ld);

            // seg cant shrink if it is a cylinder or less in size
            if (seg->size > SectorsPerCylinder) {

                if ( *size < seg->size ) {

                    if (*size < SectorsPerCylinder) {
                        delta = SectorsPerCylinder;
                    }
                    else {
                        delta = (*size / SectorsPerCylinder)*SectorsPerCylinder;
                    }

                }
                else {
                    delta = seg->size - SectorsPerCylinder;
                }

                if ( delta == *size  ) {
                    rc = 0;
                }
                else {
                    *size = delta;
                    rc = EINVAL;
                }

            }
            else {
                *size =0;
                rc = EINVAL;
            }

        }
        else {
            *size = 0;
            rc = EINVAL;
        }

    }
    else {
        *size = 0;
        rc = EINVAL;
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to run discovery code on dlist of evms objects that
 *  the engine has found so far.  We essentially need to walk the
 *  list of objects, looking for Logical Disks, and see if we
 *  recognize partitioning schemes.  If so, consume the logical
 *  disk by removing it from the dlist, and place all new segment
 *  objects on the output_object dlist. Any object we dont like in
 *  the input_object dlist must be copied to the output_object dlist.
 *
 */
static int SEG_Discover( dlist_t input_objects, dlist_t output_objects, BOOLEAN final_call)
{
    int  rc = 0;
    int  count = 0;
    discovery_parm_block_t  dparms = {output_objects, &count };

    LOGENTRY();

    if ( Discovery_Completed == FALSE ) {

        Discovery_Completed = TRUE;

        rc = ForEachItem( input_objects, SegmentDiscovery, (void *) &dparms, TRUE );

    }
    else {
        rc = CopyList( output_objects, input_objects, AppendToList );
    }

    // if we got a least a single object created ... return the count ...
    // else return the RC from the discovery code.
    if (count > 0) {
        rc = count;
    }

    LOGEXITRC();
    return rc;
}




/*
 *  Called with CREATE api for a DATA type object.  This means we assign
 *  ourselves to the disk by laying down our partition scheme, creating
 *  an MBR metadata segment and a FREESPACE segment.
 */
static int Assign_SegmentManager_ToDisk( LOGICALDISK *ld,
                                         BOOLEAN      isa_os2_disk,
                                         char        *disk_name,
                                         dlist_t      new_objects )
{
    int               rc;
    DISKSEG          *freespace;

    LOGENTRY();

    // we should not have a private data area for this disk yet because we were just
    // told to install on it.  we must have been passed a logical disk we are already
    // using and this is very wrong.
    if ( get_disk_private_data(ld) ) {
        LOG_ERROR("attempting to reinstall on disk (%s)\n", ld->name);
        rc = EINVAL;
    }
    else {

        // create disk private data area
        rc = create_disk_private_data( ld );
        if (rc == 0) {

            // toss any items that may have been left in the parent dlist
            DeleteAllItems(ld->parent_objects, FALSE);

            // create freespace on the disk
            rc = find_freespace_on_disk( ld );
            if ( rc == 0) {

                // now get the freespace segment we just created
                freespace = get_first_freespace_seg_in_list( ld->parent_objects );
                if ( freespace ) {

                    // create an MBR storage object for the new disk
                    // it will be marked dirty for later commit.
                    rc = create_mbr_For_Disk( ld, disk_name, isa_os2_disk );
                    if (rc == 0) {

                        rc = CopyList(new_objects, ld->parent_objects,AppendToList);

                    }
                    else {
                        free_disk_segment(freespace);
                        delete_disk_private_data( ld );
                    }

                }
                else {
                    rc = ENODEV;
                    delete_disk_private_data( ld );
                    LOG_ERROR("failed to create any freespace storage objects on disk %s\n", ld->name);
                }

            }
            else {
                delete_disk_private_data( ld );
                LOG_ERROR("unable to establish free space on disk %s\n", ld->name);
            }
        }
        else {
            LOG_ERROR("unable to malloc disk (%s) private data area\n", ld->name);
        }

    }

    LOGEXITRC();
    return rc;
}




/*  Called to create the specified segment.
 *
 *  - allocate memory for segment,
 *  - add segment to seg list of logical disk.
 *  - fill in seg info
 */
static int CreateDiskSegment( DISKSEG               * free_space,           /* a free space disk segment      */
                              DISKSEG             * * new_segment,          /* resulting data segment         */
                              sector_count_t          size,                 /* size in sectors                */
                              sector_count_t          sector_offset,        /* sector offset in freespace seg */
                              u_int32_t               partition_type,       /* type, e.g. 0x83=linux ext2     */
                              BOOLEAN                 Bootable,             /* partition record active flag   */
                              BOOLEAN                 primary_partition,    /* TRUE if it must be primary     */
                              char                  * partition_name,       /* for OS2 partitions             */
                              char                  * volume_name,          /* for OS2 partitions             */
                              char                  * drive_letter )        /* for OS2 partitions             */
{
    int                 rc=DLIST_SUCCESS;
    DISKSEG            *seg=NULL;
    SEG_PRIVATE_DATA   *seg_pdata;
    SEG_PRIVATE_DATA   *free_pdata;
    LOGICALDISK        *ld;
    DISK_PRIVATE_DATA  *disk_pdata;
    DISKSEG            *mbr;
    SEG_PRIVATE_DATA   *mbr_pdata=NULL;
    sector_count_t      seg_size;
    lba_t               seg_start_lba;
    lba_t               seg_end_lba;
    DLA_Entry           dla;

    sector_count_t      sizeof_freespace = free_space->size;
    lba_t               startof_freespace = free_space->start;

    BOOLEAN             create_seg_on_cylinder_boundary = FALSE;
    sector_count_t      cylsize;
    sector_count_t      trksize;


    LOGENTRY();


    /*
     *  Get private data areas
     */
    free_pdata = (SEG_PRIVATE_DATA *) free_space->private_data;
    ld         = get_logical_disk( free_space );
    disk_pdata = get_disk_private_data(ld);
    cylsize    = disk_pdata->geometry.heads * disk_pdata->geometry.sectors_per_track;
    trksize    = disk_pdata->geometry.sectors_per_track;

    /*
     *  Check for an MBR metadata segment
     *
     *  The disk must have an MBR on it else we dont have any
     *  partition table to start with.
     *
     */
    mbr = get_mbr_from_seglist( ld->parent_objects );
    if (mbr==NULL) {
        LOG_ERROR("disk has no MBR \n");
        LOGEXIT();
        return EINVAL;
    }
    else {
        mbr_pdata = (SEG_PRIVATE_DATA *)mbr->private_data;
        if (mbr_pdata==NULL) {
            LOG_ERROR("mbr segment is missing private data area\n");
            LOGEXIT();
            return ENOMEM;
        }
    }

    /*
     *  Dump debug info to engine log
     */
    LOG_DEBUG("New Seg Parms ...\n");
    LOG_DEBUG("         Size: %lld\n", size);
    LOG_DEBUG("       Offset: %lld\n", sector_offset );
    LOG_DEBUG("FreeSpace ...\n");
    LOG_DEBUG("    Start LBA: %lld\n", free_space->start);
    LOG_DEBUG("         Size: %lld\n", free_space->size);
    LOG_DEBUG("     Cyl Size: %lld\n", get_cylinder_size(ld) );
    if (starts_on_cylinder_boundary( ld, free_space->start )==TRUE) {
        LOG_DEBUG("  Cyl Bdy: Yes ... starts on cylinder boundary\n" );
    }
    else {
        LOG_DEBUG("  Cyl Bdy: No ... does not start on cylinder boundary\n");
    }

    /*
     * Determine the kind of allignment we should impose on the starting
     * LBA of this partition. All partitions will end on a cylinder
     * boundary.
     *
     * Use track allignment for starting LBA if ...
     *
     * (1) we are creating the 1st primary partition following the mbr
     *     and it will start within the 1st cylinder on the drive.
     *
     * (2) we are creating a logical drive that will start at the very
     *     beginning of an established extd partition
     *
     * Everything else will start on cylinder allignment.
     *
     */
    if ( ( primary_partition == TRUE ) &&
         ( (free_space->start+sector_offset) < cylsize )) {

        create_seg_on_cylinder_boundary = FALSE;

    }
    else if ( ( primary_partition == FALSE ) &&
              ( seg_is_within_or_adjacant_to_extended_partition( ld, free_space ) == TRUE ) ) {

        if ( (free_space->start+sector_offset >= disk_pdata->extd_partition_lba) &&
             (free_space->start+sector_offset <  disk_pdata->extd_partition_lba+trksize)) {
            create_seg_on_cylinder_boundary = FALSE;
        }
        else {
            create_seg_on_cylinder_boundary = TRUE;
        }

    }
    else {
        create_seg_on_cylinder_boundary = TRUE;
    }

    /*
     *  Start segment on a cylinder boundary if required
     */
    if ( ( create_seg_on_cylinder_boundary == TRUE ) &&
         ( starts_on_cylinder_boundary(ld, free_space->start+sector_offset) == FALSE)) {

        if ( sector_offset < cylsize ) {
            seg_start_lba   = roundup_to_cylinder_boundary(ld, free_space->start + sector_offset)+1;
        }
        else {
            seg_start_lba   = rounddown_to_cylinder_boundary(ld, free_space->start + sector_offset);
        }

    }
    else {
        seg_start_lba = free_space->start + sector_offset;
    }


    /*
     *  End segment on a cylinder boundary
     */
    seg_end_lba = seg_start_lba + size - 1;

    if ( ends_on_cylinder_boundary(ld, seg_end_lba) == FALSE ) {

        // if less than a cylinder ... round up to next cyl boundary
        if ( ((seg_end_lba - seg_start_lba + 1) / cylsize) == 0) {
            seg_end_lba = roundup_to_cylinder_boundary(ld, seg_end_lba);
        }
        else {
            seg_end_lba = rounddown_to_cylinder_boundary(ld, seg_end_lba) - 1;
        }

        // if we goofed somehow and rounded down to start of seg then fix it!
        if ( seg_start_lba >= seg_end_lba ) {
            seg_end_lba     = roundup_to_cylinder_boundary(ld, seg_start_lba+1);
        }

    }

    /*
     *  Now get its actual size
     */
    seg_size = seg_end_lba - seg_start_lba + 1;


    /*
     * Test starting LBA
     */
    if (  seg_start_lba < free_space->start ) {

        LOG_ERROR("error, with cylinder allignment, the new seg would start before the free_space segment\n");
        *new_segment = NULL;
        rc = EINVAL;
        LOGEXITRC();
        return rc;

    }

    /*
     *  Test ending LBA
     */
    if (  seg_end_lba > (free_space->start + free_space->size -1) ) {

        LOG_ERROR("error, with cylinder allignment the new segment would run past end of free_space segment\n");
        *new_segment = NULL;
        rc = EINVAL;
        LOGEXITRC();
        return rc;

    }


    /*
     *  Dump debug info to engine log
     */
    LOG_DEBUG("Create Parms:\n");
    LOG_DEBUG("FreeSpace: start_lba= %lld  sector_offset= %lld  size= %lld  name= %s\n", free_space->start, sector_offset, free_space->size,free_space->name );
    LOG_DEBUG("   NewSeg: start_lba= %lld  end_lba= %lld  size= %lld\n", seg_start_lba, seg_end_lba, seg_size);


    /*
     *  Allocate memory for the new disk data segment (DISKSEG)
     */
    seg = allocate_disk_segment(ld);
    if (seg==NULL) {
        *new_segment = NULL;
        rc = ENOMEM;
        LOGEXITRC();
        return rc;
    }
    else {
        seg_pdata = (SEG_PRIVATE_DATA *) seg->private_data;
    }

    /*
     *  Now were ready to fill out the new DISKSEG
     */
    seg->size                 = seg_size;
    seg->start                = seg_start_lba;
    seg->flags               |= SOFLAG_DIRTY;

    if (Bootable) {
        seg->flags           |= SOFLAG_BIOS_READABLE  ;
        seg_pdata->boot_ind   = ACTIVE_PARTITION;
    }

    seg_pdata->sys_id         = partition_type;


    /*
     *  Figure out if it will be added as a primary partition or
     *  as a logical drive.
     */
    if ( ( primary_partition == TRUE ) || ( Bootable == TRUE) ) {

        if ( seg_is_within_the_extended_partition( ld, seg ) == FALSE ) {

            seg_pdata->flags |= SEG_IS_PRIMARY_PARTITION;
            rc = 0;
        }
        else {
            rc = EINVAL;
            free_disk_segment(seg);
            *new_segment = NULL;
            LOG_ERROR("primary partition specified but it falls within an extended partition on the drive\n");
            LOGEXITRC();
            return rc;
        }

    }
    else {

        if ( ( disk_has_extended_partition( ld ) == FALSE ) ||
             ( seg_is_within_or_adjacant_to_extended_partition( ld, seg ) == TRUE )) {

            seg_pdata->flags |= SEG_IS_LOGICAL_PARTITION;
            rc = 0;

        }
        else {
            rc = EINVAL;
            free_disk_segment(seg);
            *new_segment = NULL;
            LOG_ERROR("primary partition not specified but cannot add segment as a logical partition on the drive\n");
            LOGEXITRC();
            return rc;
        }

    }


    /*
     *  If OS2 disk ... Build a DLAT entry for the partition
     */
    if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

        memset(&dla, 0, sizeof(DLA_Entry));

        dla.Partition_Size           = seg->size;
        dla.Partition_Start          = seg->start;
        dla.Drive_Letter             = *drive_letter;
        dla.Partition_Serial_Number  = seg_gen_serial_number((u_int32_t)&dla.Partition_Serial_Number);
        if ( strlen(partition_name) > 0) {
            strncpy(&dla.Partition_Name[0], partition_name, PARTITION_NAME_SIZE);
        }
        if ( strlen(volume_name) > 0) {
            strncpy(&dla.Volume_Name[0], volume_name, VOLUME_NAME_SIZE);
        }

    }


    /*
     *  Calculate new dimensions for the existing freespace segment
     */
    if ( seg->start == free_space->start ) {
        free_space->size  -= seg->size;
        free_space->start += seg->size;
    }
    else {
        free_space->size   = seg->start - free_space->start;
    }


    LOG_DEBUG("New FreeSpace: start_lba= %lld size= %lld\n", free_space->start, free_space->size);

    /*
     *  If the new segment is a logical partition then we need to create a
     *  Logical Drive equivalent in our DLIST, i.e. an EBR track, with
     *  an EBR table and dlat, and a logical partition.
     */
    if ( seg_pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

        rc = create_logical_partition( ld, seg, &dla, free_space, sector_offset );

    }
    else {

        rc = create_primary_partition( ld, seg, &dla );
    }


    if (rc) {
        free_space->size  = sizeof_freespace;
        free_space->start = startof_freespace;

        free_disk_segment(seg);
        *new_segment = NULL;
        LOGEXITRC();
        return rc;
    }


    /*
     *  Remove the free_space seg from the list, if we used up the entire
     *  segment for the new data segment.
     */
    if ( free_space->size == 0 ) {
        LOG_DEBUG("removing freespace segment from disk list because we used it all up\n");
        remove_diskseg_from_list(ld->parent_objects, free_space );
        free_disk_segment(free_space);
    }

    find_freespace_on_disk( ld );  // re-examine the disk for any free space and expose it by
                                   // hanging free space segments.

    mbr->flags |= SOFLAG_DIRTY;    // mark mbr dirty so we get called for a commit

    *new_segment = seg;            // return the new data segment to the caller
    rc = 0;

    LOGEXITRC();
    return rc;
}



static int GetCreateOptions(  DISKSEG         * seg,
                              LOGICALDISK     * ld,
                              option_array_t  * options,
                              sector_count_t  * size,
                              lsn_t           * sector_offset,
                              u_int32_t       * type,
                              BOOLEAN         * bootable,
                              BOOLEAN         * primary_partition,
                              char            * partition_name,
                              char            * volume_name,
                              char            * drive_letter )
                          //    char           * disk_name  )
{
    int i;
    int rc = EINVAL;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    LOGENTRY();

    // check parms
    if ((ld==NULL)||(seg==NULL)||(options==NULL)||(disk_pdata==NULL)) {
        LOG_DEBUG("error, NULL ptr in parms\n");
        LOGEXITRC();
        return rc;
    }

    // init the options before beginning
    *(drive_letter)=0x00;
    *(volume_name) =0x00;
    *(partition_name)=0x00;
    *primary_partition = FALSE;
    *bootable = FALSE;
    *type = 0;
    *sector_offset = 0;
    *size = 0;

    // pickup option values
    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            switch (options->option[i].number) {

                case SEG_CREATE_OPTION_SIZE_INDEX:
                    *size = options->option[i].value.ui64;
                    break;

                case SEG_CREATE_OPTION_OFFSET_INDEX:
                    *sector_offset = options->option[i].value.ui64;
                    break;

                case SEG_CREATE_OPTION_TYPE_INDEX:
                    *type = (u_int32_t) options->option[i].value.uc;
                    break;

                case SEG_CREATE_OPTION_BOOTABLE_INDEX:
                    *bootable = options->option[i].value.uc;
                    break;

                case SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX:
                    *primary_partition = options->option[i].value.bool;
                    break;

                case SEG_CREATE_OPTION_PARTITIONNAME_INDEX:
                    strcpy( partition_name, options->option[i].value.s );
                    break;

                case SEG_CREATE_OPTION_VOLUMENAME_INDEX:
                    strcpy( volume_name, options->option[i].value.s );
                    break;

                case SEG_CREATE_OPTION_DRIVELETTER_INDEX:
                    strncpy(drive_letter,options->option[i].value.s,1);
                    break;

                default:
                    break;

            }

        } else {

            if (strcmp(options->option[i].name, SEG_CREATE_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }
            else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_OFFSET_NAME) == 0) {
                *sector_offset = options->option[i].value.ui64;
            }
            else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_TYPE_NAME) == 0) {
                *type = options->option[i].value.ui32;
            }
            else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_BOOTABLE_NAME) == 0) {
                *bootable = options->option[i].value.uc;
            }
            else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_PRIMARY_PARTITION_NAME ) == 0) {
                *primary_partition = options->option[i].value.uc;
            }
            else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_PARTITIONNAME_NAME ) == 0) {
                strcpy( partition_name, options->option[i].value.s );
            }
            else if (strcmp(options->option[i].name, SEG_CREATE_OPTION_VOLUMENAME_NAME ) == 0) {
                strcpy( volume_name, options->option[i].value.s );
            }
            else if (strcmp(options->option[i].name,SEG_CREATE_OPTION_DRIVELETTER_NAME) == 0) {
                strncpy(drive_letter,options->option[i].value.s,1);
            }

        }
    }


    LOG_DEBUG("Create Options ...\n");
    LOG_DEBUG("          seg name: %s\n", partition_name);
    LOG_DEBUG("              size: %lld\n", *size);
    LOG_DEBUG("     sector offset: %lld\n", *sector_offset);
    LOG_DEBUG("              type: %d\n", *type);
    if (*bootable==TRUE)
        LOG_DEBUG("     make bootable: YES\n");
    else
        LOG_DEBUG("     make bootable: NO\n");
    if (*primary_partition==TRUE)
        LOG_DEBUG("      make primary: YES\n");
    else
        LOG_DEBUG("      make primary: NO\n");
    LOG_DEBUG("      drive letter: %s\n", drive_letter );
    LOG_DEBUG("       volume name: %s\n", volume_name );


    // test results
    if ( (*size > 0) &&
         (*sector_offset >= 0) &&
         ( (*size + *sector_offset) <= seg->size ) &&
         (*type != 0 ) &&
         ((*bootable == TRUE)||(*bootable==FALSE)) &&
         ((*primary_partition==TRUE)||(*primary_partition==FALSE))) {

        // OS2 partitions must have a partition name
        if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

            if ( strlen(partition_name) > 0 ) {
                rc = 0;
            }
        }
        else {
            rc = 0;
        }

    }


    LOGEXITRC();
    return rc;
}

static int GetAssignOptions(  option_array_t * options,  char  * disk_name , BOOLEAN *isa_os2_disk )
{
    int i;
    int rc = EINVAL;


    LOGENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            switch (options->option[i].number) {

                case SEG_ASSIGN_OPTION_TYPENAME_INDEX:
                    if ( strlen(options->option[i].value.s) ) {

                        if ( strncmp(options->option[i].value.s, "OS/2",4) == 0) {
                            *isa_os2_disk = TRUE;
                        }
                        else {
                            *isa_os2_disk = FALSE;
                        }
                        rc =0;
                    }
                    break;


                case SEG_ASSIGN_OPTION_DISKNAME_INDEX:
                    if ( ( strlen(options->option[i].value.s) > 0 ) &&
                         ( strlen(options->option[i].value.s) <= DISK_NAME_SIZE )) {

                        strcpy( disk_name, options->option[i].value.s );
                        rc = 0;
                    }
                    else {
                        rc = EINVAL;
                    }
                    break;

                default:
                    break;

            }

        }
        else {

            if (strcmp(options->option[i].name,SEG_ASSIGN_OPTION_TYPENAME_NAME) == 0) {

                if ( strlen(options->option[i].value.s) ) {

                    if ( strncmp(options->option[i].name, "OS/2",4) == 0) {
                        *isa_os2_disk = TRUE;
                    }
                    else {
                        *isa_os2_disk = FALSE;
                    }
                    rc =0;
                }

            }

            if (strcmp(options->option[i].name,SEG_ASSIGN_OPTION_DISKNAME_NAME) == 0) {

                if ( ( strlen(options->option[i].value.s) > 0 ) &&
                      ( strlen(options->option[i].value.s) <= DISK_NAME_SIZE )) {
                     strncpy( disk_name, options->option[i].value.s, DISK_NAME_SIZE );
                     rc = 0;
                 }
                 else {
                     rc = EINVAL;
                 }

                 break;
            }

        }

    }

/*
    LOG_DEBUG("Assign Options ... rc= %d\n", rc);
    LOG_DEBUG("          disk type: %d\n", *isa_os2_disk);
    LOG_DEBUG("          disk name: %s\n", disk_name);
*/

    LOGEXITRC();
    return rc;
}


/*
 * Create storage_object_t(s) from the list of objects using the given
 * options.  Return the newly allocated storage_object_t(s) in new_objects
 * list.
 */
static int SEG_CreateSegment( dlist_t          input_objects,
                              option_array_t * options,
                              dlist_t          new_objects)
{
    int                 rc;
    uint                object_count=0;
    void               *handle;

    DISKSEG            *free_space_seg;
    DISKSEG            *newseg;

    //  Options ...
    sector_count_t      size;
    sector_count_t      sector_offset;
    u_int32_t           type;
    BOOLEAN             bootable;
    BOOLEAN             primary_partition;
    BOOLEAN             created_disk = FALSE;
    char                partition_name[EVMS_NAME_SIZE+1];
    char                volume_name[EVMS_VOLUME_NAME_SIZE+1];
    char                drive_letter[16];
    char                disk_name[EVMS_NAME_SIZE+1];
    BOOLEAN             isa_os2_disk=FALSE;
    LOGICALDISK        *ld=NULL;

    uint                objsize;
    TAG                 objtag;




    LOGENTRY();


    // we should only be called with a single input object
    rc  = GetListSize( input_objects, &object_count );
    if ( rc==DLIST_SUCCESS ) {

        if ( object_count ==1 ) {

            // get the free space seg from the list
            rc = GoToStartOfList( input_objects );
            if ( rc == DLIST_SUCCESS ) {

                rc = BlindGetObject( input_objects, &objsize, &objtag,NULL,FALSE,(void **)&free_space_seg );
                if ( rc == DLIST_SUCCESS ) {

                    // see if we are installing on a DISK and therefore are assigning the segment
                    // manager to the DISK ...
                    // added checks for DATA SEGS for s390 work. we consume data segs on s390 so as
                    // to provide an msdos partitioning capability ... yuck.
                    if ( ( free_space_seg->object_type == DISK )||
                         ( free_space_seg->object_type == SEGMENT && free_space_seg->data_type == DATA_TYPE ))  {

                        // get disk assign options
                        rc = GetAssignOptions( options, disk_name, &isa_os2_disk );
                        if (rc) {
                            LOG_ERROR("invalid options\n");
                            rc = EINVAL;
                        }
                        else {
                            rc = Assign_SegmentManager_ToDisk( (LOGICALDISK *) free_space_seg,
                                                               isa_os2_disk,
                                                               disk_name,
                                                               new_objects );
                        }

                    }
                    else if ( ( i_can_modify_seg(free_space_seg) == TRUE ) &&
                              ( free_space_seg->data_type == FREE_SPACE_TYPE )) {


                        // get seg create options
                        rc = GetCreateOptions( free_space_seg,
                                               get_logical_disk(free_space_seg),
                                               options,
                                               &size,
                                               &sector_offset,
                                               &type,
                                               &bootable,
                                               &primary_partition,
                                               partition_name,
                                               volume_name,
                                               drive_letter );

                        if (rc) {
                            LOG_ERROR("invalid options\n");
                            LOGEXIT();
                            return EINVAL;
                        }



                        rc = CreateDiskSegment( free_space_seg,
                                                &newseg,
                                                size,
                                                sector_offset,
                                                type,
                                                bootable,
                                                primary_partition,
                                                partition_name,
                                                volume_name,
                                                drive_letter );

                        if (rc ==0) {
                            rc = InsertObject ( (dlist_t)          new_objects,
                                                (int)              sizeof(storage_object_t),
                                                (void *)           newseg,
                                                (TAG)              SEGMENT_TAG,
                                                (void *)           NULL,
                                                (Insertion_Modes)  AppendToList,
                                                (BOOLEAN)          TRUE,
                                                (void **)         &handle );
                            if (rc) {
                                LOG_ERROR("dlist error inserting new object into the engine new_objects list.\n");
                            }
                        }


                    }
                    else {
                        LOG_ERROR("object, to be consumed by create, has the wrong data_type\n");
                        rc = EINVAL;
                    }

                }
                else {
                    LOG_ERROR("error returned from dlist GetObject call\n");
                    rc = EINVAL;
                }
            }
            else {
                LOG_ERROR("error returned from dlist GotoStartOfList call\n");
                rc = EINVAL;
            }
        }
        else {
            LOG_ERROR("expected 1 object in the input dlist but found %d\n", object_count);
            rc = EINVAL;
        }
    }
    else {
        LOG_ERROR("error returned from dlist GetListSize call\n");
        rc = EINVAL;
    }


    // free disk private data if we just created the disk but failed with the segment create
    if ( (rc) && (created_disk==TRUE) ) {
        delete_disk_private_data( ld );
    }


    LOGEXITRC();
    return rc;
}


/*
 *  Called to free an embedded data segment.
 */
static int SEG_DestroyEmbeddedSegment( LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata, DISKSEG * seg )
{
    int rc=EINVAL;
    DISKSEG *container_seg=NULL;
    DISKSEG *freespace=NULL;
    uint     segcount;
    struct plugin_functions_s *dft;


    LOGENTRY();

    // get container seg that was consumed by the embedded
    // partition segments.
    container_seg = only_child(seg);

    if (container_seg) {
        LOG_DEBUG("container seg is %s\n", container_seg->name );
    }
    else {
        LOG_DEBUG("container seg not found\n");
    }

    if ( container_seg ) {

        // ask the user if it is Ok
        {
            char * choices[] = {"Yes", "No", NULL};
            int answer = 0;     // Initialize to Yes

            POPUP_MSG( &answer,
                        choices,
                        "\n\nYou are about to destroy a segment that was discovered in an embedded partition.  "
                        "This segment can be destroyed and removed from the embedded partition table, however, new \n"
                        "segments cannot be created within the embedded partition table.\n\n"
                        "The reason you may wish to destroy embedded segments is so that you can recover the primary \n"
                        "partition within which the embedded segments are found.  This is accomplished by destroying all of \n"
                        "the embedded segments that consumed the primary partition.  Once they are destroyed, the primary \n"
                        "partition will become a top most object and available for use by EVMS.\n\n"
                        "Do you want to continue and destroy the embedded segment?\n",
                        ld->name);

            if (answer == 1) {
                rc = EINVAL;
                LOGEXITRC();
                return rc;
            }

        }

        // remove embedded partition segment from disk seg list and unregister its name
        rc = remove_diskseg_from_list( ld->parent_objects, seg );

        if (rc == 0) {

            // remove embedded partition from the container segment
            DeleteObject( container_seg->parent_objects, seg );

            // reduce count of embedded partitions on this drive.
            --disk_pdata->embedded_partition_count;

            // if all embedded partitions are deleted from the
            // container seg then we can revert the container seg
            if ( GetListSize(container_seg->parent_objects, &segcount) ) {
                segcount=0;
            }

            if (segcount==0) {

                // we are about to insert the primary segment back into
                // the disk seglist. however, a freespace sement may be
                // occupying the area in which we need to insert the
                // data segment that represents the primary partition.
                // the solution is to remove freespace segments from
                // the disk prior to doing the insert.
                while ( (freespace=get_first_freespace_seg_in_list(ld->parent_objects))!=NULL) {

                    remove_diskseg_from_list( ld->parent_objects, freespace );

                    free_disk_segment(freespace);

                }

                // remove container seg from the container seglist for the disk
                revert_container_segment( container_seg );

                // and blast the front of the partition to remove the disk label
                dft = (struct plugin_functions_s *) ld->plugin->functions.plugin;
                dft->add_sectors_to_kill_list( ld, container_seg->start, 2);

            }

            // actually free embedded partition storage object
            free_disk_segment(seg);

            // rename the logical partitions
            fixup_logical_partition_names(ld);

            // reclaim any freespace on the disk
            find_freespace_on_disk( ld );

        }

    }

    LOGEXITRC();
    return rc;
}

/*
 *  Called to free a data segment.  If the corresponding disk partition
 *  is a logical drive then we also need to free the EBR segment
 *  as well.
 */
static int SEG_DestroySegment( DISKSEG * seg, dlist_t child_objects )
{
    int                        rc = EINVAL;
    SEG_PRIVATE_DATA          *pdata;
    DISKSEG                   *tseg;
    LOGICALDISK               *ld=NULL;
    DISK_PRIVATE_DATA         *disk_pdata=NULL;
    BOOLEAN                    remove_segmgr_from_disk=FALSE;
    BOOLEAN                    seg_is_mbr=FALSE;
    BOOLEAN                    say_remove_msg=TRUE;
    storage_object_t          *mbr = NULL;
    struct plugin_functions_s *dft;


    LOGENTRY();

    if ( ( seg ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type   != FREE_SPACE_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE)) { // &&
         //( seg_is_volitile(seg)==TRUE ) ) {

        ld         = get_logical_disk(seg);
        disk_pdata = get_disk_private_data(ld);
        pdata      = (SEG_PRIVATE_DATA *) seg->private_data;
        mbr        = get_mbr_from_seglist(ld->parent_objects);

        LOG_DEBUG("seg: %s\n", seg->name );

        // test private data ptrs
        if ((pdata==NULL)||(disk_pdata==NULL)) {
            LOGEXITRC();
            return rc;
        }

        // test if caller is trying to free an invalid seg type
        if ( ( seg->data_type == FREE_SPACE_TYPE )||
             ( pdata->flags & SEG_IS_EBR ) ) {
            LOGEXITRC();
            return rc;
        }

        // handle special case of embedded segment
        if ( pdata->flags & SEG_IS_EMBEDDED ) {

            rc = SEG_DestroyEmbeddedSegment( ld, disk_pdata, seg );

            if ( rc == 0 ) {
                mbr->flags |= SOFLAG_DIRTY;
            }

            LOGEXITRC();
            return rc;
        }


        // test if we are freeing the MBR segment itself and
        // need to remove the segment manager from the disk.
        if ( pdata->flags & SEG_IS_MBR ) {

            seg_is_mbr=TRUE;

            rc = GoToStartOfList( ld->parent_objects );
            if (rc == DLIST_SUCCESS) {

                rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&tseg );
                if (rc == DLIST_SUCCESS) {

                    remove_segmgr_from_disk = TRUE;

                    do {

                        // if there is just 1 data segment left on the disk then
                        // we cant remove ourselves from the disk.
                        if (tseg->data_type == DATA_TYPE) {
                            remove_segmgr_from_disk = FALSE;
                        }

                        rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,(void **)&tseg );

                    } while ((rc == DLIST_SUCCESS)&&(remove_segmgr_from_disk == TRUE));

                }
            }

            if ( remove_segmgr_from_disk == TRUE ) {
                rc = 0;
            }
            else {
                rc = EINVAL;
                LOG_ERROR("error, trying to delete mbr segment when data segments are still found on the disk\n");
                LOGEXITRC();
                return rc;
            }

        }


        // remove segment from list and unregister its name
        rc = remove_diskseg_from_list( ld->parent_objects, seg );
        if (rc == 0) {

            if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

                // remove EBR segment from list and unregister its name
                rc = remove_diskseg_from_list( ld->parent_objects, pdata->ebr );

                if (rc == 0) {

                    // make sure to fixup the ebr chain
                    fixup_EBR_Chain( ld );

                    // zap the ebr sector
                    dft = (struct plugin_functions_s *) ld->plugin->functions.plugin;
                    dft->add_sectors_to_kill_list( ld, pdata->ebr->start, 1);

                    // free memory
                    free_disk_segment(pdata->ebr);

                    // rename the logical partitions
                    fixup_logical_partition_names(ld);
                }
                else {

                    // error removing ebr ... put seg back on list
                    insert_diskseg_into_list( ld->parent_objects, seg );
                    fixup_EBR_Chain(ld);
                }

            }

            if (rc == 0) {

                // release the dlat entry if this is an OS2 disk
                if ( pdata->dla_entry != NULL) memset( pdata->dla_entry, 0, sizeof(DLA_Entry));

                // free actual storage object memory
                free_disk_segment(seg);

                // merge free space on disk
                find_freespace_on_disk( ld );

            }

        }

    }



    // if we have removed all the data segments from this disk
    // then we'll revert back to the original drive geometry in
    // case we were using an alternate geometry due to bogus
    // partitioning by some tool.
    //
    // this work isnt necessary if we are about to remove ourselves
    // from managing the disk because well be tossing our private
    // data in a moment.
    if (rc == 0 && remove_segmgr_from_disk == FALSE) {

        rc = revert_drive_geometry(ld);
        if (rc == 0) {

            // tell the user
            POPUP_MSG( NULL,
                       NULL,
                       "\nInfo, reverting to kernel reported geometry for drive %s.\n"
                       "The new geometry is C= %lld H= %d S= %d\n",
                       ld->name,
                       disk_pdata->geometry.cylinders,
                       disk_pdata->geometry.heads,
                       disk_pdata->geometry.sectors_per_track );
        }

        // test if we should do a popup msg to tell the user
        // about removing the segment manager.
        rc = GoToStartOfList( ld->parent_objects );
        if (rc == DLIST_SUCCESS) {

            rc = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **)&tseg );
            while (rc == DLIST_SUCCESS) {

                // if there is just 1 data segment left on the disk then
                // we dont bother telling the user about how to remove the
                // segment manager from the disk.
                if (tseg->data_type == DATA_TYPE) {
                    say_remove_msg = FALSE;
                }

                rc = GetNextObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,(void **)&tseg );
            }

        }

        // tell user about removing the segment manager
        if (say_remove_msg==TRUE) {
            POPUP_MSG(NULL, NULL,
                      "\nInfo, you just removed the last partition on drive %s.\n"
                      "If you also wish to remove the default segment manager from this "
                      "drive you can do so by destroying segment %s_mbr.\n",
                      ld->name, ld->name );
        }

        rc = 0;  // toss any errors with geometry revert and continue
    }            // with destroying the disk segment

    // remove ourselves from the disk ?
    if (remove_segmgr_from_disk == TRUE) {

        rc = GoToStartOfList( ld->parent_objects );
        if (rc == DLIST_SUCCESS) {

            rc = ExtractObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,(void **)&tseg );

            while (rc == DLIST_SUCCESS) {

                free_disk_segment(tseg);

                rc = ExtractObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,(void **)&tseg );
            };


        }

        // blast MBR
        ((struct plugin_functions_s *)ld->plugin->functions.plugin)->add_sectors_to_kill_list( ld, 0, 1);

        // blast DLAT
        ((struct plugin_functions_s *)ld->plugin->functions.plugin)->add_sectors_to_kill_list(ld,ld->geometry.sectors_per_track-1,1);

        // toss our disk private data area
        delete_disk_private_data( ld );

        rc = DLIST_SUCCESS;
    }

    // mark mbr dirty to get partitions committed
    if ( rc==0 && remove_segmgr_from_disk==FALSE ) {
            mbr->flags |= SOFLAG_DIRTY;
    }

    LOGEXITRC();
    return  rc;
}




static void GetExpandOptions( option_array_t * options,  sector_count_t  * size)
{
    int i;

    LOGENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == SEG_EXPAND_OPTION_SIZE_INDEX) {
                *size = options->option[i].value.ui64;
            }

        }
        else {

            if (strcmp(options->option[i].name, SEG_EXPAND_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }

        }
    }


    LOGEXIT();
}


/*
 *  Called to expand a data segment.  The segment will be expanded
 *  into the freespace segment that follows the data segment.
 */
static int SEG_Expand( DISKSEG *seg, DISKSEG *expand_object, dlist_t  objects, option_array_t *options )
{
    int               rc = EINVAL;
    sector_count_t    expand_sectors=0;
    DISKSEG          *freespace;
    lba_t             end_lba;
    sector_count_t    old_seg_size;
    LOGICALDISK      *ld=NULL;
    SEG_PRIVATE_DATA *pdata=NULL;
    sector_count_t    SectorsPerCylinder=0;

    LOGENTRY();


    // initial checks to see if we can do the expand
    if ( ( seg ) &&
         ( seg == expand_object ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type   == DATA_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE ) &&
         ( seg_is_volitile(seg)==TRUE ) ) {

        pdata     = (SEG_PRIVATE_DATA *)seg->private_data;
        ld        = get_logical_disk(seg);
        freespace = get_freespace_following_seg( seg );

        GetExpandOptions( options,  &expand_sectors);

        if ( freespace && ld && expand_sectors  ){

LOG_DEBUG("     Data Seg  Name: %s\n", seg->name);
LOG_DEBUG("              Start: %lld\n", seg->start);
LOG_DEBUG("               Size: %lld\n", seg->size);
LOG_DEBUG("     Freespace Name: %s\n", freespace->name);
LOG_DEBUG("              Start: %lld\n", freespace->start);
LOG_DEBUG("               Size: %lld\n", freespace->size);

            SectorsPerCylinder = get_cylinder_size(ld);

            // just in case ...
            if ( freespace->size < SectorsPerCylinder ) {
                LOG_ERROR("error, trying to expand into free space that is less than 1 cylinder\n");
                LOGEXITRC();
                return rc;
            }

            // you just never know ...
            if ( expand_sectors > freespace->size ) {
                expand_sectors = freespace->size;
            }

            // we expand in cylinder size chunks
            if (expand_sectors < SectorsPerCylinder) {
                 expand_sectors = SectorsPerCylinder;
            }
            else {
                expand_sectors = (expand_sectors/SectorsPerCylinder)*SectorsPerCylinder;
            }

            // do cylinder alignment
            end_lba = seg->start + seg->size + expand_sectors - 1;

            if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
                end_lba = roundup_to_cylinder_boundary( ld, end_lba );
            }

            // now adjust downwards if too big for freespace ...
            if ( end_lba > (freespace->start + freespace->size - 1) ) {
                end_lba = rounddown_to_cylinder_boundary(ld, end_lba-1 ) - 1;
            }


            // test if we can expand the data seg into the freespace area
            if ( ( end_lba > freespace->start ) &&
                 ( end_lba <= freespace->start+freespace->size - 1) ) {

                // calc actual expand sector count
                expand_sectors = end_lba - freespace->start + 1;

                // expand the data segment
                old_seg_size = seg->size;
                seg->size   += expand_sectors;

                // shrink the freespace segment
                freespace->start  += expand_sectors;
                freespace->size   -= expand_sectors;

                // success.
                rc = 0;

                // if we used up all the free space then discard the freespace seg
                if ( freespace->size == 0 ) {

                    rc = remove_diskseg_from_list( ld->parent_objects, freespace );
                    if (rc==0) {
                        free_disk_segment(freespace);
                    }
                    else {   // error ... backout the expand
                        LOG_ERROR("error, unable to remove the freespace segment from the disk dlist\n");
                        seg->size = old_seg_size;
                        freespace->start  -= expand_sectors;
                        freespace->size   += expand_sectors;
                    }

                }
            }
        }
    }


    if ( rc==0 ) {
        if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) fixup_disk_extd_partition_dimensions( ld );
        seg->flags |= SOFLAG_DIRTY;
    }


    LOGEXITRC();
    return rc;
}


static void GetShrinkOptions( option_array_t * options, sector_count_t * size)
{
    int i;

    LOGENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == SEG_SHRINK_OPTION_SIZE_INDEX) {
                *size = options->option[i].value.ui64;
            }

        }
        else {

            if (strcmp(options->option[i].name, SEG_SHRINK_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }
        }

    }

    LOGEXIT();
}


/*
 *  Called to shrink a data segment to new_size or next smaller increment.
 *  Then, update appropriate fields in the segment and make
 *  changes to freespace segments as needed.
 *
 */
static int SEG_Shrink( storage_object_t * seg,
                       storage_object_t * shrink_object,
                       dlist_t            objects,
                       option_array_t   * options )
{
    int               rc = EINVAL;
    sector_count_t    shrink_sector_count=0;
    u_int64_t         end_lba;
    LOGICALDISK      *ld=NULL;
    sector_count_t    new_seg_size=0;
    SEG_PRIVATE_DATA *pdata=NULL;
    sector_count_t    SectorsPerCylinder=0;

    LOGENTRY();

    // initial checks to see if we can do the shrink
    if ( ( seg ) &&
         ( seg == shrink_object ) &&
         ( seg->object_type == SEGMENT ) &&
         ( seg->data_type   == DATA_TYPE ) &&
         ( i_can_modify_seg( seg ) == TRUE ) &&
         ( seg_is_volitile(seg)==TRUE ) ) {

        pdata     = (SEG_PRIVATE_DATA *)seg->private_data;
        ld        = get_logical_disk(seg);

        GetShrinkOptions( options,  &shrink_sector_count);


        if ( (ld != NULL) &&
             (shrink_sector_count > 0) &&
             (shrink_sector_count < seg->size ) ) {

      LOG_DEBUG("     Data Seg  Name: %s\n",   seg->name);
      LOG_DEBUG("              Start: %lld\n", seg->start);
      LOG_DEBUG("               Size: %lld\n", seg->size);
      LOG_DEBUG("Shrink Sector Count: %lld\n", shrink_sector_count );

            SectorsPerCylinder = get_cylinder_size(ld);

            // we shrink in cylinder size chunks
            if (shrink_sector_count < SectorsPerCylinder) {
                 shrink_sector_count = SectorsPerCylinder;
            }
            else {
                shrink_sector_count = (shrink_sector_count/SectorsPerCylinder)*SectorsPerCylinder;
            }

            // resulting seg size
            new_seg_size = seg->size - shrink_sector_count;

            // make sure it ends on cylinder boundary
            end_lba = seg->start + new_seg_size - 1;
            if (ends_on_cylinder_boundary(ld, end_lba)==FALSE) {
                end_lba = rounddown_to_cylinder_boundary( ld, end_lba ) - 1;
            }

            if ( end_lba >= (seg->start + seg->size - 1) ) {
                end_lba = rounddown_to_cylinder_boundary(ld, end_lba ) - 1;
            }

            // final test if we can shrink the segment
            if (  ( end_lba > seg->start ) &&
                  ( end_lba < (seg->start+seg->size-1)) ) {

                // actual new seg size
                new_seg_size = end_lba - seg->start + 1;

                // shrink the data segment
                seg->size = new_seg_size;

                // success.
                rc = 0;

            }

            if ( rc==0 ) {

                // fixup logical drive info
                if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) fixup_disk_extd_partition_dimensions( ld );

                // collect any freespace we released
                find_freespace_on_disk( ld );

                // mark seg dirty for commit
                seg->flags |= SOFLAG_DIRTY;
            }

        }
        else {
            LOG_ERROR("error, something wrong with shrink sector count, cant shrink segment\n");
            rc = EINVAL;
        }

    }
    else {
        LOG_ERROR("error, something wrong with parms\n");
        rc = EINVAL;
    }


    LOGEXITRC();

    return rc;
}



static int SEG_AddSectorsToKillList( DISKSEG *seg, lsn_t lsn, sector_count_t count)
{
    int                         rc = EINVAL;
    LOGICALDISK                *ld;
    struct plugin_functions_s  *Fncs;

    LOGENTRY();

    if ( lsn + count > seg->size ) {
        rc = EINVAL;
    }
    else {

        ld = get_logical_disk(seg);

        if (ld) {

            Fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

            rc   = Fncs->add_sectors_to_kill_list( ld, seg->start+lsn, count);
        }

    }

    LOGEXITRC();
    return rc;
}


static int SEG_CommitChanges( storage_object_t *obj, uint phase )
{
    int          rc = 0;
    int          rc2 = 0;
    LOGICALDISK *ld=NULL;
    DISKSEG     *clean_seg;

    LOGENTRY();
    LOG_DEBUG("object= %s  commit phase= %d\n", obj->name, phase );

    // get logical disk
    if (obj->object_type == DISK) {
        ld = obj;
    }
    else if ( obj->object_type == SEGMENT ) {
        ld = get_logical_disk(obj);
    }

    // commit changes if we own the segments on the disk
    if ( i_can_modify_disk( ld ) == TRUE ) {

        // no work during commit phase 0
        if ( ( phase > 0 ) &&
             ( obj->flags & SOFLAG_DIRTY )) {

            rc = Commit_Disk_Partition_Tables(ld);

            // if successful ... mark all segments on the disk clean
            if (rc == 0) {

                if ( GoToStartOfList( ld->parent_objects ) == DLIST_SUCCESS) {

                    rc2 = GetObject( ld->parent_objects,sizeof(DISKSEG),SEGMENT_TAG,NULL,TRUE,(void **) &clean_seg );

                    while ( rc2 == DLIST_SUCCESS ) {

                        clean_seg->flags &= ~SOFLAG_DIRTY;

                        rc2 = GetNextObject( ld->parent_objects, sizeof(DISKSEG), SEGMENT_TAG, (void **) &clean_seg );
                    }
                }

            }

        }

    }
    else {
        rc = EINVAL;
    }


    LOGEXITRC();
    return rc;
}



static int SEG_Read( DISKSEG        *seg,
                     lsn_t           offset,
                     sector_count_t  count,
                     void           *buffer )
{
    int                         rc = ENODEV;
    LOGICALDISK                *ld;
    struct plugin_functions_s  *fncs;

    LOGENTRY();

    if ( offset + count > seg->size ) {
        rc = EINVAL;
    }
    else {

        ld = get_logical_disk(seg);

        if (ld) {

            fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

            rc   = fncs->read( ld, seg->start+offset, count, buffer);
        }

    }

    LOGEXITRC();
    return rc;
}

static int SEG_Write( DISKSEG        *seg,
                      lsn_t           offset,
                      sector_count_t  count,
                      void           *buffer )
{
    int                         rc = ENODEV;
    LOGICALDISK                *ld;
    struct plugin_functions_s  *fncs;


    LOGENTRY();

    if ( offset + count > seg->size ) {
        rc = EINVAL;
    }
    else {

        ld = get_logical_disk(seg);

        if (ld) {

            fncs = (struct plugin_functions_s *)ld->plugin->functions.plugin;

            rc   = fncs->write( ld, seg->start+offset, count, buffer);
        }

    }

    LOGEXITRC();
    return rc;
}


/*
 * This call notifies you that your object is being made into (or part of)
 * a volume or that your object is no longer part of a volume.  The "flag"
 * parameter indicates whether the volume is being created (TRUE) or
 * removed (FALSE).
 */
static void SEG_set_volume(storage_object_t * object, BOOLEAN flag)
{
    return;
}



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s sft={

    // the following routines are found above
    setup_evms_plugin:                   SEG_SetupEVMSPlugin,
    cleanup_evms_plugin:                 SEG_Cleanup,
    can_set_volume:                      SEG_can_set_volume,
    can_delete:                          SEG_CanDestroy,
    can_expand:                          SEG_CanExpand,
    can_expand_by:                       SEG_CanExpandBy,
    can_shrink:                          SEG_CanShrink,
    can_shrink_by:                       SEG_CanShrinkBy,
    discover:                            SEG_Discover,
    create:                              SEG_CreateSegment,
    delete:                              SEG_DestroySegment,
    expand:                              SEG_Expand,
    shrink:                              SEG_Shrink,
    add_sectors_to_kill_list:            SEG_AddSectorsToKillList,
    commit_changes:                      SEG_CommitChanges,
    read:                                SEG_Read,
    write:                               SEG_Write,
    set_volume:                          SEG_set_volume,

    // the following routines found in segoptions.c
    get_option_count:                    SEG_GetOptionCount,
    init_task:                           SEG_InitTask,
    set_option:                          SEG_SetOption,
    set_objects:                         SEG_SetObjects,
    get_info:                            SEG_GetInfo,
    get_plugin_info:                     SEG_GetPluginInfo
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t segmgr_plugin_record = {

    id:                               SetPluginID(EVMS_OEM_IBM, EVMS_SEGMENT_MANAGER, 1 ),

    version:                          {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

    required_api_version:             {3,0,0},

    short_name:                       "DefaultSegMgr",
    long_name:                        "Default Storage Segment Manager",
    oem_name:                         "IBM",

    functions:                        {plugin: &sft},

    container_functions:              NULL

};

// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
    &segmgr_plugin_record,
    NULL
};
