/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aix_pv.c
 *
 * Description: This file contains functions related to managing AIX PVs.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "aixregmgr.h"


/* aix_find_pv_by_number
 *
 *	Search the objects_consumed list in this container for the PV with
 *	the specified number.
 */
storage_object_t * aix_find_pv_by_number(storage_container_t	* container,
					unsigned int		pv_num )
{
	storage_object_t	* object;
	aix_pv_data_t		* pv;
	int			rc;

	LOG_ENTRY;

	FOR_EACH(object, container->objects_consumed) {
		pv = object->consuming_private_data;
		if ( pv->pv_head->pv_num == pv_num ) {
			RETURN(object);
		}
	}

	RETURN(NULL);
}

/* aix_deallocate_pv
 *
 *	Free up all memory used by the specified PV.
 */
void aix_deallocate_pv( storage_object_t * object )
{
	aix_pv_data_t * pv = object->consuming_private_data;

	LOG_ENTRY;

	if ( pv ) {
		object->consuming_private_data = NULL;

		if ( pv->ipl ) {
			aix_engine->engine_free(pv->ipl);
		}

		if ( pv->lvm ) {
			aix_engine->engine_free(pv->lvm);
		}

		aix_engine->engine_free(pv);
	}

	LOG_EXIT(0);
}


/* aix_allocate_pv
 *
 *	Allocate and initialize a PV structure for the specified object.
 */
int aix_allocate_pv(	storage_object_t	* object,
			aix_ipl_rec_t		* ipl,
			aix_lvm_rec_t		* lvm )
{
	aix_pv_data_t	* new_pv;

	LOG_ENTRY;

	// Allocate the pv structure
	new_pv = aix_engine->engine_alloc(sizeof(aix_pv_data_t));
	if ( ! new_pv ) {
		LOG_CRITICAL("Memory error creating PV structure for %s.\n", object->name);
		aix_engine->engine_free(ipl);
		aix_engine->engine_free(lvm);
		RETURN(ENOMEM);
	}

	new_pv->ipl		= ipl;
	new_pv->lvm		= lvm;

	object->consuming_private_data = new_pv;

	RETURN(0);
}


/* aix_compare_timestamps
 *
 *	Compare the various time-stamps in the VG headers and trailers for the
 *	specified PV to determine which VGDA is active.
 */
int aix_compare_timestamps( storage_object_t * object )
{
	aix_pv_data_t * pv = object->consuming_private_data;

	LOG_ENTRY;

	if ( COMPARE_TIMESTAMPS(pv->vg_head[0]->vg_timestamp, pv->vg_tail[0]->timestamp) ) {
		if ( COMPARE_TIMESTAMPS(pv->vg_head[1]->vg_timestamp, pv->vg_tail[1]->timestamp) ) {
			if ( COMPARE_TIMESTAMPS(pv->vg_head[0]->vg_timestamp, pv->vg_head[1]->vg_timestamp) ) {
				// All timestamps match. Yea!
				pv->pv_state = AIX_PV_STATE_VALID;
			}
			else {
				// Both VGDAs are good, but timestamps are
				// different. Can't tell yet which one is
				// correct. 
				pv->pv_state = AIX_PV_STATE_EITHER_VGDA;
			}
		}
		else {
			// First VGDA is good, second is bad.
			pv->pv_state = AIX_PV_STATE_FIRST_VGDA;
		}
	}
	else {
		if ( COMPARE_TIMESTAMPS(pv->vg_head[1]->vg_timestamp, pv->vg_tail[1]->timestamp) ) {
			// First VGDA is bad, second is good.
			pv->pv_state = AIX_PV_STATE_SECOND_VGDA;
		}
		else {
			// This should never happen.
			LOG_CRITICAL("All four VG timestamps for %s are different. What the hell?!?\n", object->name);
			RETURN(EINVAL);
		}
	}

	RETURN(0);
}


/* aix_read_pv_metadata
 *
 *	Read the AIX metadata from the specified object. Check for appropriate
 *	signatures. If found, read both VG headers and both VG trailers, and
 *	make an initial comparison of the timestamps to determine which VGDA
 *	should be used later.
 */
int aix_read_pv_metadata( storage_object_t * object )
{
	aix_ipl_rec_t	* ipl;
	aix_lvm_rec_t	* lvm;
	int		rc;

	LOG_ENTRY;

	// Read the IPL record from this object.
	rc = aix_read_ipl(object, &ipl);
	if (rc) {
		// This object is not an AIX PV.
		RETURN(rc);
	}

	// Read the LVM record from this object.
	rc = aix_read_lvm(object, &lvm);
	if (rc) {
		// This object is not managed by the AIX LVM.
		aix_engine->engine_free(ipl);
		RETURN(rc);
	}

	// Allocate a PV data structure. This attaches the PV
	// private data to the specified object.
	rc = aix_allocate_pv(object, ipl, lvm);
	if (rc) {
		RETURN(rc);
	}

	// Read the VG headers and trailers.
	rc = aix_read_vg_headers(object);
	if (rc) {
		RETURN(rc);
	}
	rc = aix_read_vg_trailers(object);
	if (rc) {
		RETURN(rc);
	}

	// Compare the timestamps to determine the state of the PV.
	rc = aix_compare_timestamps(object);
	if (rc) {
		RETURN(rc);
	}

	RETURN(0);
}

