/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: memman.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include <fullengine.h>
#include "memman.h"
#include "engine.h"


#define MEM_OBJ_HEAD_SIGNATURE  0x54455448  /* "MEMH" */

typedef struct mem_object_header_s {
    u_int32_t   signature;
    void        (*free_function)(void *);
} mem_object_header_t;

/*
 * Allocate a structure for returning to an application.
 * A header, which contains a signature and a pointer to a function for
 * freeing the contents of the user structure, is prepended to the buffer
 * that is allocated.
 */
void * alloc_app_struct(uint size, void (*free_function)(void *)) {

    void * pMem;

    mem_object_header_t * mem_obj = calloc(1, sizeof(mem_object_header_t) + size);

    LOG_PROC_ENTRY();

    if (mem_obj != NULL) {
        mem_obj->signature = MEM_OBJ_HEAD_SIGNATURE;
        mem_obj->free_function = free_function;

        pMem = mem_obj + 1;

    } else {
        pMem = NULL;
    }

    LOG_PROC_EXIT_PTR(pMem);
    return pMem;
}


/*
 * Free an application buffer.
 * Look for a header before the beginning of the buffer.  If one is there and
 * is has a function for freeing the contents of the buffer, then call the
 * function.  If we don't find a header before the buffer, ignore this call.
 * In that case we don't know where the memory came from so we don't know how
 * to free it.  It will be cleaned up when the process exits.
 */
void evms_free(void * buffer) {

    mem_object_header_t * mem_obj = buffer - sizeof(mem_object_header_t);

    LOG_PROC_ENTRY();

    LOG_DEBUG("Request to free application buffer at %p.\n", buffer);

    if (buffer != NULL) {
        if (mem_obj->signature == MEM_OBJ_HEAD_SIGNATURE) {
            if (mem_obj->free_function != NULL) {
                mem_obj->free_function(buffer);

            } else {
                LOG_DEBUG("Application buffer has no supplementary free_function().\n");
            }

            free(mem_obj);

        } else {
            LOG_WARNING("Application buffer does not have our memory object header.  Request ignored.\n");
        }
    }

    LOG_PROC_EXIT_VOID();
}
