/* ethos-c-plugin-loader.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include <Python.h>
#include <gmodule.h>
#include <pyglib.h>
#include <pygobject.h>

#include <ethos/ethos-error.h>
#include <ethos/ethos-plugin-loader.h>
#include <ethos/ethos-plugin-info-private.h>
#include <ethos/ethos-version.h>
#include "ethos-python-plugin-loader.h"

static PyObject* PyEthosPlugin_Type = NULL;
static PyObject* PyEthos_Module     = NULL;

struct _EthosPythonPluginLoaderPrivate
{
	gboolean do_finalize;
	gboolean init_failed;
};

static G_CONST_RETURN gchar*
ethos_python_plugin_loader_get_name (EthosPluginLoader *plugin_loader)
{
	return "python";
}

static void
ethos_python_plugin_loader_gc (EthosPluginLoader *plugin_loader)
{
	PyGILState_STATE state = PyGILState_Ensure ();
	while (PyGC_Collect ())
		;
	PyGILState_Release (state);
}

static void
ethos_python_plugin_loader_initialize (EthosPluginLoader *plugin_loader,
                                       EthosManager      *manager)
{
	EthosPythonPluginLoaderPrivate *priv;

	PyObject    *pymodule  = NULL,
		    *pydict    = NULL,
		    *pyversion = NULL;

	g_return_if_fail (ETHOS_IS_PYTHON_PLUGIN_LOADER (plugin_loader));
	g_return_if_fail (ETHOS_IS_MANAGER (manager));

	priv = ETHOS_PYTHON_PLUGIN_LOADER (plugin_loader)->priv;

	if (!Py_IsInitialized ()) {
		priv->do_finalize = TRUE;
		Py_InitializeEx (0);
	}

	if (!(pymodule = PyImport_ImportModule ("ethos"))) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		priv->init_failed = TRUE;
		return;
	}

	if (!(pydict = PyModule_GetDict (pymodule))) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		return;
	}

	if (!(pyversion = Py_BuildValue ("(iii)",
	                                 ETHOS_MAJOR_VERSION,
	                                 ETHOS_MINOR_VERSION,
	                                 ETHOS_MICRO_VERSION)))
	{
		if (PyErr_Occurred ())
			PyErr_Print ();
		return;
	}

	PyDict_SetItemString (pydict, "version", pyversion);
	Py_XDECREF (pyversion);

	if (!(PyEthosPlugin_Type = PyDict_GetItemString (pydict, "Plugin"))) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		return;
	}

	Py_INCREF (PyEthos_Module = pymodule);
	Py_INCREF (PyEthosPlugin_Type);
}

static void
ethos_python_plugin_loader_unload (EthosPluginLoader *plugin_loader)
{
	EthosPythonPluginLoaderPrivate *priv;

	g_return_if_fail (ETHOS_IS_PYTHON_PLUGIN_LOADER (plugin_loader));

	priv = ETHOS_PYTHON_PLUGIN_LOADER (plugin_loader)->priv;

	if (priv->do_finalize) {
		Py_Finalize ();
	}
}

static void
ensure_plugin_dir_in_sys_path (EthosPluginInfo *plugin_info)
{
	const gchar *path;
	gchar       *dir;
	PyObject    *pysyspath = NULL,
		    *pypath    = NULL;

	if (!(path = ethos_plugin_info_get_filename (plugin_info)))
		return;

	if (!(dir = g_path_get_dirname (path)))
		return;

	if (!(pypath = PyString_FromString (dir)))
		goto handle_error;

	if (!(pysyspath = PySys_GetObject ("path")))
		goto handle_error;

	if (!PySequence_Contains (pysyspath, pypath))
		PyList_Insert (pysyspath, 0, pypath);

	goto cleanup;

handle_error:
	if (PyErr_Occurred ())
		PyErr_Print ();

cleanup:
	g_free (dir);
	Py_XDECREF (pypath);
}

static EthosPlugin*
ethos_python_plugin_loader_load (EthosPluginLoader  *plugin_loader,
                                 EthosPluginInfo    *plugin_info,
                                 GError            **error)
{
	EthosPythonPluginLoaderPrivate *priv;

	const gchar *module;
	PyObject    *pymodule = NULL,
		    *pydict   = NULL;
	EthosPlugin *plugin   = NULL;

	g_return_val_if_fail (ETHOS_IS_PYTHON_PLUGIN_LOADER (plugin_loader), NULL);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);

	priv = ETHOS_PYTHON_PLUGIN_LOADER (plugin_loader)->priv;

	if (priv->init_failed) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "An error occurred during python initialization."
		             " All python plugins are disabled.");
		return NULL;
	}

	ensure_plugin_dir_in_sys_path (plugin_info);

	if (!(module = ethos_plugin_info_get_module (plugin_info))) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "Plugin description is missing \"Module=\"");
		return NULL;
	}

	if (!(pymodule = PyImport_ImportModule (module))) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "Module \"%s\" could not be loaded.",
		             module);
		return NULL;
	}

	if (!(pydict = PyModule_GetDict (pymodule))) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "Error retrieving __dict__ from module");
		return NULL;
	}

	PyObject     *key,
		     *value,
		     *pyplugin,
		     *pytype = NULL;
	Py_ssize_t    pos    = 0;

	while (PyDict_Next (pydict, &pos, &key, &value)) {
		if (!PyType_Check (value)) continue;
		if (PyType_IsSubtype ((PyTypeObject*)value,
		                      (PyTypeObject*)PyEthosPlugin_Type)) {
			pytype = value;
			break;
		}
	}

	if (!pytype) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "The python module %s did not include an "
		             "EthosPlugin.", module);
		return NULL;
	}

	g_assert (PyCallable_Check (pytype));

	PyObject *args   = PyTuple_New (0),
		 *kwargs = PyDict_New ();

	pyplugin = PyObject_Call (pytype, args, kwargs);

	Py_XDECREF (args);
	Py_XDECREF (kwargs);
	Py_XDECREF (pytype);

	if (!pyplugin) {
		if (PyErr_Occurred ())
			PyErr_Print ();
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN,
		             "Could not create instance of python plugin");
		return NULL;
	}

	plugin = ETHOS_PLUGIN (((PyGObject*)pyplugin)->obj);

	return plugin;
}

static void
ethos_python_plugin_loader_base_init (EthosPluginLoaderIface *iface)
{
	iface->get_name   = ethos_python_plugin_loader_get_name;
	iface->gc         = ethos_python_plugin_loader_gc;
	iface->initialize = ethos_python_plugin_loader_initialize;
	iface->unload     = ethos_python_plugin_loader_unload;
	iface->load       = ethos_python_plugin_loader_load;
}

G_DEFINE_TYPE_EXTENDED (EthosPythonPluginLoader,
                        ethos_python_plugin_loader,
                        G_TYPE_OBJECT,
                        0,
                        G_IMPLEMENT_INTERFACE (ETHOS_TYPE_PLUGIN_LOADER,
                                               ethos_python_plugin_loader_base_init));

static void
ethos_python_plugin_loader_finalize (GObject *object)
{
	G_OBJECT_CLASS (ethos_python_plugin_loader_parent_class)->finalize (object);
}

static void
ethos_python_plugin_loader_class_init (EthosPythonPluginLoaderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_python_plugin_loader_finalize;
	g_type_class_add_private (object_class, sizeof (EthosPythonPluginLoaderPrivate));
}

static void
ethos_python_plugin_loader_init (EthosPythonPluginLoader *plugin_loader)
{
	plugin_loader->priv = G_TYPE_INSTANCE_GET_PRIVATE (plugin_loader,
	                                                   ETHOS_TYPE_PYTHON_PLUGIN_LOADER,
	                                                   EthosPythonPluginLoaderPrivate);
}

EthosPluginLoader*
ethos_python_plugin_loader_new ()
{
	return g_object_new (ETHOS_TYPE_PYTHON_PLUGIN_LOADER, NULL);
}

G_MODULE_EXPORT EthosPluginLoader*
ethos_plugin_loader_register (void)
{
	return ethos_python_plugin_loader_new ();
}
