#-----------------------------------------------------------------------------
#
#  Copyright (c) 2006 by Enthought, Inc.
#  All rights reserved.
#
#  Author: Greg Rogers
#
#-----------------------------------------------------------------------------

from enthought.traits.api import Bool, HasTraits, TraitError
from enthought.traits.ui.api import Handler


class HandlerModel( HasTraits ):
    """ State used by the handler. """
    
    # When False, the Model.name trait is not allowed to be empty, ''.
    allow_empty_string = Bool


class MyModelHandler( Handler ):
    """ Custom Handler for the MyModel object and view.
    Validate MyModel.name is consitent with the HandlerModel flag. 
    """

    ###########################################################################
    ### Handler interface
    ###########################################################################
    
    def init(self, info):
        """ Force validation of the name trait w.r.t. the allow_empty_string
        flag.
        
        This method is called after all user interface elements have been
        created, but before the user interface is displayed. Use this method to
        further customize the user interface before it is displayed.
        """
        
        # can't call handlermodel_allow_empty_string_changed, not initialized yet.
        self._force_name_validation(info)
        return
    
    def setattr ( self, info, object, name, value ):
        """ Validate the request to change the named trait on object to the
        specified value.  Vaildation errors raise TraitError.
        """
        if name == 'name':
            self._validate_name(info, value)
        
        return super(MyModelHandler,self).setattr( info, object, name, value )

    ###########################################################################
    ### change notification handlers
    ###########################################################################

    def handlermodel_allow_empty_string_changed(self, info):
        """ 'allow_empty_string' has changed, check the name trait to ensure
        that it is consistent with the current setting.
        """
        if info.initialized:
            self._force_name_validation(info)
        return

    ###########################################################################
    ### MyModelHandler protected methods
    ###########################################################################
    
    def _force_name_validation(self, info):
        """ Force the name trait to be revalidated by the trait handler
        by poking the ui control for the name trait.
        """
        info.ui._editors[0].update_object(None)
        
        return
    
    def _validate_name(self, info, value):
        """ Raise TraitError if the proposed value for name is not consistent
        with the allow_empty_string setting.
        """
        handler_model = info.ui.context['handlermodel']
        if not handler_model.allow_empty_string and value.strip() == '':
            raise TraitError, 'empty string not allowed'
        
        return
    

### EOF
    