#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: David C. Morrill
# Date: 11/29/2004
# Description: Simple example using traits to convert from one unit system to
#  another.
#
#  Note that other than the 'anytrait_changed' method, and the last line, which
#  launches the application, the rest of the code is simply declarative.
#------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from enthought.traits.api    import HasStrictTraits, Trait, CFloat
from enthought.traits.ui.api import View, Group

#-------------------------------------------------------------------------------
#  Help text:
#-------------------------------------------------------------------------------

ViewHelp = """
This program converts length measurements from one unit system to another.

<p>Select the input and output units using the drop down combo-boxes in the
<b>Input:</b> and <b>Output:</b> sections respectively. Type the input quantity 
to convert into the left most text box. The output value corresponding to the 
current input value will automatically be updated in the <b>Output:</b> 
section.</p>

<p>Use the <b>Undo</b> and <b>ReDo</b> buttons to undo and redo changes you
have made to any of the input fields.</p>
"""

#-------------------------------------------------------------------------------
#  Trait definitions:
#-------------------------------------------------------------------------------

# Units trait maps all units to centimeters:
Units = Trait( 'inches', { 'inches':      2.54,
                           'feet':        (12 * 2.54),
                           'yards':       (36 * 2.54),
                           'miles':       (5280 * 12 * 2.54),
                           'millimeters': 0.1,
                           'centimeters': 1.0,
                           'meters':      100.0,
                           'kilometers':  100000.0 } )
                                 
#-------------------------------------------------------------------------------
#  'Converter' class:
#-------------------------------------------------------------------------------

class Converter ( HasStrictTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------
    
    input_amount  = CFloat( 12.0,    desc = "the input quantity" )
    input_units   = Units( 'inches', desc = "the input quantity's units" )
    output_amount = CFloat( 1.0,     desc = "the output quantity" ) 
    output_units  = Units( 'feet',   desc = "the output quantity's units" )

    #---------------------------------------------------------------------------
    #  User interface views:
    #---------------------------------------------------------------------------
        
    traits_view = View( Group( ( 'input_amount',   'input_units',  
                                 '-[Input:]' ),
                               ( 'output_amount~', 'output_units', 
                                 '-[Output:]' ), 
                               orientation = 'horizontal',
                               show_border = False,
                               help        = ViewHelp ),
                        title = 'Units Converter' )
    
    #---------------------------------------------------------------------------
    #  Event handlers:
    #---------------------------------------------------------------------------
    
    def _anytrait_changed ( self ):
        self.output_amount = ((self.input_amount * self.input_units_) / 
                              self.output_units_)
                         
#-------------------------------------------------------------------------------
#  Create a converter and display its user interface:
#-------------------------------------------------------------------------------
                         
Converter().configure_traits()

