#!/usr/bin/perl
#
# Copyright 1999 by Ben Collins <bcollins@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#

$|=1;

use Net::LDAP;
use Net::LDAP::LDIF;

$version='0.5.0.0';

# Server settings
$server = "bugs.debian.org";
$port   = "35567";
$base   = "ou=Bugs,o=Debian Project,c=US";
$attrs  = [
	'bugid',
	'originater',
	'date',
	'subject',
	'package',
	'severity',
	'done',
	'mergedwith'
	  ];

%fields = (
	bugid		=> 'Bug ID' ,
	originater	=> 'Submitter' ,
	date		=> 'Date Submitted' ,
	subject		=> 'Summary' ,
	package		=> 'Package' ,
	severity	=> 'Severity' ,
	done		=> 'Closed By',
	mergedwith	=> 'Merged With'
	  );

# stupid unordered hashes :P
@field_keys = ('bugid', 'subject', 'severity', 'package', 'originater', 'date',
	'done', 'mergedwith');

# Some default options, these can be changed to suit
$purge_closed  = 0;
$merged_only   = 0;
$do_directory  = ".";
$do_save       = "./bugs";
$do_stats      = 0;
$min_thresh    = 0;
$max_thresh    = 5;

# initializers
$do_package    = "";
$do_control    = "";
$do_bugid      = "";
$total_bugs    = 0;
$total_closed  = 0;
@merged_bugs   = ();

%severities    = (
	fixed        => 0,
	wishlist     => 0,
	normal       => 0,
	important    => 0,
	critical     => 0,
	grave        => 0,
	release_crit => 0
	);

%thresholds   = (
	fixed        => 0,
	wishlist     => 1,
	normal       => 2,
	important    => 3,
	critical     => 4,
	grave        => 5,
	);

# Used to warn of unimplemented features
sub not_impl ($) {
    my $func = shift;
    print "Warning: $func is not yet implemented.\n";
}

# simple usage output
sub do_usage () {
    print <<EOF;
GetBugs v$version by Ben Collins <bcollins\@debian.org>
Usage: getbugs [ options ] < command >

  Commands (one of):
    summary  Retrieve summary of bugs <default>
    update   Updates bug summaries
    report   Retrieve reports for bugs
    log      Retrieve logs for bugs
    close    closes the bug referenced in --package or --bugid

  Options:
    -p,--package <package>  Operate the command only on this
                            package. By default this saves output
                            to the current directory.
    -b,--bugid <bug#>       Operate the command only on this bug #
    -c,--control <control>  Operate the command only on packages
                            in this control file.
    -s,--save <dir>         Save files to this directory (defaults
                            to ./bugs/ ).
    -d,--directory <dir>    Search for control files in this
                            directory (default is to search current
                            directory recursively).
    -i,--purge-closed       Do not keep information on closed bugs.
    -l,--low-threshold     Minimum bug severity to keep.
    -t,--high-threshold    Maximum bug severity to keep.
    -m,--merged-only        When encountering merged bugs, only list
                            the first one we come across.
    -x,--stats              Give stats (for control and package lookups)
    -v,--version            Print version info and exit.
    -h,--help               This usage output.

NOTE: * --directory cannot be used with --package or --control
      * If no options are given, getbugs will search the
        current directory for debian/control files and execute
        the command on the packages found in all of them

EOF
    exit 1;
}

sub do_badexit ($) {
    my $msg = shift;
    print "\nError: $msg\n(use --help for usage)\n\n";
    exit 1;
}

# Parse the arguments, if any
while (@ARGV && $ARGV[0] =~ m/^-/) {
    $_=shift(@ARGV);
    if (m/^-h$/ || m/^--help/) {
	do_usage();
    } elsif (m/^-v$/ || m/^--version/) {
	print "GetBugs v$version by Ben Collins <bcollins\@debian.org>\n";
	exit 0;
    } elsif (m/^-i/ || m/^--purge-closed/) {
	print "(not keeping closed bugs)\n";
	$purge_closed=1;
    } elsif (m/^-l/ || m/^--low-threshold/) {
	$tmp = shift;
	if ($tmp =~ m/(^-|^$)/) {
	    do_badexit("--low-threshold requires an argument");
	}
	if (!defined($thresholds{$tmp})) {
	    do_badexit("unknown severity $tmp");
	}
	$min_thresh = $thresholds{$tmp};
    } elsif (m/^-t/ || m/^--high-threshold/) {
	$tmp = shift;
	if ($tmp =~ m/(^-|^$)/) {
	    do_badexit("--high-threshold requires an argument");
	}
	if (!defined($thresholds{$tmp})) {
	    do_badexit("unknown severity $tmp");
	}
	$max_thresh = $thresholds{$tmp};
    } elsif (m/^-m/ || m/^--merged-only/) {
	print "(only showing unique merged bugs)\n";
	$merged_only=1;
    } elsif (m/^-x$/ || m/^--stats/) {
	$do_stats = 1;
    } elsif (m/^-p$/ || m/^--package/) {
	$do_package=shift;
	if ($do_package =~ m/(^-|^$)/) {
	    do_badexit("--package requires an argument");
	}
    } elsif (m/^-c$/ || m/^--control/) {
	$do_control=shift;
	if ($do_control =~ m/(^-|^$)/) {
	    do_badexit("--control requires an argument");
	}
    } elsif (m/^-d$/ || m/^--directory/) {
	$do_directory=shift;
	if ($do_directory =~ m/(^-|^$)/) {
	    do_badexit("--directory requires an argument");
	}
    } elsif (m/^-s$/ || m/^--save/) {
	$do_save=shift;
	if ($do_save =~ m/(^-|^$)/) {
	    do_badexit("--save requires an argument");
	}
    } elsif (m/^-b$/ || m/^--bugid/) {
	$do_bugid=shift;
	if ($do_bugid =~ m/(^-|^$)/) {
	    do_badexit("--bugid requires an argument");
	}
    } else {
	do_badexit("unknown option $_");
    }
}

# A few sanity checks before we continue
if ($do_package ne "" && $do_bugid ne "") {
    do_badexit("--package and --bugid cannot be used together");
} elsif (($do_package ne ""  || $do_bugid ne "") && $do_directory ne ".") {
    do_badexit("--directory cannot be used with --package or --bugid");
} elsif (($do_package ne "" || $do_bugid ne "") && $do_control ne "") {
    do_badexit("--control cannot be used with --package or --bugid");
} elsif ($do_directory ne "." && $do_control ne "") {
    do_badexit("--control and --directory cannot be used together");
} elsif ($max_thresh < $min_thresh) {
    do_badexit("Max threshold must be greater than or equal to the min threshold");
}

# Outputs the results from the LDAP search into a formated file
# in $do_save directory with a file name of the package
sub output_package ($;$) {
    my $ldap = shift;
    my $package = shift;

    my $count  = 0;
    my $closed = 0;

    print "$package: ";

    my $mesg = $ldap->search (
	base   => "$main::base",
	filter => "(package=$package)",
	attrs  => @attrs
	) or do_badexit($@);

    open (SUMM, "> ${main::do_save}/${package}");
    foreach $entry ($mesg->entries) {
	$count++;
	if ($merged_only) {
	    my @test = grep (/@{$entry->get('bugid')}[0]/, @main::merged_bugs);
	    if (@test) {
		next;
	    } else {
		my @merged = @{$entry->get('mergedwith')};
		if (@merged) { push @main::merged_bugs, (@merged); }
	    }
	}
	if (defined ($entry->get('done'))) {
	    $closed++;
	    if ($main::purge_closed) { next; }
	}
	if ($thresholds{@{$entry->get('severity')}[0]} < $min_thresh ||
	    $thresholds{@{$entry->get('severity')}[0]} > $max_thresh) { next; }
	if ($do_stats) {
	    ($_)=@{$entry->get('severity')};
	    $main::severities{$_}++;
	    if (m/^(important|grave|critical)$/) { $main::severities{'release_crit'}++; }
	}
	foreach $attr (@field_keys) {
	    next if $attr eq 'done' && !defined($entry->get('done'));
	    next if $attr eq 'mergedwith' && !@{$entry->get('mergedwith')};
	    print SUMM "$main::fields{$attr}: @{$entry->get($attr)}\n";
	}
	print SUMM "\n";
    }
    close SUMM;

    if ( $count == 0 ) {
	print "bug free.\n";
    } else {
	print "$count bugs";
	if ( $closed != 0 ) {
	    print ", $closed of which are closed";
	}
	print ".\n";
    }
    if ( $count == 0 || ( -z "${main::do_save}/${package}" )) {
	system ("rm -f ${main::do_save}/${package}");
    }
    $main::total_bugs += $count;
    $main::total_closed += $closed;
}

# Outputs the results from the LDAP search on $do_bugid to 
# stdout. Called with "summary", "report" or "log".
sub output_bugid ($;$) {
    my $ldap = shift;
    my $type = shift;
    my $mesg;
  foreach $bugid (split(',', $main::do_bugid)) {
    if ($type eq "summary") {
	$mesg = $ldap->search (
	    base   => "$main::base",
	    filter => "(bugid=$bugid)",
	    ) or do_badexit($@);
	$entry = $mesg->pop_entry;
	if (!defined($entry->get('bugid'))) {
	    print STDERR "$bugid does not exist\n";
	} else {
	    foreach $attr (@field_keys) {
		next if $attr eq 'done' && !defined($entry->get('done'));
		print "$main::fields{$attr}: @{$entry->get($attr)}\n";
	    }
	}
    } elsif ($type eq "report") {
	$mesg = $ldap->search (
	    base   => "$main::base",
	    filter => "(bugid=$bugid)",
	    attrs  => [ 'report' ],
	    ) or do_badexit($@);
	$entry = $mesg->pop_entry;
	if(defined ($entry)) { @attr = $entry->get("report"); }
	if(defined ($attr[0])) {
	    open (PAGER, "| /usr/bin/sensible-pager");
	    print PAGER "$attr[0]\n";
	    close PAGER;
	}
    } elsif ($type eq "log") {
	$mesg = $ldap->search (
	    base   => "$main::base",
	    filter => "(bugid=$bugid)",
	    attrs  => [ 'log' ],
	    ) or do_badexit($@);
	$entry = $mesg->pop_entry;
	if(defined ($entry)) { @attr = $entry->get("log"); }
	if(defined ($attr[0])) {
	    open (PAGER, "| /usr/bin/sensible-pager");
	    print PAGER "$attr[0]\n";
	    close PAGER;
	}
    } else {
	do_badexit("output_bugid() called with unknown type '$type'");
    }
    print "\n";
  }
}

# Totals up all the open and closed bugs
sub do_totals () {
    my $open = $main::total_bugs - $main::total_closed;
    my $total_msg;
    my $closed_msg;
    my $open_msg;
    if ($main::total_bugs == 1) {
	$total_msg = "$main::total_bugs bug";
    } else {
	$total_msg = "$main::total_bugs bugs";
    }
    if ($main::total_closed == 1) {
	$closed_msg = ", $main::total_closed is closed";
    } elsif ($main::total_closed == 0) {
	$closed_msg = "";
    } else {
	$closed_msg = ", $main::total_closed are closed";
    }
    if ($open == 1) {
	$open_msg = ", $open is open";
    } elsif (($open == 0 && $main::total_bugs == 0) || $main::total_bugs == $open) {
	$open_msg = "";
    } else {
	$open_msg = ", $open are open";
    }
    print "Totals: ${total_msg}${closed_msg}${open_msg}.\n";
    if ($do_stats) {
	print "\nStatistics...\n";
	my $index = 0;
	if ($main::severities{'fixed'}) {
	    print "  NMU Fixes:        $main::severities{'fixed'}\n";
	    $index += $main::severities{'fixed'} * 0;
	}
	if ($main::severities{'wishlist'}) {
	    print "  Wishlist:         $main::severities{'wishlist'}\n";
	    $index += $main::severities{'wishlist'} * 1;
	}
	if ($main::severities{'normal'}) {
	    print "  Normal:           $main::severities{'normal'}\n";
	    $index += $main::severities{'normal'} * 2;
	}
	if ($main::severities{'important'}) {
	    print "  Important:        $main::severities{'important'}\n";
	    $index += $main::severities{'important'} * 3;
	}
	if ($main::severities{'critical'}) {
	    print "  Critical:         $main::severities{'critical'}\n";
	    $index += $main::severities{'critical'} * 4;
	}
	if ($main::severities{'grave'}) {
	    print "  Grave:            $main::severities{'grave'}\n";
	    $index += $main::severities{'grave'} * 5;
	}
	if ($main::severities{'release_crit'}) {
	    print "  Release Critical: $main::severities{'release_crit'}\n";
	}
	$index = $index / $open unless $open == 0;
	print "\nAverage BugIndex(tm) for this set: $index\n";
    }
}

sub ldap_connect() {
    my $ldap = Net::LDAP->new($server, port => $port) or do_badexit($@);
    $ldap->bind;
    return $ldap;
}

# First arg left is the command, if not, use summary
$command=shift;
if ($command eq "") { $command="summary"; }

if ($command eq "summary") {
    $ldap = ldap_connect();

    if ($do_package ne "") {
	$do_save="./";
	output_package($ldap, $do_package);
	$ldap->unbind;
	exit 0;
    }	

    if ($do_bugid ne "") {
	output_bugid($ldap, "summary");
	$ldap->unbind;
	exit 0;
    }

    if (! -d $do_save) {
	print "Creating $do_save...\n";
	`mkdir -p $do_save > /dev/null 2>&1`;
    }

    if (! -d $do_directory) {
	do_badexit ("$do_directory does not exist");
    } elsif ($do_control ne "" && ! -f $do_control) {
	do_badexit ("$do_control does not exist");
    }

    $do_control = `find $do_directory -name debian -type d` unless ($do_control ne "");
    foreach $debian (split('\n', $do_control)) {
	chomp($debian);
	chomp($test=`basename $debian`);
	if ($test ne "control") {
	    $control="$debian/control";
	} else {
	    $control=$debian;
	}
	if ( ! -f $control ) {
	    next;
	}
	chomp ($source=`grep ^Source: $control`);
	$source =~ s/Source: //;
	print "Checking packages in ${source}...\n";
	foreach $package (`grep ^Package: $control`) {
	    chop($package);
	    $package =~ s/Package: //;
	    output_package($ldap, $package);
	}
	print "\n";
    }
    $ldap->unbind;
    do_totals();
    exit 0;
} elsif ($command eq "report") {
    if ($do_bugid ne "") {
	$ldap = ldap_connect();
	output_bugid($ldap, "report");
	$ldap->unbind;
	exit 0;
    } else {
	not_impl ($command);
    }
} elsif ($command eq "log") {
    if ($do_bugid ne "") {
	$ldap = ldap_connect();
	output_bugid($ldap, "log");
	$ldap->unbind;
	exit 0;
    } else {
	not_impl ($command);
    }
} elsif ($command eq "update") {
    not_impl ($command);
    exit 0;
} elsif ($command eq "close") {
    if ($do_bugid ne "" && $do_package ne "") {
	do_badexit("must supply only one of --package or --bugid with 'close'");
    }
    if ($do_bugid ne "") {
	$do_bugid =~ s/#//g;
	@buglist = split(',', $do_bugid);
	open( M, "|/usr/lib/sendmail") || do_badexit("can't open sendmail");
	print M "X-Mailer: getbugs $version\n";
	$firstbug = shift(@buglist);
	print M "To: ${firstbug}-done\@bugs.debian.org";
	while(@buglist) {
	    $this_bug = shift(@buglist);
	    if ($this_bug =~ m/^\d*$/) {
		print M ", ${this_bug}-done\@bugs.debian.org";
	    } else {
		print "W: $this_bug is not a bug id\n";
	    }
	}
	print "Subject? ";
	$subject = <STDIN>;
	print M "\nSubject: $subject\n\n";
	print "\n";
	print "Body of the email (end with a '.' on a line by itself:\n";
	while (<STDIN>) {
	    if (m/^\.$/) { last; } else {
		print M $_;
	    }
	}
	close M;
	print "Mail sent, bugs will be closed.\n";
    } elsif ($do_package ne "") {
	$ldap = ldap_connect();
	$ldap->unbind;
    } else {
	do_badexit("must supply one of --package or --bugid with 'close'");
    }
    exit 0;
} else {
    do_badexit("Unknown command '$command'");
}

# How did we get here?
exit 1;
