/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_logger.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Test_Logger Class 
 *
 * $Id: test_logger.hpp,v 1.12 2005/06/23 09:54:27 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/test_logger.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_LOGGER_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_LOGGER_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/namespace.hpp>

// used as base class
#include <diagnostics/frame/logger.hpp>

// used in the interface by value
#include <diagnostics/frame/level.hpp>


// used in the interface by reference 
#include <vector>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

// used in ther interface by reference
// diagnostics/unittest/test_system/test_run_result.hpp
class Test_Run_Result;

/**
 * @class Test_Logger diagnostics/unittest/test_system/test_logger.hpp
 *
 * This class is used to fill a vector of Test_Run_Results. For each
 * @ref Test_Case which is executed while this logger is registered
 * (via @ref Logging_Config::register_logger), a @ref Test_Run_Result
 * is added to the vector of Test_Run_Results. All records which
 * appear between a @ref TYPE_TESTCASE_ENTER and a @ref
 * TYPE_TESTCASE_EXIT and which are either of @ref
 * TYPE_FAILED_ASSERTION or at @ref LEVEL_TEST are fed into the
 * corresponding @ref Test_Run_Result. Also, we feed those records of
 * type @ref TYPE_FAILED_CHECK into the @ref Test_Run_Result, which
 * come with a level larger than the target_level.
 *
 * In other words, once a @ref TYPE_TESTCASE_ENTER occurs, the logger
 * becomes active (@ref STATE_ACTIVE) until a @ref TYPE_TESTCASE_EXIT
 * occurs to bring the logger back into a passive state (@ref STATE_PASSIVE).
 *
 * Each such @ref Test_Run_Result is initialized with build-level of
 * the test-code (which must be supplied to the logger in its
 * constructor).
 *
 * @nosubgrouping
 */
class Test_Logger :
    public Logger
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
public:
	typedef Test_Logger Self;

    typedef ::std::vector<Test_Run_Result*> Test_Run_Results_t;

    /**
     * @brief Abstract State of this class. 
     */
    typedef enum { 
	/**
	 * @brief the logger is currently recording, i.e., storing the
	 * incoming records into a corresponding @ref Test_Run_Result.
	 */
	STATE_ACTIVE, 
	/**
	 * @brief the logger does not record at the moment, i.e., it
	 * drops all incoming records.
	 */
	STATE_PASSIVE
    } Abstract_State_t;
    
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief Assembly-Cstr: @a build_level, the @a target_level and
     * the vector of Test_Run_Results to be filled by the logger.
     *
     * The @a build_level is compilation/link-level of the test-code
     * itself -- this level is stored in each @ref Test_Run_Result
     * since a Test_Case can behave differently under different
     * build_levels, in fact, in higher build_levels we employ checks
     * which can not afforded in production code. Therefore, the code
     * behaves differently, and consequently, it has to be tested at
     * all levels.
     *
     * The @a target_level is lowest build-level for which a Test_Case
     * should work correctly. If a Test_Case is targeted for @ref
     * LEVEL_PROD, it should work properly in @ref LEVEL_DEBUG and in
     * @ref LEVEL_AUDIT -- however, in none of these three levels, a
     * check or assertion of @ref LEVEL_AUDIT or @ref LEVEL_DEBUG
     * should be triggered.
     *
     * The converse is not true: A Test targeted for @ref LEVEL_DEBUG
     * might validate the additional checks of the LEVEL_DEBUG, and
     * therefore would not succeed if built and executed at level
     * LEVEL_PROD.
     * 
     * @pre @a build_level != @ref LEVEL_TEST
     * @pre @a build_level != @ref LEVEL_SYSTEM
     * @pre @a target_level != @ref LEVEL_TEST
     * @pre @a target_level != @ref LEVEL_SYSTEM
     * @pre @a build_level >= @a target_level
     *
     * @throw Test_System_Exception if the precondtion is not met
     */
    Test_Logger(Level_t const build_level,
		Level_t const target_level,
		Test_Run_Results_t & test_run_results);

    /**
     * @brief do nothing
     *
     * @throw never
     */
    virtual ~Test_Logger();

private:
	/**
	 * @brief not available 
	 */
	Self & operator=(Self const & other);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    /**
     * @brief the abstract state of the instance
     *
     * @throw never
     */
    Abstract_State_t abstract_state() const;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Modifiers
     * @{
     */
public:
    /**
     * @brief dispatching the records to the @ref Test_Run_Result
     *
     * This method uses @ref DIAGNOSTICS_PANIC_LOG to report problems
     * -- thus it never throws. For each complete execution of a @ref
     * Test_Case (defined to start with a @ref TYPE_TESTCASE_ENTER and
     * to finish with a @ref TYPE_TESTCASE_EXIT) it opens a @ref
     * Test_Run_Result and feeds all records belonging to the
     * execution of this Test_Case into the corresponding
     * Test_Run_Result. 
     *
     * @attention In the following postcondition we present all valid
     * transitions. If none of the above implicants is met, a message
     * is written to the @ref DIAGNOSTICS_PANIC_LOG -- and the record
     * is ignored.
     *
     * @note if @a record.level()== @ref LEVEL_SYSTEM then the
     * internal @ref Test_Run_Result will ignore the record. However,
     * we do not check, since @ref LEVEL_SYSTEM messages occur at a
     * low rate.
     *
     * @post (PRE.abstract_state() == @ref STATE_PASSIVE 
     *       &&  @a record.type() == @ref TYPE_TESTCASE_ENTER) 
     *       --> POST.abstract_state()== @ref STATE_ACTIVE 
     *           [a new Test_Run_Result is opened and the current @a record is added to it]
     *
     * @post (PRE.abstract_state() == @ref STATE_ACTIVE 
     *       &&  @a record.type() == @ref TYPE_TESTCASE_EXIT) 
     *       --> POST.abstract_state()== @ref STATE_PASSIVE 
     *           [the current @a record is added to the current @ref Test_Run_Result, 
     *           then the @ref Test_Run_Result is closed]
     *
     * @post (PRE.abstract_state() == @ref STATE_PASSIVE 
     *       &&  @a record.type() != TYPE_TESTCASE_*) 
     *       --> POST.abstract_state()== @ref STATE_PASSIVE 
     *
     * @post (PRE.abstract_state() == @ref STATE_ACTIVE 
     *       &&  @a record.type() != TYPE_TESTCASE_*) 
     *       --> POST.abstract_state()== @ref STATE_ACTIVE 
     *           [@a record is added to the current Test_Run_Result, iff
     *           record.level() == @ref LEVEL_TEST 
     *           || record.type() == @ref TYPE_FAILED_ASSERTION 
	 *           || record.type() == @ref TYPE_UNEXPECTED_EXCEPTION
     *           || (record.type() == @ref TYPE_FAILED_CHECK 
     *               && record.level()> PRE.m_target_level]
     *
     *
     * @throw never
     */
    virtual void log(Record const & record);

    /**
     * @brief closing -- do nothing
     *
     * @note In the current usage, the Test_Logger is not receiving
     * this message, since it is unregistered after testing. However,
     * if it ever happens, it will do exactly nothing.
     *
     * @throw never
     */
    virtual void close();
    // @}


    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State
     * @{
     */
private:
    Level_t const m_build_level;
    Level_t const m_target_level;
    Test_Run_Results_t & m_test_run_results;
    // @}
};


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
