#! /bin/bash

# debrelease: a devscripts wrapper around dupload/dput which calls
#             dupload/dput with the correct .changes file as parameter.
#             All command line options are passed onto dupload.
#
# Written and copyright 1999 by Julian Gilbey <jdg@debian.org> 
# Based on the original 'release' script by
#  Christoph Lameter <clameter@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

set -e

PROGNAME=`basename $0`

usage () {
    echo \
"Usage: $PROGNAME [debrelease options] [dupload/dput options]
  Run dupload on the newly created changes file.
  Debrelease options:
    --dupload       Use dupload to upload files (default)
    --dput          Use dput to upload files
    -a<arch>        Search for .changes file made for Debian build <arch>
    -t<target>      Search for .changes file made for GNU <target> arch
    -S              Search for source-only .changes file instead of arch one
    --help          Show this message
    --version       Show version and copyright information"
}

version () {
    echo \
"This is $PROGNAME, from the Debian devscripts package, version ###VERSION###
This code is copyright 1999 by Julian Gilbey, all rights reserved.
Based on original code by Christoph Lameter.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of the
GNU General Public License, version 2 or later."
}

mustsetvar () {
    if [ "x$2" = x ]
    then
	echo >&2 "$0: unable to determine $3"
	exit 1
    else
	# echo "$0: $3 is $2"
	eval "$1=\"\$2\""
    fi
}

# Boilerplate: set config variables
DEBRELEASE_UPLOADER=dupload

for file in /etc/devscripts.conf ~/.devscripts
do
    [ -r $file ] && . $file
done

case "$DEBRELEASE_UPLOADER" in
    dupload|dput) ;;
    *) DEBRELEASE_UPLOADER=dupload ;;
esac

sourceonly=

while [ $# != 0 ]
do
    case "$1" in
	-a*) targetarch="`echo \"$1\" | sed -e 's/^-a//'`" ;;
	-t*) targetgnusystem="`echo \"$1\" | sed -e 's/^-t//'`" ;;
	-S) sourceonly=source ;;
	--dupload) DEBRELEASE_UPLOADER=dupload ;;
	--dput) DEBRELEASE_UPLOADER=dput ;;
	--help)	usage; exit 0 ;;
	--version) version; exit 0 ;;
	*) break ;;
    esac
    shift
done

# Look for .changes file via debian/changelog
until [ -f debian/changelog ]; do
    cd ..
    if [ `pwd` = "/" ]; then
	echo "Cannot find debian/changelog anywhere." >&2
	exit 1
    fi
done

mustsetvar package "`dpkg-parsechangelog | sed -n 's/^Source: //p'`" \
    "source package"
mustsetvar version "`dpkg-parsechangelog | sed -n 's/^Version: //p'`" \
    "source version"

if ! command -v dpkg-architecture > /dev/null 2>&1; then
    echo "This program depends on dpkg-architecture; your dpkg is far too old" >& 1
    exit 1
fi

if [ "x$sourceonly" = "xsource" ]; then
    arch=source
else
    mustsetvar arch "`dpkg-architecture -a${targetarch} -t${targetgnusystem} -qDEB_HOST_ARCH`" "build architecture"
fi

sversion=`echo "$version" | perl -pe 's/^\d+://'`
pva="${package}_${sversion}_${arch}"
pvs="${package}_${sversion}_source"
changes="../$pva.changes"
schanges="../$pvs.changes"

if [ ! -r "$changes" ]; then
    if [ "$arch" != source ]; then
	if [ -r "$schanges" ]; then
	    changes=$schanges
	else
	    echo "Could not read changes file $changes!" 2>&1
	    exit 1
	fi
    else
	echo "Could not read changes file $changes!" 2>&1
	exit 1
    fi
fi

exec $DEBRELEASE_UPLOADER "$@" "$changes"

echo "Failed to exec $DEBRELEASE_UPLOADER!" >&2
echo "Aborting...." >&2
exit 1
