/*
 * CvsGraph graphical representation generator of brances and revisions
 * of a file in cvs/rcs.
 *
 * Copyright (C) 2001  B. Stultiens
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <regex.h>
#include <errno.h>
#include <getopt.h>
#include <ctype.h>
#include <time.h>

#include <gd.h>
#include <gdfontt.h>

#include "config.h"
#include "cvsgraph.h"
#include "utils.h"
#include "readconf.h"
#include "rcs.h"

#if !defined(HAVE_IMAGE_GIF) && !defined(HAVE_IMAGE_PNG) && !defined(HAVE_IMAGE_JPEG)
# error No image output format available. Check libgd
#endif


/*#define DEBUG		1*/

#define CONFFILENAME	"cvsgraph.conf"

#ifndef ETCDIR
# define ETCDIR		"/usr/local/etc"
#endif

#ifndef MAX
# define MAX(a,b)	((a) > (b) ? (a) : (b))
#endif

#ifndef MIN
# define MIN(a,b)	((a) < (b) ? (a) : (b))
#endif

#define ALIGN_HL	0x00
#define ALIGN_HC	0x01
#define ALIGN_HR	0x02
#define ALIGN_HX	0x0f
#define ALIGN_VT	0x00
#define ALIGN_VC	0x10
#define ALIGN_VB	0x20
#define ALIGN_VX	0xf0

/*
 **************************************************************************
 * Globals
 **************************************************************************
 */

config_t conf;
int debuglevel;
color_t white_color = {255, 255, 255, 0};
color_t black_color = {0, 0, 0, 0};


/*
 **************************************************************************
 * Dubug routines
 **************************************************************************
 */
void dump_rev(char *p, rev_t *r)
{
	printf("%s", p);
	if(r)
		printf("'%s', '%s', %d\n", r->rev, r->branch, r->isbranch);
	else
		printf("<null>\n");
}

void dump_id(char *p, char *d)
{
	printf("%s", p);
	if(d)
		printf("'%s'\n", d);
	else
		printf("<null>\n");
}

void dump_idrev(char *p, idrev_t *t)
{
	printf("%s", p);
	if(t)
	{
		printf("'%s' -> ", t->id);
		dump_rev("", t->rev);
	}
	else
		printf("<null>\n");
}

void dump_tag(char *p, tag_t *t)
{
	printf("%s", p);
	if(t)
	{
		printf("'%s' -> ", t->tag);
		dump_rev("", t->rev);
	}
	else
		printf("<null>\n");
}

void dump_delta(char *p, delta_t *d)
{
	int i;
	printf("%sdelta.rev   : ", p);
	dump_rev("", d->rev);
	printf("%sdelta.date  : %s\n", p, d->date);
	printf("%sdelta.author: %s\n", p, d->author);
	printf("%sdelta.state : %s\n", p, d->state);
	for(i = 0; d->branches && i < d->branches->nrevs; i++)
	{
		printf("%sdelta.branch: ", p);
		dump_rev("", d->branches->revs[i]);
	}
	printf("%sdelta.next  : ", p);
	dump_rev("", d->next);
	printf("\n");
}

void dump_dtext(char *p, dtext_t *d)
{
	printf("%sdtext.rev  : ", p);
	dump_rev("", d->rev);
	printf("%sdtext.log  : %d bytes\n", p, d->log ? strlen(d->log) : -1);
	printf("%sdtext.text : %d bytes\n", p, d->text ? strlen(d->text) : -1);
	printf("\n");
}

void dump_rcsfile(rcsfile_t *rcs)
{
	int i;
	printf("root   : '%s'\n", rcs->root);
	printf("module : '%s'\n", rcs->module);
	printf("file   : '%s'\n", rcs->file);
	dump_rev("head   : ", rcs->head);
	dump_rev("branch : ", rcs->branch);
	printf("access :\n");
	for(i = 0; rcs->access && i < rcs->access->nids; i++)
		dump_id("\t", rcs->access->ids[i]);
	printf("tags   :\n");
	for(i = 0; rcs->tags && i < rcs->tags->ntags; i++)
		dump_tag("\t", rcs->tags->tags[i]);
	printf("locks  :%s\n", rcs->strict ? " (strict)" : "");
	for(i = 0; rcs->locks && i < rcs->locks->nidrevs; i++)
		dump_idrev("\t", rcs->locks->idrevs[i]);
	printf("comment: '%s'\n", rcs->comment);
	printf("expand : '%s'\n", rcs->expand ? rcs->expand : "(default -kv)");
	printf("deltas :\n");
	for(i = 0; rcs->deltas && i < rcs->deltas->ndeltas; i++)
		dump_delta("\t", rcs->deltas->deltas[i]);
	printf("desc   : '%s'\n", rcs->desc);
	printf("dtexts :\n");
	for(i = 0; rcs->dtexts && i < rcs->dtexts->ndtexts; i++)
		dump_dtext("\t", rcs->dtexts->dtexts[i]);

	fflush(stdout);
}

/*
 **************************************************************************
 * Read the rcs file
 **************************************************************************
 */
rcsfile_t *get_rcsfile(const char *cvsroot, const char *module, const char *file)
{
	char *cmd = NULL;
	int rv;

	cmd = xmalloc(strlen(cvsroot) + strlen(module) + strlen(file) + 1);
	sprintf(cmd, "%s%s%s", cvsroot, module, file);
	if(!(rcsin = fopen(cmd, "r")))
	{
		perror(cmd);
		return NULL;
	}
	input_file = cmd;
	line_number = 1;
	rv = rcsparse();
	fclose(rcsin);
	if(rv)
		return NULL;
	xfree(cmd);
	input_file = NULL;
	rcsfile->root = xstrdup(cvsroot);
	rcsfile->module = xstrdup(module);
	rcsfile->file = xstrdup(file);
	return rcsfile;
}

/*
 **************************************************************************
 * Sort and find helpers
 **************************************************************************
 */
int count_dots(const char *s)
{
	int i;
	for(i = 0; *s; s++)
	{
		if(*s == '.')
			i++;
	}
	return i;
}

int compare_rev(int bcmp, const rev_t *r1, const rev_t *r2)
{
	int d1, d2;
	char *c1, *c2;
	char *v1, *v2;
	char *s1, *s2;
	int retval = 0;
	assert(r1 != NULL);
	assert(r2 != NULL);
	if(bcmp)
	{
		assert(r1->branch != NULL);
		assert(r2->branch != NULL);
		c1 = r1->branch;
		c2 = r2->branch;
	}
	else
	{
		assert(r1->rev != NULL);
		assert(r2->rev != NULL);
		c1 = r1->rev;
		c2 = r2->rev;
	}

	d1 = count_dots(c1);
	d2 = count_dots(c2);
	if(d1 != d2)
	{
		return d1 - d2;
	}

	s1 = v1 = xstrdup(c1);
	s2 = v2 = xstrdup(c2);
	while(1)
	{
		char *vc1 = strchr(s1, '.');
		char *vc2 = strchr(s2, '.');
		if(vc1 && vc2)
			*vc1 = *vc2 = '\0';
		if(*s1 && *s2)
		{
			d1 = atoi(s1);
			d2 = atoi(s2);
			if(d1 != d2)
			{
				retval = d1 - d2;
				break;
			}
		}
		if(!vc1 || !vc2)
			break;
		s1 = vc1 + 1;
		s2 = vc2 + 1;
	}
	xfree(v1);
	xfree(v2);
	return retval;
}

/*
 **************************************************************************
 * Reorganise the rcsfile for the branches
 *
 * Basically, we have a list of deltas (i.e. administrative info on
 * revisions) and a list of delta text (the actual logs and diffs).
 * The deltas are linked through the 'next' and the 'branches' fields
 * which describe the tree-structure of revisions.
 * The reorganisation means that we put each delta and corresponding
 * delta text in a revision structure and assign it to a specific
 * branch. This is required because we want to be able to calculate
 * the bounding boxes of each branch. The revisions expand vertically
 * and the branches expand horizontally.
 * The reorganisation is performed in these steps:
 * 1 - sort deltas and detla text on revision number for quick lookup
 * 2 - start at the denoted head revision:
 *	* create a branch structure and add this revision
 *	* for each 'branches' in the delta do:
 *		- walk all 'branches' of the delta and recursively goto 2
 *		  with the denoted branch delta as new head
 *		- backlink the newly create sub-branch to the head revision
 *		  so that we can draw them recursively
 *	* set head to the 'next' field and goto 2 until no next is
 *	  available
 * 3 - update the administration
 **************************************************************************
 */
int sort_delta(const void *d1, const void *d2)
{
	return compare_rev(0, (*(delta_t **)d1)->rev, (*(delta_t **)d2)->rev);
}

int search_delta(const void *r, const void *d)
{
	return compare_rev(0, (rev_t *)r, (*(delta_t **)d)->rev);
}

delta_t *find_delta(delta_t **dl, int n, rev_t *r)
{
	delta_t **d;
	d = bsearch(r, dl, n, sizeof(*dl), search_delta);
	if(!d)
		return NULL;
	return *d;
}

int sort_dtext(const void *d1, const void *d2)
{
	return compare_rev(0, (*(dtext_t **)d1)->rev, (*(dtext_t **)d2)->rev);
}

int search_dtext(const void *r, const void *d)
{
	return compare_rev(0, (rev_t *)r, (*(dtext_t **)d)->rev);
}

dtext_t *find_dtext(dtext_t **dl, int n, rev_t *r)
{
	dtext_t **d;
	d = bsearch(r, dl, n, sizeof(*dl), search_dtext);
	if(!d)
		return NULL;
	return *d;
}

rev_t *dup_rev(const rev_t *r)
{
	rev_t *t = xmalloc(sizeof(*t));
	t->rev = xstrdup(r->rev);
	t->branch = xstrdup(r->branch);
	t->isbranch = r->isbranch;
	return t;
}

branch_t *new_branch(delta_t *d, dtext_t *t)
{
	branch_t *b = xmalloc(sizeof(*b));
	revision_t *r = xmalloc(sizeof(*r));
	r->delta = d;
	r->dtext = t;
	r->rev = d->rev;
	r->branch = b;
	b->branch = dup_rev(d->rev);
	b->branch->isbranch = 1;
	b->nrevs = 1;
	b->revs = xmalloc(sizeof(b->revs[0]));
	b->revs[0] = r;
	return b;
}

revision_t *add_to_branch(branch_t *b, delta_t *d, dtext_t *t)
{
	revision_t *r = xmalloc(sizeof(*r));
	r->delta = d;
	r->dtext = t;
	r->rev = d->rev;
	r->branch = b;
	b->revs = xrealloc(b->revs, (b->nrevs+1) * sizeof(b->revs[0]));
	b->revs[b->nrevs] = r;
	b->nrevs++;
	return r;
}

void build_branch(branch_t ***bl, int *nbl, delta_t **sdl, int nsdl, dtext_t **sdt, int nsdt, delta_t *head)
{
	branch_t *b;
	dtext_t *text;
	revision_t *currev;

	assert(head != NULL);

	if(head->flag)
	{
		fprintf(stderr, "Circular reference on '%s' in branchpoint\n", head->rev->rev);
		return;
	}
	head->flag++;
	text = find_dtext(sdt, nsdt, head->rev);

	/* Create a new branch for this head */
	b = new_branch(head, text);
	*bl = xrealloc(*bl, (*nbl+1)*sizeof((*bl)[0]));
	(*bl)[*nbl] = b;
	(*nbl)++;
	currev = b->revs[0];
	while(1)
	{
		/* Process all sub-branches */
		if(head->branches)
		{
			int i;
			for(i = 0; i < head->branches->nrevs; i++)
			{
				delta_t *d = find_delta(sdl, nsdl, head->branches->revs[i]);
				int btag = *nbl;
				if(!d)
					continue;
				build_branch(bl, nbl, sdl, nsdl, sdt, nsdt, d);

				/* Set the new branch's origin */
				(*bl)[btag]->branchpoint = currev;

				/* Add branch to this revision */
				currev->branches = xrealloc(currev->branches, (currev->nbranches+1)*sizeof(currev->branches[0]));
				currev->branches[currev->nbranches] = (*bl)[btag];
				currev->nbranches++;
			}
		}

		/* Walk through the next list */
		if(!head->next)
			return;

		head = find_delta(sdl, nsdl, head->next);
		if(!head)
		{
			fprintf(stderr, "Next revision (%s) not found in deltalist\n", head->next->rev);
			return;
		}
		if(head->flag)
		{
			fprintf(stderr, "Circular reference on '%s'\n", head->rev->rev);
			return;
		}
		head->flag++;
		text = find_dtext(sdt, nsdt, head->rev);
		currev = add_to_branch(b, head, text);
	}
}

int reorganise_branches(rcsfile_t *rcs)
{
	delta_t **sdelta;
	int nsdelta;
	dtext_t **sdtext;
	int nsdtext;
	delta_t *head;
	branch_t **bl;
	int nbl;
	int i;

	assert(rcs->deltas != NULL);
	assert(rcs->head != NULL);

	/* Make a new list for quick lookup */
	nsdelta = rcs->deltas->ndeltas;
	sdelta = xmalloc(nsdelta * sizeof(sdelta[0]));
	memcpy(sdelta, rcs->deltas->deltas, nsdelta * sizeof(sdelta[0]));
	qsort(sdelta, nsdelta, sizeof(sdelta[0]), sort_delta);

	/* Do the same for the delta text */
	nsdtext = rcs->dtexts->ndtexts;
	sdtext = xmalloc(nsdtext * sizeof(sdtext[0]));
	memcpy(sdtext, rcs->dtexts->dtexts, nsdtext * sizeof(sdtext[0]));
	qsort(sdtext, nsdtext, sizeof(sdtext[0]), sort_dtext);

	/* Start from the head of the trunk */
	head = find_delta(sdelta, nsdelta, rcs->head);
	if(!head)
	{
		fprintf(stderr, "Head revision (%s) not found in deltalist\n", rcs->head->rev);
		return 0;
	}
	bl = NULL;
	nbl = 0;
	build_branch(&bl, &nbl, sdelta, nsdelta, sdtext, nsdtext, head);

	/* Reverse the head branch */
	for(i = 0; i < bl[0]->nrevs/2; i++)
	{
		revision_t *r;
		r = bl[0]->revs[i];
		bl[0]->revs[i] = bl[0]->revs[bl[0]->nrevs-i-1];
		bl[0]->revs[bl[0]->nrevs-i-1] = r;
	}

	/* Update the branch-number of the head because it was reversed */
	xfree(bl[0]->branch->branch);
	bl[0]->branch->branch = xstrdup(bl[0]->revs[0]->rev->branch);

	/* Keep the admin */
	rcs->branches = bl;
	rcs->nbranches = nbl;
	rcs->sdelta = sdelta;
	rcs->nsdelta = nsdelta;
	rcs->sdtext = sdtext;
	rcs->nsdtext = nsdtext;
	rcs->active = bl[0];
	return 1;
}

/*
 **************************************************************************
 * Assign the symbolic tags to the revisions and branches
 *
 * The tags point to revision numbers somewhere in the tree structure
 * of branches and revisions. First we make a sorted list of all
 * revisions and then we assign each tag to the proper revision.
 **************************************************************************
 */
int sort_revision(const void *r1, const void *r2)
{
	return compare_rev(0, (*(revision_t **)r1)->delta->rev, (*(revision_t **)r2)->delta->rev);
}

int search_revision(const void *t, const void *r)
{
	return compare_rev(0, (rev_t *)t, (*(revision_t **)r)->delta->rev);
}

int sort_branch(const void *b1, const void *b2)
{
	return compare_rev(1, (*(branch_t **)b1)->branch, (*(branch_t **)b2)->branch);
}

int search_branch(const void *t, const void *b)
{
	return compare_rev(1, (rev_t *)t, (*(branch_t **)b)->branch);
}

char *previous_rev(const char *c)
{
	int dots = count_dots(c);
	char *cptr;
	char *r;
	if(!dots)
	{
		fprintf(stderr, "FIXME: previous_rev(\"%s\"): Cannot determine parent branch revision\n", c);
		return xstrdup("1.0");	/* FIXME: don't know what the parent is */
	}
	if(dots & 1)
	{
		/* Is is a revision we want the parent of */
		r = xstrdup(c);
		cptr = strrchr(r, '.');
		assert(cptr != NULL);
		if(dots == 1)
		{
			fprintf(stderr, "FIXME: previous_rev(\"%s\"): Going beyond top-level?\n", c);
			/* FIXME: What is the parent of 1.1? */
			cptr[1] = '\0';
			strcat(r, "0");
			return r;
		}
		/* Here we have a "x.x[.x.x]+" case */
		*cptr = '\0';
		cptr = strrchr(r, '.');
		assert(cptr != NULL);
		*cptr = '\0';
		return r;
	}
	/* It is a branch we want the parent of */
	r = xstrdup(c);
	cptr = strrchr(r, '.');
	assert(cptr != NULL);
	*cptr = '\0';
	return r;
}

int assign_tags(rcsfile_t *rcs)
{
	int i;
	int nr;

	for(i = nr = 0; i < rcs->nbranches; i++)
		nr += rcs->branches[i]->nrevs;

	rcs->srev = xmalloc(nr * sizeof(rcs->srev[0]));
	rcs->nsrev = nr;
	for(i = nr = 0; i < rcs->nbranches; i++)
	{
		memcpy(&rcs->srev[nr], rcs->branches[i]->revs, rcs->branches[i]->nrevs * sizeof(rcs->branches[i]->revs[0]));
		nr += rcs->branches[i]->nrevs;
	}

	qsort(rcs->srev, rcs->nsrev, sizeof(rcs->srev[0]), sort_revision);
	qsort(rcs->branches, rcs->nbranches, sizeof(rcs->branches[0]), sort_branch);

	if(!rcs->branch)
	{
		/* The main trunk is the active trunk */
		rcs->tags->tags = xrealloc(rcs->tags->tags, (rcs->tags->ntags+1)*sizeof(rcs->tags->tags[0]));
		rcs->tags->tags[rcs->tags->ntags] = xmalloc(sizeof(tag_t));
		rcs->tags->tags[rcs->tags->ntags]->tag = xstrdup("MAIN");
		rcs->tags->tags[rcs->tags->ntags]->rev = xmalloc(sizeof(rev_t));
		rcs->tags->tags[rcs->tags->ntags]->rev->rev = xstrdup(rcs->active->branch->rev);
		rcs->tags->tags[rcs->tags->ntags]->rev->branch = xstrdup(rcs->active->branch->branch);
		rcs->tags->tags[rcs->tags->ntags]->rev->isbranch = 1;
		rcs->tags->ntags++;
	}

	/* We should have at least two tags (HEAD and MAIN) */
	assert(rcs->tags != 0);

	for(i = 0; i < rcs->tags->ntags; i++)
	{
		tag_t *t = rcs->tags->tags[i];
		if(t->rev->isbranch)
		{
			branch_t **b;
add_btag:
			b = bsearch(t->rev, rcs->branches, rcs->nbranches, sizeof(rcs->branches[0]), search_branch);
			if(!b)
			{
				rev_t rev;
				revision_t **r;
				/* This happens for the magic branch numbers if there are
				 * no commits within the new branch yet. So, we add the
				 * branch and try to continue.
				 */
				rev.rev = previous_rev(t->rev->branch);
				rev.branch = NULL;
				rev.isbranch = 0;
				r = bsearch(&rev, rcs->srev, rcs->nsrev, sizeof(rcs->srev[0]), search_revision);
				xfree(rev.rev);
				if(!r)
				{
					if(!quiet)
						fprintf(stderr, "No branch found for tag '%s:%s'\n", t->tag, t->rev->branch);
				}
				else
				{
					rcs->branches = xrealloc(rcs->branches, (rcs->nbranches+1)*sizeof(rcs->branches[0]));
					rcs->branches[rcs->nbranches] = xmalloc(sizeof(branch_t));
					rcs->branches[rcs->nbranches]->branch = dup_rev(t->rev);
					rcs->branches[rcs->nbranches]->branchpoint = *r;
					(*r)->branches = xrealloc((*r)->branches, ((*r)->nbranches+1)*sizeof((*r)->branches[0]));
					(*r)->branches[(*r)->nbranches] = rcs->branches[rcs->nbranches];
					(*r)->nbranches++;
					rcs->nbranches++;
					/* Resort the branches */
					qsort(rcs->branches, rcs->nbranches, sizeof(rcs->branches[0]), sort_branch);
					goto add_btag;
				}
			}
			else
			{
				branch_t *bb = *b;
				bb->tags = xrealloc(bb->tags, (bb->ntags+1)*sizeof(bb->tags[0]));
				bb->tags[bb->ntags] = t;
				bb->ntags++;
			}
		}
		else
		{
			revision_t **r = bsearch(t->rev, rcs->srev, rcs->nsrev, sizeof(rcs->srev[0]), search_revision);
			if(!r)
			{
				if(!quiet)
					fprintf(stderr, "No revision found for tag '%s:%s'\n", t->tag, t->rev->rev);
			}
			else
			{
				revision_t *rr = *r;
				rr->tags = xrealloc(rr->tags, (rr->ntags+1)*sizeof(rr->tags[0]));
				rr->tags[rr->ntags] = t;
				rr->ntags++;
			}
		}
	}

	/* We need to reset the first in the list of branches to the
	 * active branch to ensure the drawing of all
	 */
	if(rcs->active != rcs->branches[0])
	{
		branch_t **b = bsearch(rcs->active->branch, rcs->branches, rcs->nbranches, sizeof(rcs->branches[0]), search_branch);
		branch_t *t;
		assert(b != NULL);
		t = *b;
		*b = rcs->branches[0];
		rcs->branches[0] = t;
	}
	return 1;
}

/*
 **************************************************************************
 * String expansion
 **************************************************************************
 */
static char *_string;
static int _nstring;
static int _nastring;

void add_string_str(const char *s)
{
	int l = strlen(s) + 1;
	if(_nstring + l > _nastring)
	{
		_nastring += 128;
		_string = xrealloc(_string, _nastring * sizeof(_string[0]));
	}
	memcpy(_string+_nstring, s, l);
	_nstring += l-1;
}

void add_string_ch(int ch)
{
	char buf[2];
	buf[0] = ch;
	buf[1] = '\0';
	add_string_str(buf);
}

void add_string_date(const char *d)
{
	struct tm tm, *tmp;
	int n;
	time_t t;
	char *buf;
	int nbuf;

	memset(&tm, 0, sizeof(tm));
	n = sscanf(d, "%d.%d.%d.%d.%d.%d",
			&tm.tm_year, &tm.tm_mon, &tm.tm_mday,
			&tm.tm_hour, &tm.tm_min, &tm.tm_sec);
	tm.tm_mon--;
	if(tm.tm_year > 1900)
		tm.tm_year -= 1900;
	t = mktime(&tm);
	if(n != 6 || t == (time_t)(-1))
	{
		add_string_str("<invalid date>");
		return;
	}

	tmp = localtime(&t);
	nbuf = strlen(conf.date_format) * 16;	/* Should be enough to hold all types of expansions */
	buf = xmalloc(nbuf);
	strftime(buf, nbuf, conf.date_format, tmp);
	add_string_str(buf);
	xfree(buf);
}

char *expand_string(const char *s, rcsfile_t *rcs, revision_t *r, rev_t *rev, rev_t *prev, tag_t *tag)
{
	char nb[32];
	char nr[32];
	char *base;

	if(!s)
		return xstrdup("");

	_nstring = 0;
	if(_string)
		_string[0] = '\0';

	sprintf(nb, "%d", rcs->nbranches);
	sprintf(nr, "%d", rcs->nsrev);
	for(; *s; s++)
	{
		if(*s == '%')
		{
			switch(*++s)
			{
			case 'c':
			case 'C':
				add_string_str(conf.cvsroot);
				if(*s == 'C' && conf.cvsroot[0] && conf.cvsroot[strlen(conf.cvsroot)-1] == '/')
				{
					/* Strip the trailing '/' */
					_nstring--;
					_string[_nstring] = '\0';
				}
				break;
			case 'f':
			case 'F':
				base = strrchr(rcs->file, '/');
				if(!base)
					add_string_str(rcs->file);
				else
					add_string_str(base+1);
				if(*s == 'F' && _string[_nstring-1] == 'v' && _string[_nstring-2] == ',')
				{
					_nstring -= 2;
					_string[_nstring] = '\0';
				}
				break;
			case 'p':
				base = strrchr(rcs->file, '/');
				if(base)
				{
					char *c = xstrdup(rcs->file);
					base = strrchr(c, '/');
					assert(base != NULL);
					base[1] = '\0';
					add_string_str(c);
					xfree(c);
				}
				/*
				 * We should not add anything here because we can encounter
				 * a completely empty path, in which case we do not want
				 * to add any slash. This prevents a inadvertent root redirect.
				 *
				 * else
				 *	add_string_str("/");
				 */
				break;
			case 'm':
			case 'M':
				add_string_str(conf.cvsmodule);
				if(*s == 'M' && conf.cvsmodule[0] && conf.cvsmodule[strlen(conf.cvsmodule)-1] == '/')
				{
					/* Strip the trailing '/' */
					_nstring--;
					_string[_nstring] = '\0';
				}
				break;
			case 'r': add_string_str(nr); break;
			case 'b': add_string_str(nb); break;
			case '%': add_string_ch('%'); break;
			case '0': if(conf.expand[0]) add_string_str(conf.expand[0]); break;
			case '1': if(conf.expand[1]) add_string_str(conf.expand[1]); break;
			case '2': if(conf.expand[2]) add_string_str(conf.expand[2]); break;
			case '3': if(conf.expand[3]) add_string_str(conf.expand[3]); break;
			case '4': if(conf.expand[4]) add_string_str(conf.expand[4]); break;
			case '5': if(conf.expand[5]) add_string_str(conf.expand[5]); break;
			case '6': if(conf.expand[6]) add_string_str(conf.expand[6]); break;
			case '7': if(conf.expand[7]) add_string_str(conf.expand[7]); break;
			case '8': if(conf.expand[8]) add_string_str(conf.expand[8]); break;
			case '9': if(conf.expand[9]) add_string_str(conf.expand[9]); break;
			case 'R': if(rev && rev->rev) add_string_str(rev->rev); break;
			case 'P': if(prev && prev->rev) add_string_str(prev->rev); break;
			case 'B': if(rev && rev->branch) add_string_str(rev->branch); break;
			case 't': if(tag && tag->tag) add_string_str(tag->tag); break;
			case 'd': if(r && r->delta && r->delta->date) add_string_date(r->delta->date); break;
			case 's': if(r && r->delta && r->delta->state) add_string_str(r->delta->state); break;
			case 'a': if(r && r->delta && r->delta->author) add_string_str(r->delta->author); break;
			default:
				add_string_ch('%');
				add_string_ch(*s);
				break;
			}
		}
		else
			add_string_ch(*s);
	}
	return xstrdup(_string);
}

/*
 **************************************************************************
 * Drawing routines
 **************************************************************************
 */
int get_swidth(const char *s, font_t *f)
{
	int n;
	int m;
	if(!s || !*s)
		return 0;
	for(n = m = 0; *s; n++, s++)
	{
		if(*s == '\n')
		{
			if(n > m)
				m = n;
			n = 0;
		}
	}
	if(n > m)
		m = n;
	return m * (*f)->w;
}

int get_sheight(const char *s, font_t *f)
{
	int nl;
	if(!s || !*s)
		return 0;
	for(nl = 1; *s; s++)
	{
		if(*s == '\n' && s[1])
			nl++;
	}
	return nl * (*f)->h;
}

void draw_rbox(gdImagePtr im, int x1, int y1, int x2, int y2, int r, color_t *color, color_t *bgcolor)
{
	int r2 = 2*r;
	gdImageLine(im, x1+r, y1, x2-r, y1, color->id);
	gdImageLine(im, x1+r, y2, x2-r, y2, color->id);
	gdImageLine(im, x1, y1+r, x1, y2-r, color->id);
	gdImageLine(im, x2, y1+r, x2, y2-r, color->id);
	if(conf.box_shadow)
	{
		gdImageLine(im, x1+r+1, y2+1, x2-r, y2+1, black_color.id);
		gdImageLine(im, x2+1, y1+r+1, x2+1, y2-r, black_color.id);
	}
	if(r)
	{
		gdImageArc(im, x1+r, y1+r, r2, r2, 180, 270, color->id);
		gdImageArc(im, x2-r, y1+r, r2, r2, 270, 360, color->id);
		gdImageArc(im, x1+r, y2-r, r2, r2,  90, 180, color->id);
		gdImageArc(im, x2-r, y2-r, r2, r2,   0,  90, color->id);
		if(conf.box_shadow)
		{
			/* FIXME: Pixelization is not correct here */
			gdImageArc(im, x2-r+1, y2-r+1, r2, r2,   0,  90, black_color.id);
		}
	}
	gdImageFillToBorder(im, (x1+x2)/2, (y1+y2)/2, color->id, bgcolor->id);
}

void draw_string(gdImagePtr im, char *s, font_t *f, int x, int y, int align, color_t *c)
{
	int xx, yy;
	switch(align & ALIGN_HX)
	{
	default:
	case ALIGN_HL: xx = 0; break;
	case ALIGN_HC: xx = -get_swidth(s, f)/2; break;
	case ALIGN_HR: xx = -get_swidth(s, f); break;
	}
	switch(align & ALIGN_VX)
	{
	default:
	case ALIGN_VT: yy = 0; break;
	case ALIGN_VC: yy = -get_sheight(s, f)/2; break;
	case ALIGN_VB: yy = -get_sheight(s, f); break;
	}
	gdImageString(im, *f, x+xx+1, y+yy, s, c->id);
}

void draw_stringnl(gdImagePtr im, char *s, font_t *f, int x, int y, int align, color_t *c)
{
	char *t;
	char *d;
	d = s = xstrdup(s);
	do
	{
		t = strchr(s, '\n');
		if(t)
			*t = '\0';
		draw_string(im, s, f, x, y, align, c);
		y += get_sheight(s, f);
		s = t+1;
	} while(t);
	xfree(d);
}

void draw_rev(gdImagePtr im, int cx, int ty, revision_t *r)
{
	int lx = cx - r->w/2;
	int rx = lx + r->w;
	int i;
	draw_rbox(im, lx, ty, rx, ty+r->h, 0, &conf.rev_color, &conf.rev_bgcolor);
	ty += conf.rev_tspace;
	draw_string(im, r->rev->rev, &conf.rev_font, cx, ty, ALIGN_HC, &conf.rev_color);
	ty += get_sheight(r->rev->rev, &conf.rev_font);
	draw_stringnl(im, r->revtext, &conf.rev_text_font, cx, ty, ALIGN_HC, &conf.rev_text_color);
	ty += get_sheight(r->revtext, &conf.rev_text_font);
	for(i = 0; i < r->ntags; i++)
	{
		draw_string(im, r->tags[i]->tag, &conf.tag_font, cx, ty, ALIGN_HC, &conf.tag_color);
		ty += get_sheight(r->tags[i]->tag, &conf.tag_font);
	}
}

void draw_branch(gdImagePtr im, int cx, int ty, branch_t *b)
{
	int lx = cx - b->w/2;
	int rx = lx + b->w;
	int yy;
	int i;
	/*draw_rbox(im, cx-b->tw/2-1, ty-1, cx+b->tw/2+1, ty+b->th+1, 0, &conf.title_color);*/
	draw_rbox(im, lx, ty, rx, ty+b->h, 5, &conf.branch_color, &conf.branch_bgcolor);
	yy = conf.branch_tspace;
	draw_string(im, b->branch->branch, &conf.branch_font, cx, ty+yy, ALIGN_HC, &conf.branch_color);
	yy += get_sheight(b->branch->branch, &conf.branch_font);
	for(i = 0; i < b->ntags; i++)
	{
		draw_string(im, b->tags[i]->tag, &conf.branch_font, cx, ty+yy, ALIGN_HC, &conf.branch_color);
		yy += get_sheight(b->tags[i]->tag, &conf.branch_font);
	}

	ty += b->h;
	for(i = 0; i < b->nrevs; i++)
	{
		gdImageLine(im, cx, ty, cx, ty+conf.rev_minline, conf.rev_color.id);
		ty += conf.rev_minline;
		draw_rev(im, cx, ty, b->revs[i]);
		ty += b->revs[i]->h;
	}
}

void draw_connector(gdImagePtr im, branch_t *b)
{
	revision_t *r = b->branchpoint;
	int x1 = r->cx + r->w/2 + 2;
	int y1 = r->y + r->h/2;
	int x2 = b->cx;
	int y2 = b->y;
	gdImageLine(im, x1, y1, x2, y1, conf.branch_color.id);
	gdImageLine(im, x2, y1, x2, y2, conf.branch_color.id);
}

void alloc_color(gdImagePtr im, color_t *c)
{
	c->id = gdImageColorAllocate(im, c->r, c->g, c->b);
}

gdImagePtr make_image(rcsfile_t *rcs)
{
	gdImagePtr im;
	int i;
	char *cptr;

	im = gdImageCreate(rcs->tw+conf.margin_left+conf.margin_right, rcs->th+conf.margin_top+conf.margin_bottom);
	alloc_color(im, &conf.color_bg);
	alloc_color(im, &conf.tag_color);
	alloc_color(im, &conf.rev_color);
	alloc_color(im, &conf.rev_bgcolor);
	alloc_color(im, &conf.rev_text_color);
	alloc_color(im, &conf.branch_color);
	alloc_color(im, &conf.branch_bgcolor);
	alloc_color(im, &conf.title_color);
	alloc_color(im, &black_color);
	alloc_color(im, &white_color);

	for(i = 0; i < rcs->nbranches; i++)
		draw_branch(im, rcs->branches[i]->cx, rcs->branches[i]->y, rcs->branches[i]);
	for(i = 0; i < rcs->nbranches; i++)
	{
		if(rcs->branches[i]->branchpoint)
			draw_connector(im, rcs->branches[i]);
	}
	cptr = expand_string(conf.title, rcs, NULL, NULL, NULL, NULL);
	draw_stringnl(im, cptr, &conf.title_font, conf.title_x, conf.title_y, conf.title_align, &conf.title_color);
	xfree(cptr);

	return im;
}

/*
 **************************************************************************
 * Layout routines
 **************************************************************************
 */
void move_branch(branch_t *b, int x, int y)
{
	int i;
	b->cx += x;
	b->y += y;
	for(i = 0; i < b->nrevs; i++)
	{
		b->revs[i]->cx += x;
		b->revs[i]->y += y;
	}
}

void reposition_branch(revision_t *r, int *x, int *w)
{
	int i, j;
	for(j = 0; j < r->nbranches; j++)
	{
		branch_t *b = r->branches[j];
		*x += *w + conf.rev_minline + b->tw/2 - b->cx;
		*w = b->tw/2;
		move_branch(b, *x, r->y + r->h/2 + conf.branch_connect);
		*x = b->cx;
		/* Recurse to move branches of branched revisions */
		for(i = b->nrevs-1; i >= 0; i--)
		{
			reposition_branch(b->revs[i], x, w);
		}
	}
}

void rect_union(int *x, int *y, int *w, int *h, branch_t *b)
{
	int x1 = *x;
	int x2 = x1 + *w;
	int y1 = *y;
	int y2 = y1 + *h;
	int xx1 = b->cx - b->tw/2;
	int xx2 = xx1 + b->tw;
	int yy1 = b->y;
	int yy2 = yy1 + b->th;
	x1 = MIN(x1, xx1);
	x2 = MAX(x2, xx2);
	y1 = MIN(y1, yy1);
	y2 = MAX(y2, yy2);
	*x = x1;
	*y = y1;
	*w = x2 - x1;
	*h = y2 - y1;
}

int branch_intersects(int top, int bottom, int left, branch_t *b)
{
	int br = b->cx + b->tw/2;
	int bt = b->y - conf.branch_connect - conf.branch_margin/2;
	int bb = b->y + b->th + conf.branch_margin/2;
	return !(bt > bottom || bb < top || br >= left);
}

int kern_branch(rcsfile_t *rcs, branch_t *b)
{
	int left = b->cx - b->tw/2;
	int top = b->y - conf.branch_connect - conf.branch_margin/2;
	int bottom = b->y + b->th + conf.branch_margin/2;
	int i;
	int xpos = 0;

	for(i = 0; i < rcs->nbranches; i++)
	{
		branch_t *bp = rcs->branches[i];
		if(bp == b)
			continue;
		if(branch_intersects(top, bottom, left, bp))
		{
			int m = bp->cx + bp->tw/2 + conf.branch_margin;
			if(m > xpos)
				xpos = m;
		}
	}
	if(xpos && (b->cx - b->tw/2) - xpos > 0)
	{
		move_branch(b, xpos - (b->cx - b->tw/2), 0);
		return 1;
	}
	return 0;
}

void make_layout(rcsfile_t *rcs)
{
	int i, j;
	int x, y;
	int w, h;
	int w2;
	int moved;

	/* Calculate the box-sizes of the revisions */
	for(i = 0; i < rcs->nsrev; i++)
	{
		revision_t *rp;
		int w;
		int h;
		rp = rcs->srev[i];
		rp->revtext = expand_string(conf.rev_text, rcs, rp, rp->rev, NULL, rp->ntags ? rp->tags[0] : NULL);
		w = get_swidth(rp->revtext, &conf.rev_text_font);
		j = get_swidth(rp->rev->rev, &conf.rev_font);
		if(j > w)
			w = j;
		h = get_sheight(rp->revtext, &conf.rev_text_font) + get_sheight(rp->rev->rev, &conf.rev_font);
		for(j = 0; j < rp->ntags; j++)
		{
			int ww = get_swidth(rp->tags[j]->tag, &conf.tag_font);
			if(ww > w) w = ww;
			h += get_sheight(rp->tags[j]->tag, &conf.tag_font) + conf.rev_separator;
		}
		rp->w = w + conf.rev_lspace + conf.rev_rspace;
		rp->h = h + conf.rev_tspace + conf.rev_bspace;
	}

	/* Calculate the box-sizes of the branches */
	for(i = 0; i < rcs->nbranches; i++)
	{
		branch_t *bp = rcs->branches[i];
		int w;
		int h;
		w = get_swidth(bp->branch->branch, &conf.branch_font);
		h = get_sheight(bp->branch->branch, &conf.branch_font);
		for(j = 0; j < bp->ntags; j++)
		{
			int ww = get_swidth(bp->tags[j]->tag, &conf.branch_font);
			if(ww > w) w = ww;
			h += get_sheight(bp->tags[j]->tag, &conf.branch_font);
		}
		w += conf.branch_lspace + conf.branch_rspace;
		h += conf.branch_tspace + conf.branch_bspace;
		bp->w = w;
		bp->h = h;
		for(j = 0; j < bp->nrevs; j++)
		{
			if(bp->revs[j]->w > w)
				w = bp->revs[j]->w;
			h += bp->revs[j]->h + conf.rev_minline;
		}
		bp->th = h;
		bp->tw = w;
	}

	/* Calculate the relative positions of revs in a branch */
	for(i = 0; i < rcs->nbranches; i++)
	{
		branch_t *b = rcs->branches[i];
		x = b->tw/2;
		y = b->h;
		b->cx = x;
		b->y = 0;
		for(j = 0; j < b->nrevs; j++)
		{
			y += conf.rev_minline;
			b->revs[j]->cx = x;
			b->revs[j]->y = y;
			y += b->revs[j]->h;
		}
	}

	/* Reposition the branches */
	x = rcs->branches[0]->cx;
	w2 = rcs->branches[0]->tw / 2;
	for(i = rcs->branches[0]->nrevs-1; i >= 0; i--)
	{
		reposition_branch(rcs->branches[0]->revs[i], &x, &w2);
	}

	/* Try to move branches left if there is room (kerning) */
	for(moved = 1; moved; )
	{
		moved = 0;
		for(i = 1; i < rcs->nbranches; i++)
		{
			moved += kern_branch(rcs, rcs->branches[i]);
		}
	}

	/* Move everything w.r.t. the top-left margin */
	for(i = 0; i < rcs->nbranches; i++)
		move_branch(rcs->branches[i], conf.margin_left, conf.margin_top);

	/* Calculate overall image size */
	x = rcs->branches[0]->cx - rcs->branches[0]->tw/2;
	y = rcs->branches[0]->y;
	w = rcs->branches[0]->tw;
	h = rcs->branches[0]->th;
	for(i = 1; i < rcs->nbranches; i++)
		rect_union(&x, &y, &w, &h, rcs->branches[i]);
	rcs->tw = w;
	rcs->th = h;
}

/*
 **************************************************************************
 * Imagemap functions
 **************************************************************************
 */
void make_imagemap(rcsfile_t *rcs, FILE *fp)
{
	int i, j;
	char *href;
	char *alt;
	fprintf(fp, "<map name=\"%s\">\n", conf.map_name);
	for(i = 0; i < rcs->nbranches; i++)
	{
		branch_t *b = rcs->branches[i];
		tag_t *tag = b->ntags ? b->tags[0] : NULL;
		href = expand_string(conf.map_branch_href, rcs, NULL, b->branch, NULL, tag);
		alt = expand_string(conf.map_branch_alt, rcs, NULL, b->branch, NULL, tag);
		fprintf(fp, "\t<area shape=\"rect\" %s coords=\"%d,%d,%d,%d\" %s>\n",
				href,
				b->cx - b->w/2, b->y, b->cx + b->w/2, b->y + b->h,
				alt);
		xfree(href);
		xfree(alt);
		for(j = 0; j < b->nrevs; j++)
		{
			revision_t *r = b->revs[j];
			tag = r->ntags ? r->tags[0] : NULL;
			href = expand_string(conf.map_rev_href, rcs, r, r->rev, NULL, tag);
			alt = expand_string(conf.map_rev_alt, rcs, r, r->rev, NULL, tag);
			fprintf(fp, "\t<area shape=\"rect\" %s coords=\"%d,%d,%d,%d\" %s>\n",
				href,
				r->cx - r->w/2, r->y, r->cx + r->w/2, r->y + r->h,
				alt);
			xfree(href);
			xfree(alt);
			if(j > 0)
			{
				revision_t *r1 = b->revs[j-1];
				int x1 = r->cx - MIN(r->w, r1->w)/4;
				int x2 = r->cx + MIN(r->w, r1->w)/4;
				href = expand_string(conf.map_diff_href, rcs, r, r->rev, r1->rev, tag);
				alt = expand_string(conf.map_diff_alt, rcs, r, r->rev, r1->rev, tag);
				fprintf(fp, "\t<area shape=\"rect\" %s coords=\"%d,%d,%d,%d\" %s>\n",
					href,
					x1, r1->y + r1->h + 1, x2, r->y - 1,
					alt);
				xfree(href);
				xfree(alt);
			}
		}
	}
	fprintf(fp, "</map>\n");
}

/*
 **************************************************************************
 * Configuration
 **************************************************************************
 */
int read_config(const char *path)
{
	FILE *fp;
	int r;

	if(path)
	{
		if((fp = fopen(path, "r")) == NULL)
		{
			return 0;
		}
		else
			input_file = path;
	}
	else
	{
		if((fp = fopen("./" CONFFILENAME, "r")) == NULL)
		{
			if((fp = fopen(ETCDIR "/" CONFFILENAME, "r")) == NULL)
			{
				return 0;
			}
			else
				input_file = ETCDIR "/" CONFFILENAME;
		}
		else
			input_file = "./" CONFFILENAME;
	}

	yyin = fp;
	r = yyparse();
	fclose(fp);
	input_file = NULL;
	return r == 0;
}

/*
 **************************************************************************
 * Program entry
 **************************************************************************
 */
static const char usage_str[] =
	"Usage: cvsgraph [options] <file>\n"
	"  -c <file>    Read alternative config from <file>\n"
	"  -d <level>   Enable debug mode at <level>\n"
	"  -h           This message\n"
	"  -i           Generate an imagemap instead of image\n"
	"  -M <name>    Use <name> as imagemap name\n"
	"  -m <mod>     Use <mod> as cvs module\n"
	"  -o <file>    Output to <file>\n"
	"  -q           Be quiet (i.e. no warnings)\n"
	"  -r <path>    Use <path> as cvsroot path\n"
	"  -V           Print version and exit\n"
	"  -[0-9] <txt> Use <txt> for expansion\n"
	;

#define VERSION_STR	"1.1.2"
#define NOTICE_STR	"Copyright (c) 2001 B.Stultiens"

void append_slash(char **path)
{
	int l;
	assert(path != NULL);
	assert(*path != NULL);
	l = strlen(*path);
	if(!l || (*path)[l-1] == '/')
		return;
	*path = xrealloc(*path, l+2);
	strcat(*path, "/");
}

int main(int argc, char *argv[])
{
	extern int yy_flex_debug;
	extern int rcs_flex_debug;
	extern int yydebug;
	extern int rcsdebug;
	int optc;
	char *confpath = NULL;
	char *outfile = NULL;
	char *cvsroot = NULL;
	char *cvsmodule = NULL;
	int imagemap = 0;
	char *imgmapname = NULL;
	int lose = 0;
	FILE *fp;
	rcsfile_t *rcs;
	gdImagePtr im;

	while((optc = getopt(argc, argv, "0:1:2:3:4:5:6:7:8:9:c:d:hiM:m:o:qr:V")) != EOF)
	{
		switch(optc)
		{
		case 'c':
			confpath = xstrdup(optarg);
			break;
		case 'd':
			debuglevel = strtol(optarg, NULL, 0);
			break;
		case 'i':
			imagemap = 1;
			break;
		case 'M':
			imgmapname = xstrdup(optarg);
			break;
		case 'm':
			cvsmodule = xstrdup(optarg);
			break;
		case 'o':
			outfile = xstrdup(optarg);
			break;
		case 'q':
			quiet = 1;
			break;
		case 'r':
			cvsroot = xstrdup(optarg);
			break;
		case 'V':
			fprintf(stdout, "cvsgraph v%s, %s\n", VERSION_STR, NOTICE_STR);
			return 0;
		case 'h':
			fprintf(stdout, "%s", usage_str);
			return 0;
		default:
			if(isdigit(optc))
			{
				conf.expand[optc-'0'] = xstrdup(optarg);
			}
			else
				lose++;
		}
	}

	if(optind >= argc)
	{
		fprintf(stderr, "Missing inputfile\n");
		lose++;
	}

	if(lose)
	{
		fprintf(stderr, "%s", usage_str);
		return 1;
	}

	if(debuglevel)
	{
		setvbuf(stdout, NULL, 0, _IONBF);
		setvbuf(stderr, NULL, 0, _IONBF);
	}
	yy_flex_debug = (debuglevel & DEBUG_CONF_LEX) != 0;
	rcs_flex_debug = (debuglevel & DEBUG_RCS_LEX) != 0;
	yydebug = (debuglevel & DEBUG_CONF_YACC) != 0;
	rcsdebug = (debuglevel & DEBUG_RCS_YACC) != 0;

	/* Set defaults */
	conf.tag_font		= gdFontTiny;
	conf.rev_font		= gdFontTiny;
	conf.branch_font	= gdFontTiny;
	conf.title_font		= gdFontTiny;
	conf.rev_text_font	= gdFontTiny;

	conf.cvsroot		= xstrdup("");
	conf.cvsmodule		= xstrdup("");
	conf.date_format	= xstrdup("%d-%b-%Y %H:%M:%S");
	conf.title		= xstrdup("");
	conf.map_name		= xstrdup("CvsGraphImageMap");
	conf.map_branch_href	= xstrdup("href=\"unset: conf.map_branch_href\"");
	conf.map_branch_alt	= xstrdup("alt=\"%B\"");
	conf.map_rev_href	= xstrdup("href=\"unset: conf.map_rev_href\"");
	conf.map_rev_alt	= xstrdup("alt=\"%R\"");
	conf.map_diff_href	= xstrdup("href=\"unset: conf.map_diff_href\"");
	conf.map_diff_alt	= xstrdup("alt=\"%P &lt;-&gt; %R\"");
	conf.rev_text		= xstrdup("%d");

	conf.color_bg		= white_color;
	conf.branch_bgcolor	= white_color;
	conf.branch_color	= black_color;
	conf.rev_color		= black_color;
	conf.rev_bgcolor	= white_color;
	conf.tag_color		= black_color;
	conf.title_color	= black_color;
	conf.rev_text_color	= black_color;

	conf.image_quality	= 100;

	if(!read_config(confpath))
	{
		fprintf(stderr, "Error reading config file\n");
		return 1;
	}

	/* Set overrides */
	if(cvsroot)	conf.cvsroot = cvsroot;
	if(cvsmodule)	conf.cvsmodule = cvsmodule;
	if(imgmapname)	conf.map_name = imgmapname;

	append_slash(&conf.cvsroot);
	append_slash(&conf.cvsmodule);

	rcs = get_rcsfile(conf.cvsroot, conf.cvsmodule, argv[optind]);
	if(!rcs)
		return 1;

	if(debuglevel & DEBUG_RCS_FILE)
		dump_rcsfile(rcs);

	if(!reorganise_branches(rcs))
		return 1;

	if(!assign_tags(rcs))
		return 1;

	if(outfile)
	{
		if((fp = fopen(outfile, "w")) == NULL)
		{
			perror(outfile);
			return 1;
		}
	}
	else
		fp = stdout;

	make_layout(rcs);

	if(!imagemap)
	{
		/* Create an image */
		im = make_image(rcs);

		switch(conf.image_type)
		{
#ifdef HAVE_IMAGE_GIF
# ifndef HAVE_IMAGE_PNG
		default:
# endif
		case IMAGE_GIF:
			gdImageGif(im, fp);
			break;
#endif
#ifdef HAVE_IMAGE_PNG
		default:
		case IMAGE_PNG:
			gdImagePng(im, fp);
			break;
#endif
#ifdef HAVE_IMAGE_JPEG
# if !defined(HAVE_IMAGE_GIF) && !defined(HAVE_IMAGE_PNG)
		default:
# endif
		case IMAGE_JPEG:
			gdImageJpeg(im, fp, conf.image_quality);
			break;
#endif
		}

		gdImageDestroy(im);
	}
	else
	{
		/* Create an imagemap */
		make_imagemap(rcs, fp);
	}

	if(outfile)
		fclose(fp);

	return 0;
}

