/*
 * CvsGraph graphical representation generator of brances and revisions
 * of a file in cvs/rcs.
 *
 * Copyright (C) 2001  B. Stultiens
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gd.h>
#include <gdfontt.h>
#include <gdfonts.h>
#include <gdfontmb.h>
#include <gdfontl.h>
#include <gdfontg.h>

#include "cvsgraph.h"
#include "utils.h"
#include "readconf.h"


static void assign_num(int key, int val);
static void assign_str(int key, char *str);
static void assign_color(int key, char *str);

static const char err_semicolon[] = "Missing ';'";
static const char err_assign[] = "Missing '='";

%}

%union{
	char	*str;
	int	num;
}

%token tBRANCH_BGCOLOR tBRANCH_BSPACE tBRANCH_COLOR tBRANCH_FONT tBRANCH_LSPACE tBRANCH_RSPACE tBRANCH_TSPACE
%token tBRANCH_CONNECT tBRANCH_MARGIN
%token tCOLOR_BG tCVSMODULE tCVSROOT tDATE_FORMAT tBOX_SHADOW
%token tREV_COLOR tREV_BGCOLOR tREV_FONT tREV_SEPARATOR tREV_MINLINE tREV_MAXLINE tREV_LSPACE
%token tREV_RSPACE tREV_TSPACE tREV_BSPACE tREV_TEXT tREV_TEXT_COLOR tREV_TEXT_FONT
%token tTAG_COLOR tTAG_FONT
%token tTITLE tTITLE_X tTITLE_Y tTITLE_FONT tTITLE_ALIGN tTITLE_COLOR
%token tMARGIN_TOP tMARGIN_BOTTOM tMARGIN_LEFT tMARGIN_RIGHT
%token tIMAGE_TYPE tIMAGE_QUALITY
%token tMAP_NAME tMAP_BRANCH_HREF tMAP_BRANCH_ALT tMAP_REV_HREF tMAP_REV_ALT tMAP_DIFF_HREF tMAP_DIFF_ALT
%token <str> tUNKNOWN
%token <str> tSTRING
%token <num> tNUMBER

%type <num> ikey skey ckey

%%
conffile: /* Empty */
	| lines
	;

lines	: line
	| lines line
	;

line	: ikey '=' tNUMBER ';'		{ assign_num($1, $3); }
	| ikey '=' tNUMBER error	{ yyerror(err_semicolon); }
	| ikey '=' error		{ yyerror("Number expected"); }
	| ikey error			{ yyerror(err_assign); }
	| skey '=' tSTRING ';'		{ assign_str($1, $3); }
	| skey '=' tSTRING error	{ yyerror(err_semicolon); }
	| skey '=' error		{ yyerror("String expected"); }
	| skey error			{ yyerror(err_assign); }
	| ckey '=' tSTRING ';'		{ assign_color($1, $3);	}
	| ckey '=' tSTRING error	{ yyerror(err_semicolon); }
	| ckey '=' error		{ yyerror("Color-string expected"); }
	| ckey error			{ yyerror(err_assign); }
	| tUNKNOWN '=' notsc ';'	{ yywarning("Unknown config parameter '%s' ignored", $1); xfree($1); }
	| tUNKNOWN '=' notsc error	{ yyerror(err_semicolon); }
	| tUNKNOWN '=' error		{ yyerror("String or number expected"); }
	| tUNKNOWN error		{ yyerror(err_assign); }
	| error				{ yyerror("Keyword expected"); }
	;

notsc	: tSTRING	{ xfree($1); }
	| tNUMBER	{}
	;

ikey	: tBRANCH_FONT			{ $$ = tBRANCH_FONT; }
	| tBRANCH_LSPACE        	{ $$ = tBRANCH_LSPACE; }
	| tBRANCH_RSPACE        	{ $$ = tBRANCH_RSPACE; }
	| tBRANCH_TSPACE        	{ $$ = tBRANCH_TSPACE; }
	| tBRANCH_BSPACE        	{ $$ = tBRANCH_BSPACE; }
	| tBRANCH_CONNECT        	{ $$ = tBRANCH_CONNECT; }
	| tBRANCH_MARGIN        	{ $$ = tBRANCH_MARGIN; }
	| tREV_FONT             	{ $$ = tREV_FONT; }
	| tREV_SEPARATOR        	{ $$ = tREV_SEPARATOR; }
	| tREV_MINLINE          	{ $$ = tREV_MINLINE; }
	| tREV_MAXLINE          	{ $$ = tREV_MAXLINE; }
	| tREV_LSPACE           	{ $$ = tREV_LSPACE; }
	| tREV_RSPACE           	{ $$ = tREV_RSPACE; }
	| tREV_TSPACE           	{ $$ = tREV_TSPACE; }
	| tREV_BSPACE           	{ $$ = tREV_BSPACE; }
	| tREV_TEXT_FONT             	{ $$ = tREV_TEXT_FONT; }
	| tTAG_FONT             	{ $$ = tTAG_FONT; }
	| tTITLE_X              	{ $$ = tTITLE_X; }
	| tTITLE_Y              	{ $$ = tTITLE_Y; }
	| tTITLE_FONT           	{ $$ = tTITLE_FONT; }
	| tTITLE_ALIGN          	{ $$ = tTITLE_ALIGN; }
	| tMARGIN_TOP           	{ $$ = tMARGIN_TOP; }
	| tMARGIN_BOTTOM        	{ $$ = tMARGIN_BOTTOM; }
	| tMARGIN_LEFT          	{ $$ = tMARGIN_LEFT; }
	| tMARGIN_RIGHT         	{ $$ = tMARGIN_RIGHT; }
	| tIMAGE_TYPE	         	{ $$ = tIMAGE_TYPE; }
	| tIMAGE_QUALITY         	{ $$ = tIMAGE_QUALITY; }
	| tBOX_SHADOW	         	{ $$ = tBOX_SHADOW; }
	;                       
                                
skey	: tCVSMODULE            	{ $$ = tCVSMODULE; }
	| tCVSROOT              	{ $$ = tCVSROOT; }
	| tTITLE                	{ $$ = tTITLE; }
	| tDATE_FORMAT                	{ $$ = tDATE_FORMAT; }
	| tMAP_NAME			{ $$ = tMAP_NAME; }
	| tMAP_BRANCH_HREF		{ $$ = tMAP_BRANCH_HREF; }
	| tMAP_BRANCH_ALT		{ $$ = tMAP_BRANCH_ALT; }
	| tMAP_REV_HREF			{ $$ = tMAP_REV_HREF; }
	| tMAP_REV_ALT			{ $$ = tMAP_REV_ALT; }
	| tMAP_DIFF_HREF		{ $$ = tMAP_DIFF_HREF; }
	| tMAP_DIFF_ALT			{ $$ = tMAP_DIFF_ALT; }
	| tREV_TEXT                	{ $$ = tREV_TEXT; }
	;                       
                                
ckey	: tCOLOR_BG             	{ $$ = tCOLOR_BG; }
	| tBRANCH_BGCOLOR       	{ $$ = tBRANCH_BGCOLOR; }
	| tBRANCH_COLOR         	{ $$ = tBRANCH_COLOR; }
	| tREV_COLOR            	{ $$ = tREV_COLOR; }
	| tREV_BGCOLOR            	{ $$ = tREV_BGCOLOR; }
	| tREV_TEXT_COLOR            	{ $$ = tREV_TEXT_COLOR; }
	| tTAG_COLOR            	{ $$ = tTAG_COLOR; }
	| tTITLE_COLOR            	{ $$ = tTITLE_COLOR; }
	;

%%

static gdFontPtr get_font(int id)
{
	switch(id)
	{
	case 0:	return gdFontTiny;
	case 1:	return gdFontSmall;
	default:
	case 2:	return gdFontMediumBold;
	case 3:	return gdFontLarge;
	case 4:	return gdFontGiant;
	}
}

static int bounds(int val, int mini, int maxi)
{
	if(val < mini || val > maxi)
	{
		yywarning("Parameter must be between %d and %d", mini, maxi);
		val = val < mini ? mini : maxi;
	}
	return val;
}

static void assign_num(int key, int val)
{
	switch(key)
	{
	case tBRANCH_FONT:	conf.branch_font = get_font(val); break;
	case tBRANCH_LSPACE:	conf.branch_lspace = val; break;
	case tBRANCH_RSPACE:	conf.branch_rspace = val; break;
	case tBRANCH_TSPACE:	conf.branch_tspace = val; break;
	case tBRANCH_BSPACE:	conf.branch_bspace = val; break;
	case tBRANCH_CONNECT:	conf.branch_connect = val; break;
	case tBRANCH_MARGIN:	conf.branch_margin = val; break;
	case tREV_FONT:		conf.rev_font = get_font(val); break;
	case tREV_SEPARATOR:	conf.rev_separator = val; break;
	case tREV_MINLINE:	conf.rev_minline = val; break;
	case tREV_MAXLINE:	conf.rev_maxline = val; break;
	case tREV_LSPACE:	conf.rev_lspace = val; break;
	case tREV_RSPACE:	conf.rev_rspace = val; break;
	case tREV_TSPACE:	conf.rev_tspace = val; break;
	case tREV_BSPACE:	conf.rev_bspace = val; break;
	case tREV_TEXT_FONT:	conf.rev_text_font = get_font(val); break;
	case tTAG_FONT:		conf.tag_font = get_font(val); break;
	case tTITLE_X:		conf.title_x = val; break;
	case tTITLE_Y:		conf.title_y = val; break;
	case tTITLE_FONT:	conf.title_font = get_font(val); break;
	case tTITLE_ALIGN:	conf.title_align = val; break;
	case tMARGIN_TOP:	conf.margin_top = val; break;
	case tMARGIN_BOTTOM:	conf.margin_bottom = val; break;
	case tMARGIN_LEFT:	conf.margin_left = val; break;
	case tMARGIN_RIGHT:	conf.margin_right = val; break;
	case tIMAGE_TYPE:	conf.image_type = val; break;
	case tIMAGE_QUALITY:	conf.image_quality = bounds(val, 1, 100); break;
	case tBOX_SHADOW:	conf.box_shadow = val; break;
	default:
		yyerror("internal: Invalid num key %d -> %d", key, val);
	}
}

static void assign_str(int key, char *str)
{
	switch(key)
	{
	case tCVSMODULE:	conf.cvsmodule = str; break;
	case tCVSROOT:		conf.cvsroot = str; break;
	case tTITLE:		conf.title = str; break;
	case tDATE_FORMAT:	conf.date_format = str; break;
	case tMAP_NAME:		conf.map_name = str; break;
	case tMAP_BRANCH_HREF:	conf.map_branch_href = str; break;
	case tMAP_BRANCH_ALT:	conf.map_branch_alt = str; break;
	case tMAP_REV_HREF:	conf.map_rev_href = str; break;
	case tMAP_REV_ALT:	conf.map_rev_alt = str; break;
	case tMAP_DIFF_HREF:	conf.map_diff_href = str; break;
	case tMAP_DIFF_ALT:	conf.map_diff_alt = str; break;
	case tREV_TEXT:		conf.rev_text = str; break;
	default:
		yyerror("internal: Invalid string key %d -> '%s'", key, str);
	}
}

static void set_color(color_t *c, char *s)
{
	char *cptr;
	if(*s != '#' || strlen(s) != 7)
	{
colorerror:
		yyerror("Invalid color value");
	}
	c->b = strtol(s+5, &cptr, 16);
	if(*cptr)
		goto colorerror;
	s[5] = '\0';
	c->g = strtol(s+3, &cptr, 16);
	if(*cptr)
		goto colorerror;
	s[3] = '\0';
	c->r = strtol(s+1, &cptr, 16);
	if(*cptr)
		goto colorerror;
}

static void assign_color(int key, char *str)
{
	switch(key)
	{
	case tCOLOR_BG:		set_color(&conf.color_bg, str);		break;
	case tBRANCH_BGCOLOR:	set_color(&conf.branch_bgcolor, str);	break;
	case tBRANCH_COLOR:	set_color(&conf.branch_color, str);	break;
	case tREV_COLOR:	set_color(&conf.rev_color, str);	break;
	case tREV_BGCOLOR:	set_color(&conf.rev_bgcolor, str);	break;
	case tREV_TEXT_COLOR:	set_color(&conf.rev_text_color, str);	break;
	case tTAG_COLOR:	set_color(&conf.tag_color, str);	break;
	case tTITLE_COLOR:	set_color(&conf.title_color, str);	break;
	default:
		yyerror("internal: Invalid color key %d -> '%s'", key, str);
	}
	xfree(str);
}

