/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

/* Console driver handling, may have a look at CCE 0.11's console.c
   and Linux kernel source code: /usr/src/linux/drivers/char/console.c

   The below code doesn't handle all of the ANSI required ESC stuff
 */

#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>
#if defined(linux)
#include	<termio.h>
#include	<malloc.h>
#elif defined(__FreeBSD__)
#include	<termios.h>
#include	<stdlib.h>
#endif
#include	<sys/types.h>

#include	<getcap.h>
#include	<defs.h>
#include	<term.h>
#include	<font.h>
#include	<vt.h>
#include	<vc.h>

#define	CHAR_NUL	'\x00'
#define	CHAR_BEL	'\x07'
#define	CHAR_BS		'\x08'
#define	CHAR_HT		'\x09'
#define	CHAR_LF		'\x0A'
#define	CHAR_VT		'\x0B'
#define	CHAR_FF		'\x0C'
#define	CHAR_CR		'\x0D'
#define	CHAR_SO		'\x0E'
#define	CHAR_SI		'\x0F'
#define	CHAR_XON	'\x11'
#define	CHAR_XOFF	'\x12'
#define	CHAR_CAN	'\x18'
#define	CHAR_SUB	'\x1A'
#define	CHAR_ESC	'\x1B'
#define	CHAR_DEL	'\x7F'
#define	CHAR_CSI	'\x9B'
#define	CHAR_SS2	'\x8E'

#define	LEN_REPORT	9

static int scroll;
LangInfo lInfo;

/*  Call the routines in vc.c
    TextInsertChar/TextDeleteChar
    ScrollUp/ScrollDown
    TextSput/TextWput
    TextMoveUp/TextMoveDown
    TextClearAll/TextClearEol/TextClearEos/TextClearChars
    Beep
*/

/****************************************************************************
 *                         SaveAttr/RestoreAttr                             *
 ****************************************************************************/

static void SaveAttr(ConInfo *con)
{
    struct attrStack *tmp;

    tmp = (struct attrStack *)malloc(sizeof(struct attrStack));
    if (con->saveAttr) tmp->prev = con->saveAttr;
    else tmp->prev = NULL;
    con->saveAttr = tmp;
    con->saveAttr->x = con->x;
    con->saveAttr->y = con->y;
    con->saveAttr->attr = con->attr;
    con->saveAttr->fcol = con->fcol;
    con->saveAttr->bcol = con->bcol;
}

static void RestoreAttr(ConInfo *con)
{
    struct attrStack *tmp;

    if (con->saveAttr) 
    {
	con->x = con->saveAttr->x;
	con->y = con->saveAttr->y;
	con->attr = con->saveAttr->attr;
	con->fcol = con->saveAttr->fcol;
	con->bcol = con->saveAttr->bcol;
	tmp = con->saveAttr;
	con->saveAttr = tmp->prev;
	free(tmp);
    }
}

/**************************************************************************
 *                               EscSetAttr                               *
 **************************************************************************/

static void	EscSetAttr(ConInfo *con,int col)
{
    static u_char table[] = {0, 4, 2, 6, 1, 5, 3, 7};
    u_char	swp;
    
    switch(col) 
    {
    case 0:	/* off all attributes */
	con->bcol = 0;
	con->fcol = 7;
	con->attr = 0;
	break;
    case 1:  // highlight, fcolor
	con->attr |= ATTR_HIGH;
//	if (con->fcol) con->fcol |= 8;
        con->fcol |= 8;
	break;
    case 21:  // clear highlight
	con->attr &= ~ATTR_HIGH;
	con->fcol &= ~8;
	break;
    case 4:	// underline, bcolor
	con->attr |= ATTR_ULINE;
	con->bcol |= 8;
	break;
    case 24:   // clear underline
	con->attr &= ~ATTR_ULINE;
	con->bcol &= ~8;
	break;
    case 7:  // reverse
	if (!(con->attr & ATTR_REVERSE)) 
        {
	    con->attr |= ATTR_REVERSE;
	    swp = con->fcol & 7;
	    if (con->attr & ATTR_ULINE) swp |= 8;
	    con->fcol = con->bcol & 7;
	    if ((con->attr & ATTR_HIGH) && con->fcol) con->fcol |= 8;
	    con->bcol = swp;
	}
	break;
    case 27:  // clear reverse
	if (con->attr & ATTR_REVERSE) 
        {
	    con->attr &= ~ATTR_REVERSE;
	    swp = con->fcol & 7;
	    if (con->attr & ATTR_ULINE) swp |= 8;
	    con->fcol = con->bcol & 7;
	    if ((con->attr & ATTR_HIGH) && con->fcol) con->fcol |= 8;
	    con->bcol = swp;
	}
	break;
    default:
	if (col >= 30 && col <= 37)  // assign foreground color
        {
	    swp = table[col - 30];
	    if (con->attr & ATTR_REVERSE) 
            {
		if (con->attr & ATTR_ULINE) swp |= 8;
		con->bcol = swp;
	    } 
            else 
            {
		if (con->attr & ATTR_HIGH) swp |= 8;
		con->fcol = swp;
	    }
	} 
        else if (col >= 40 && col <= 47)  // assign background color
        {
	    swp = table[col - 40];
	    if (con->attr & ATTR_REVERSE) 
            {
		if (con->attr & ATTR_HIGH) swp |= 8;
		con->fcol = swp;
	    } 
            else 
            {
		if (con->attr & ATTR_ULINE) swp |= 8;
		con->bcol = swp;
	    }
	}
	break;
    }
}

static void	VtSetMode(ConInfo *con,u_char mode, bool sw)
{
    switch(mode) 
    {
    case 4:
	con->ins = sw;
	break;
    case 25:
	cursorInfo.sw = sw;
	break;
    }
}

static void EscReport(ConInfo *con,u_char mode, u_short arg)
{
    static char report[LEN_REPORT];
    
    switch(mode) 
    {
    case 'n':
	if (arg == 6) 
        {
	    int x = (con->x < con->xmax) ? con->x : con->xmax;
	    int y = (con->y < con->ymax) ? con->y : con->ymax;
	    sprintf(report, "\x1B[%d;%dR", y + 1, x + 1);
	} 
        else if (arg == 5)
	    strcpy(report, "\x1B[0n\0");
	break;
    case 'c':
	if (arg == 0) strcpy(report, "\x1B[?6c\0");
	break;
    }
    write(con->masterPty, report, strlen(report));
}

#define	MAX_NARG	8

/* ESC [ ..... */
static void EscBracket(ConInfo *con,u_char ch)
{
    u_char	n;
    static u_short varg[MAX_NARG], narg, question;

    if (ch >= '0' && ch <= '9') 
    {
	varg[narg] = (varg[narg] * 10) + (ch - '0');
    }
    else if (ch == ';') 
    {
	if (narg < MAX_NARG) narg ++;
	else con->esc = NULL;
    } 
    else 
    {
	con->esc = NULL;
        
	switch(ch) 
        {
   	case 'K':
   	    TextClearEol(con,varg[0]);
   	    break;
   	case 'J':
   	    TextClearEos(con,varg[0]);
   	    break;
        // ESC[%dX - I can't find it in /etc/termcap, but dialog-based
        // program used it. - holly
        case 'X':
            TextClearChars(con, varg[0]);
            break;
   	case 'A':
   	    con->y -= varg[0] ? varg[0]: 1;
   	    if (con->y < con->ymin) 
               {
   		scroll -= con->y - con->ymin;
   		con->y = con->ymin;
   	    }
   	    break;
   	case 'B':
   	    con->y += varg[0] ? varg[0]: 1;
   	    if (con->y > con->ymax) 
               {
   		scroll += con->y - con->ymin;
   		con->y = con->ymax;
   	    }
   	    break;
   	case 'C':
   	    con->x += varg[0] ? varg[0]: 1;
   	    con->wrap = FALSE;
   	    break;
   	case 'D':
   	    con->x -= varg[0] ? varg[0]: 1;
   	    con->wrap = FALSE;
   	    break;
   	case 'G':
   	    con->x = varg[0] ? varg[0] - 1: 0;
   	    con->wrap = FALSE;
   	    break;
   	case 'P':
   	    TextDeleteChar(con,varg[0] ? varg[0]: 1);
   	    break;
   	case '@':
   	    TextInsertChar(con,varg[0] ? varg[0]: 1);
   	    break;
   	case 'L':
   	    TextMoveDown(con,con->y, con->ymax, varg[0] ? varg[0] : 1);
   	    break;
   	case 'M':
   	    TextMoveUp(con,con->y, con->ymax, varg[0] ? varg[0] : 1);
   	    break;
   	case 'H':
   	case 'f':
   	    if (varg[1]) con->x = varg[1] - 1;
   	    else con->x = 0;
   	    con->wrap = FALSE;
   	case 'd':
   	    con->y = varg[0] ? varg[0] - 1: 0;
   	    break;
   	case 'm':
   	    for (n = 0; n <= narg; n ++)
   		EscSetAttr(con,varg[n]);
   	    break;
   	case 'r':
   	    con->ymin = varg[0]? (varg[0] - 1): 0;
   	    con->ymax = varg[1]? (varg[1] - 1): (dispInfo.tymax - 1); // Fixed by Holly Lee - Sep 26, 1999
   	    con->x = 0;
   	    con->y = con->ymin;
   	    con->wrap = FALSE;
   	    if (con->ymin || con->ymax != dispInfo.tymax)
   		con->soft = TRUE;
   	    else
   		con->soft = FALSE;
   	    break;
   	case 'l':
   	    for (n = 0; n <= narg; n ++)
   		VtSetMode(con,varg[n], FALSE);
   	    break;
   	case 'h':
   	    for (n = 0; n <= narg; n ++)
   		VtSetMode(con,varg[n], TRUE);
   	    break;
   	case '?':
   	    question = TRUE;
   	    con->esc = EscBracket;
   	    break;
   	case 's':
   	    SaveAttr(con);
   	    break;
   	case 'u':
   	    RestoreAttr(con);
   	    break;
   	case 'n':
   	case 'c':
   	    if (question != TRUE)
   		EscReport(con,ch, varg[0]);
   	    break;
   	case 'R':
   	    break;
	}
        
	if (con->esc == NULL) {
	    question = narg = 0;
            for ( n = 0; n < MAX_NARG; n++ )
                varg[n] = 0;
        }
    }
}

static void EscSetDCodeG0(ConInfo *con,u_char ch)
{
    int i;

    switch(ch) 
    {
    case '(': /* EscSetDCodeG0 */
    case ')': /* EscSetDCodeG1 */
	return;
    case '@':
	ch = 'B';
    default:
	i = 0;
	while (fDRegs[i].sign0) 
        {
	    if (fDRegs[i].sign0 == ch) 
            {
		con->db = (u_char)i|LATCH_1;
		break;
	    }
	    i ++;
	}
	con->trans = CS_DBCS;
	break;
    }
    con->esc = NULL;
}

static void EscSetSCodeG0(ConInfo *con,u_char ch)
{
    int i=0;

    switch(ch) {
    case 'U':
	con->g[0] = CS_GRAPH;
	break;
    default:
	while (fSRegs[i].sign0) 
        {
	    if (fSRegs[i].sign0 == ch) 
            {
		con->sb = (u_char)i;
		con->g[0] = CS_LEFT;
		break;
	    } 
            else if (fSRegs[i].sign1 == ch) 
            {
		con->sb = (u_char)i;
		con->g[0] = CS_RIGHT;
		break;
	    }
	    i ++;
	}
    }
    con->trans = con->g[0];
    con->esc = NULL;
}

static void EscSetSCodeG1(ConInfo *con,u_char ch)
{
    switch(ch) 
    {
    case 'U':
	con->g[1] = CS_LEFT;
	break;
    case '0':
	con->g[1] = CS_GRAPH;
	break;
    case 'A':
    case 'J':
    case 'B':
	break;
    }
    con->trans = con->g[1];
    con->esc = NULL;
}

/* Escape code handling */
static void EscStart(ConInfo *con,u_char ch)
{
    con->esc = NULL;
    switch(ch) 
   {
    case '[':
	con->esc = EscBracket;
	break;
    case '$':/* Set Double Byte Code */
	con->esc = EscSetDCodeG0;
	break;
    case '(':/* Set 94 to G0 */
    case ',':/* Set 96 to G0 */
	con->esc = EscSetSCodeG0;
	break;
    case ')':/* Set G1 */
	con->esc = EscSetSCodeG1;
	break;
    case 'E':
	con->x = 0;
	con->wrap = FALSE;
    case 'D':
	if (con->y == con->ymax) scroll ++;
	else con->y ++;
	break;
    case 'M':
	if (con->y == con->ymin) scroll --;
	else con->y --;
	break;	
    case 'c':  /* reset terminal */
	con->fcol = 7;
	con->attr = 0;
	con->knj1 = con->bcol = 0;
	con->wrap = FALSE;
	con->trans = CS_LEFT;
	con->sb = lInfo.sb;
	con->db = lInfo.db|LATCH_1;
    case '*':
	con->x = con->y = 0;
	con->wrap = FALSE;
	TextClearAll(con);
	break;
    case '7':
	SaveAttr(con);
	break;
    case '8':
	RestoreAttr(con);
	con->wrap = FALSE;
	break;
    }
}

/***********************************************************************
 *                        DBCS determining routines                    *
 ***********************************************************************/

/*
int isPrevW2(ConInfo *con)
{
  u_int addr;
  if (con->x <= 0) return 0;
  addr = TextAddress(con,con->x-1, con->y);
  return(*(con->flagBuff + addr) & LATCH_2);
}

int isNextW2(ConInfo *con)
{
  u_int addr;
  if (con->x >= con->xmax) return 0;
  addr = TextAddress(con,con->x+1, con->y);
  return(*(con->flagBuff + addr) & LATCH_2);
}

int isPrevW1(ConInfo *con)
{
  u_int addr;
  addr = TextAddress(con,con->x-1, con->y);
  return(*(con->flagBuff + addr) & LATCH_1);
}
*/

static inline bool iskanji_byte1(u_char c)
{
    switch(lInfo.sysCoding) 
    {
    case CODE_SJIS:
        return (c >=0x81 && c<=0x9F) || (c >=0xE0 && c <=0xFC);
    case CODE_GB:
        return (c >= 0xA1 && c <= 0xFE);
    case CODE_BIG5:
       return (c >=0xa1 && c<=0xfe);  //first byte
    default:
        return (c & 0x80);
    }
}

static inline bool iskanji_byte2(u_char c)
{
    switch(lInfo.sysCoding) 
   {
    case CODE_SJIS:
        return (c >=0x81 && c<=0x9F) || (c >=0xE0 && c <=0xFC);
    case CODE_GB:
        return (c >= 0xA1 && c <= 0xFE);
    case CODE_BIG5:
       return ((c >=0x40 && c<=0x7e) || (c >=0xa1 && c<=0xfe)); // second byte
    default:
        return (c & 0x80);
    }
}

static inline bool iskanji(u_char c)
{
    switch(lInfo.sysCoding) 
   {
    case CODE_SJIS:
	return (c >=0x81 && c<=0x9F) || (c >=0xE0 && c <=0xFC);
    case CODE_GB:
        return (c >= 0xA1 && c <= 0xFE);
    case CODE_BIG5:
       return (c >=0xa1 && c<=0xfe);  //first byte
       return ((c >=0x40 && c<=0x7e) || (c >=0xa1 && c<=0xfe));
             // second byte
    default:
	return (c & 0x80);
    }
}

/****************************************************************************
 *               very important interface, output the buff                  *
 ****************************************************************************/

void	VtWrite(ConInfo *con,const char *buff, int nchars)
{
    u_char	ch;

  /* This is all the output strings */
#if 0
    {
	FILE *fff;
        int i;

	fff = fopen("/tmp/ccegb-esc.log", "a");
     for(i = 0; i < nchars; i++)
        fprintf(fff,"%c",buff[i]);
        fprintf(fff," con.x=%d, con.y=%d, conmax=%d\n",con->x, con->y, con->xmax);
	fclose(fff);
    }
#endif

   while (nchars-- > 0) 
   {
	ch = *buff++;
	if (! ch)
	    continue;
	if (con->esc) 
	    con->esc(con,ch);   // in escape mode, parse the parameters
	else switch (ch) 
        {
	case CHAR_BEL:  // 0x07
	    Beep();
	    break;
	case CHAR_DEL:  // 0x7F
	    break;
	case CHAR_BS:   // 0x08
	    if (con->x) con->x--;
	    con->wrap = FALSE;
	    break;
	case CHAR_HT:  //0x09
	    con->x += con->tab - (con->x % con->tab);
	    con->wrap = FALSE;
	    if (con->x > con->xmax) con->x -= con->xmax + 1;
	    else break;
	case CHAR_VT:  // 0x0B
	case CHAR_FF:  // 0x0C
#if 1
	    con->trans = CS_LEFT;
	    con->sb = lInfo.sb;
	    con->db = lInfo.db|LATCH_1;
        /* no break? */
#endif
	case CHAR_LF:   // 0x0A
	    con->wrap = FALSE;
	    if (con->y == con->ymax) scroll ++;
	    else con->y ++;
	    break;
	case CHAR_CR:  // 0x0D
	    con->x = 0;
	    con->wrap = FALSE;
	    break;
	case CHAR_ESC:  // 0x1B
	    con->esc = EscStart;
	    continue;
	case CHAR_SO:  // 0x0E
	    con->trans = con->g[1] | G1_SET;
	    continue;
	case CHAR_SI:  // 0x0F
	    con->trans = con->g[0];
	    continue;
	default:
	    if (con->x == con->xmax + 1) 
            {
		con->wrap = TRUE;
		con->x --;
	    }

	    if (con->wrap) 
            {
		con->x -= con->xmax;
		if (con->y == con->ymax) scroll ++;
		else con->y ++;
		con->wrap = FALSE;
		buff --;
		nchars ++;
		break;
	    }

#if 1
	    if (con->knj1) 
            {
                /* special handling for Japanese char */
		if (con->knj1 & 0x80) 
                {
                  switch(lInfo.sysCoding) 
                  {
		  case CODE_EUC:
		    if (con->knj1 == (u_char)CHAR_SS2) 
                    {
			/* handling 'kata-kana' */
			if (con->ins) TextInsertChar(con,1);
			TextSput(con,ch);
			con->x ++;
			con->knj1 = 0;
			continue;
		    }
		    con->knj1 &= 0x7F;
		    ch &= 0x7F;
		    break;
		  case CODE_SJIS:
		    sjistojis(con->knj1, ch);
		    break;
		  }  // case
                }
                else
                {
                    if (con->db == (DF_BIG5_0|LATCH_1))
                        muletobig5(con->db, con->knj1, ch);
                }

                if (con->ins) TextInsertChar(con,2);
                TextWput(con,con->knj1, ch);
                con->x += 2;
                con->knj1 = 0;
                continue;
            } 
            else if (con->trans == CS_DBCS
                       || (iskanji(ch) && con->trans == CS_LEFT)) 
            {
                if (con->x == con->xmax) con->wrap = TRUE;
                con->knj1 = ch;
                    /* First Half of the double-byte char */
                continue;
            } 
            else 
            {
                if (con->ins) TextInsertChar(con,1);
                TextSput(con,con->trans == CS_RIGHT ? ch | 0x80: ch);
                con->x ++;
                continue;
            }
#endif

#if 0
            if (isPrevW1() && iskanji_byte2(ch)) 
            {
                if (con->ins) TextInsertChar(1);
                TextWput2(ch);
                con->x++;
                continue;
            } 
            else if (con->x < con->xmax && (con->trans == CS_DBCS
                       || (iskanji_byte1(ch) && con->trans == CS_LEFT))) 
            {
                if (con->ins) TextInsertChar(1);
                TextWput1(ch);
                con->x++;
                continue;
            } 
            else 
            {
                if (con->ins) TextInsertChar(1);
                TextSput(con->trans == CS_RIGHT ? ch | 0x80: ch);
                con->x ++;
                continue;
            }
#endif

        }  /* switch */

	if (scroll > 0) 
	    ScrollUp(con, scroll);
	else if (scroll < 0)
	    ScrollDown(con, - scroll);
	scroll = 0;
    }  /* while */
}

static int	ConfigCoding(const char *confstr)
{
    char reg[3][MAX_COLS];
    int n, i;

    *reg[0] = *reg[1] = *reg[2] = '\0';
    sscanf(confstr, "%s %s %s", reg[0], reg[1], reg[2]);
    for (i = 0; i < 3 && *reg[i]; i ++) 
    {
	n = (int)CodingByRegistry(reg[i]);
	if (n < 0) 
	{
	    if (!strcasecmp(reg[i], "EUC"))
		lInfo.sysCoding = CODE_EUC;
	    else if (!strcasecmp(reg[i], "SJIS"))
		lInfo.sysCoding = CODE_SJIS;
	    else if (!strcasecmp(reg[i], "BIG5"))
		lInfo.sysCoding = CODE_BIG5;
            else if (!strcasecmp(reg[i], "GB"))
                lInfo.sysCoding = CODE_GB;
	    else
		lInfo.sysCoding = 0;
	} else if (n & CHR_DBC)
	    lInfo.db = n & ~CHR_DFLD;
	else
	    lInfo.sb = n & ~CHR_SFLD;
    }
    return SUCCESS;
}

void	VtInit(void)
{
    /*  Japanese default
    DefineCap("Coding", ConfigCoding,
	      "JISX0201.1976-0 JISX0208.1983-0 EUCJ"); 
    */
     
    /* Chinese default */
    DefineCap("Coding", ConfigCoding,
              "ISO8859-1 GB2312.1980-0 GB");
}

void	VtStart(ConInfo *con)
{
   /* A serious and strange bug here, at first ymin and knj1 is not
      initialized to 0, it works fine in Linux (maybe the malloc in
      Linux will automatically zero the memory), but doesn't work in
      FreeBSD. FreeBSD's calloc zero the memory, but malloc not!!!
      So I need to initialize all the member here */

    con->x = con->y = 0;
    con->xmax = dispInfo.txmax-1;
    con->ymax = dispInfo.tymax-1;
    con->ymin = 0;   
    con->tab = 8;
    con->fcol = 7;
    con->bcol = 0;
    con->attr = 0;
    con->esc = NULL;
    con->g[0] = con->g[1] = CS_LEFT;
    con->trans = con->soft = con->ins = con->wrap = FALSE;
    con->sb = lInfo.sb;
    con->db = lInfo.db|LATCH_1;
    con->knj1 = 0;
    cursorInfo.sw = TRUE;

    con->textBuff = (u_char *)calloc(dispInfo.txmax, dispInfo.tymax);
    con->attrBuff = (u_char *)calloc(dispInfo.txmax, dispInfo.tymax);
    con->flagBuff = (u_char *)calloc(dispInfo.txmax, dispInfo.tymax); 
    con->textSize = (dispInfo.txmax) * (dispInfo.tymax);

    con->currentScroll = con->scrollLine = 0;
    con->textHead = 0;

    con->textClear = FALSE;
    con->terminate = 0;
}

void VtCleanup(ConInfo *con)
{
    if (con == NULL) return;
    con->scrollLine = con->textHead = con->currentScroll = 0;
    
    SafeFree((void **)&(con->textBuff));
    SafeFree((void **)&(con->attrBuff));
    SafeFree((void **)&(con->flagBuff));
}
