/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <glib.h>
#include <ctype.h>

#include "message.h"
#include "ch_utils.h"
#include "msg_utils.h"
#include "debug.h"

GSList *messages=NULL;

extern GdkColor colour_white;
extern GdkColor colour_black;
extern GdkColor colour_blue;
extern GdkColor colour_green;
extern GdkColor colour_red;
extern GdkColor colour_brown;
extern GdkColor colour_purple;
extern GdkColor colour_orange;
extern GdkColor colour_yellow;
extern GdkColor colour_lightgreen;
extern GdkColor colour_cyan;
extern GdkColor colour_lightcyan;
extern GdkColor colour_lightblue;
extern GdkColor colour_pink;
extern GdkColor colour_grey;
extern GdkColor colour_lightgrey;
extern GdkFont *font_bold;
extern GdkFont *font_normal;
extern GdkFont *font_italic;
extern GdkFont *font_bold_italic;

extern GdkFont *font_normal;
extern GdkFont *font_bold;
extern GdkFont *font_italic;
extern GdkFont *font_bold_italic;

GdkColor *irc_colours[] = {
  &colour_white,
  &colour_black,
  &colour_blue,
  &colour_green,
  &colour_red,
  &colour_brown,
  &colour_purple,
  &colour_orange,
  &colour_yellow,
  &colour_lightgreen,
  &colour_cyan,
  &colour_lightcyan,
  &colour_lightblue,
  &colour_pink,
  &colour_grey,
  &colour_lightgrey
};

int colour_range = 15;  /* 0 - 15 */

Message *message_new(MsgType type, Connection *connection, char *source)
{
  Message *message;

  bs_function_enter();

  message = g_malloc(sizeof(Message));
  message->type = type;
  message->connection = connection;
  if (source) {
    message->source = g_strdup(source);
  } else {
    message->source = NULL;
  }
  message->parts = NULL;

  messages = g_slist_append(messages, message);

  bs_function_leave();
  return(message);
}

MessagePart *message_part_new(GdkColor *fg_colour, GdkColor *bg_colour, GdkFont *font, char *text)
{
  MessagePart *message_part;

  bs_function_enter();

  message_part = g_malloc(sizeof(MessagePart));
  message_part->fg_colour = fg_colour;
  message_part->bg_colour = bg_colour;
  message_part->font = font;
  message_part->text = g_strdup(text);

  bs_function_leave();
  return(message_part);
}

GSList *message_part_parse_new(GSList *parts_list, GdkColor *default_colour, char *text)
{
  char *text_copy, *start, *current, *end;
  int fg_number=0, bg_number=1;
  GdkColor *fg_colour;
  GdkColor *bg_colour;
  unsigned bold = FALSE;
  unsigned reverse = FALSE;
  unsigned underline = FALSE;

  bs_function_enter();

  fg_colour = default_colour;
  bg_colour = &colour_black;

  text_copy = g_strdup(text);
  start = current = text_copy;
  end = start + strlen(text_copy);

  while (*current && (current < end)) {
    if (*current == '\003') {
      current[0] = '\0';
      if (strlen(start)) {
	parts_list = g_slist_append(parts_list, 
			   message_part_new( (reverse ? bg_colour : fg_colour), 
				       (reverse ? fg_colour : bg_colour),
				       (bold ? (underline ? font_bold_italic : font_bold) : (underline ? font_italic : font_normal) ),
				       start));
      }
      current++;
      if ( (current < end) && isdigit(current[0]) ) {
	fg_number = current[0] - '0';
	current++;
	if ( (current < end) && isdigit(current[0])) {
	    fg_number *= 10;
	    fg_number += current[0] - '0';
	    current++;
	}
	if ( (current < end) && (current[0] == ',') ) {
	  start = current++;
	  if ( (current < end) && isdigit(current[0]) ) {
	    bg_number = current[0] - '0';
	    current++;
	  } else start = current--;
	  if ( (current < end) && isdigit(current[0])) {
	    bg_number *= 10;
	    bg_number += current[0] - '0';
	    current++;
	  }  
	}
	if (fg_number > colour_range) {
	  fg_colour = default_colour;
	} else {
	  fg_colour = irc_colours[fg_number];
	}
	if (bg_number > colour_range) {
	  bg_colour = &colour_black;
	} else {
	  bg_colour = irc_colours[bg_number];
	}
      }
      start = current;
      fg_number = 0;
      bg_number = 1;
    } else if (*current == '\002') { /* ^B */
      current[0] = '\0';
      if (strlen(start)) {
	parts_list = g_slist_append(parts_list, 
			    message_part_new( (reverse ? bg_colour : fg_colour), 
				       (reverse ? fg_colour : bg_colour),
				       (bold ? (underline ? font_bold_italic : font_bold) : (underline ? font_italic : font_normal) ),
				       start));
      }
      bold = !bold;
      start = ++current;
    } else if (*current == '\026') { /* ^V */
      current[0] = '\0';
      if (strlen(start)) {
	parts_list = g_slist_append(parts_list, 
			   message_part_new( (reverse ? bg_colour : fg_colour), 
				       (reverse ? fg_colour : bg_colour),
                                       (bold ? (underline ? font_bold_italic : font_bold) : (underline ? font_italic : font_normal) ),
				       start));
      }
      reverse = !reverse;
      start = ++current;
    } else if (*current == '\037') { /* ^_ */
      current[0] = '\0';
      if (strlen(start)) {
	parts_list = g_slist_append(parts_list, 
			   message_part_new( (reverse ? bg_colour : fg_colour), 
				       (reverse ? fg_colour : bg_colour),
                                       (bold ? (underline ? font_bold_italic : font_bold) : (underline ? font_italic : font_normal) ),
				       start));
      }
      underline = !underline;
      start = ++current;
    } else {
      current++;
    }
  }
  if (strlen(start)) {
    parts_list = g_slist_append(parts_list, 
			 message_part_new( (reverse ? bg_colour : fg_colour), 
				       (reverse ? fg_colour : bg_colour),
                                       (bold ? (underline ? font_bold_italic : font_bold) : (underline ? font_italic : font_normal) ),
					start));
  }

  bs_function_leave();
  return(parts_list);
}

void message_parts_free(GSList *message_parts)
{
  GSList *glist_entry;
  MessagePart *message_part;

  bs_function_enter();

  glist_entry = message_parts;
  while (glist_entry) {
    message_part = glist_entry->data;
    g_free(message_part->text);
    g_free(message_part);
    glist_entry = g_slist_next(glist_entry);
  }
  g_slist_free(message_parts);

  bs_function_leave();
  return;
}

void messages_write()
{
  GSList *message_ptr;
  Message *message;
  BezBaseWindow *window;
  GSList *glist_entry;
  MessagePart *message_part;
  ChannelInfo *target_channel;
  int numchars=0;

  bs_function_enter();

  message_ptr = messages;

  while (message_ptr) {
    message = message_ptr->data;

    if (!message->connection) {
      printf("\n\n\nERROR: message->connection == NULL\n\n\n");
      message_ptr = g_slist_next(message_ptr);
      continue;
    }
    if (!message->parts) {
      message_ptr = g_slist_next(message_ptr);
      continue;
    }

    switch (message->type)
      {
      case MT_CONSOLE:
	window = BEZ_BASE_WINDOW(message->connection->console);
	break;
      case MT_CHANNEL:
	if (!message->source) {
	  if (message->connection->current_window &&
	      (message->connection->current_window->type == BEZ_CHANNEL) ) {
	    window = BEZ_BASE_WINDOW(message->connection->current_window);
	  } else {
	    window = BEZ_BASE_WINDOW(message->connection->console);
	  }
	} else {
	  target_channel = find_channel(message->connection->channels, message->source);
	  if ( (window = BEZ_BASE_WINDOW(target_channel->window)) == NULL ) {
	    /* Window hasn't been created yet, send to console window */;
	    window = BEZ_BASE_WINDOW(message->connection->console);
	  }
	}
	break;
      case MT_NICK:
	if ( !(window = BEZ_BASE_WINDOW(find_message(message->connection->messages, message->source))) ) {
	  if (message->connection->current_window) {
	    window = BEZ_BASE_WINDOW(message->connection->current_window);
	  } else {
	    window = BEZ_BASE_WINDOW(message->connection->console);
	  }
	}
	break;
      default:
	target_channel = find_channel(message->connection->channels, message->source);
	if ( (window = BEZ_BASE_WINDOW(target_channel->window)) == NULL ) {
	  window = BEZ_BASE_WINDOW(message->connection->console);
	}
      }

    gtk_text_freeze(GTK_TEXT(window->text));
    if (window->numlines) {
      gtk_text_insert (GTK_TEXT (window->text), NULL, NULL, NULL, "\n", -1);
    }

    glist_entry = message->parts;
    while (glist_entry) {
      message_part = glist_entry->data;
      
      gtk_text_insert (GTK_TEXT (window->text), 
		       (message_part->font ? message_part->font : font_normal), 
		       message_part->fg_colour, 
		       message_part->bg_colour, message_part->text, -1);
      glist_entry = g_slist_next(glist_entry);
    }
    
    gtk_text_thaw(GTK_TEXT(window->text));
    
    if (++(window->numlines) > 200) {
      gtk_text_freeze(GTK_TEXT(window->text));
      while( GTK_TEXT_INDEX( GTK_TEXT(window->text), numchars) != '\n') {
	numchars++;
      }
      gtk_text_set_point( GTK_TEXT(window->text), 0);
      gtk_text_forward_delete(  GTK_TEXT(window->text), numchars+1);
      gtk_text_set_point( GTK_TEXT(window->text), 
			  gtk_text_get_length( GTK_TEXT(window->text)));
      (window->numlines)--;
      gtk_text_thaw(GTK_TEXT(window->text));
    }
  
    GTK_TEXT(window->text)->vadj->value = GTK_TEXT(window->text)->vadj->upper - 
      GTK_TEXT(window->text)->vadj->page_size;
    gtk_signal_emit_by_name ( GTK_OBJECT(GTK_TEXT(window->text)->vadj), "value_changed");

    message_parts_free(message->parts);
    if (message->source) {
      g_free(message->source);
    }
    g_free(message);

    message_ptr = g_slist_next(message_ptr);
  }

  g_slist_free(messages);
  messages = NULL;

  bs_function_leave();
  return;
}
