/* cmd-switch.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "libfsutils/rmrf.h"
#include "libarch/archive.h"
#include "libarch/chatter.h"
#include "libarch/conflict-handling.h"
#include "libarch/inode-sig.h"
#include "libarch/local-cache.h"
#include "libarch/make-changeset.h"
#include "libarch/my.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/proj-tree-lint.h"
#include "libarch/patch-logs.h"
#include "libarch/changeset-report.h"
#include "commands/cmdutils.h"
#include "commands/cmd.h"
#include "commands/apply-delta.h"
#include "commands/switch.h"
#include "commands/version.h"



static t_uchar * usage = "[options] [package]";

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string and exit.") \
  OP (opt_dir, "d", "dir DIR", 1, \
      "Change to DIR first.") \
  OP (opt_quiet, "q", "quiet", 0, \
      "Suppress progress information")


t_uchar arch_cmd_switch_help[] = (
 "change the working trees version to that of package\n"
 
 "change the working trees version to that of package, and make\n"
 "equivalent to revision. Preserves uncommitted changes.\n"
 "\n"
 "Note: baz automatically switches to the _latest_ revision in the\n"
 "supplied version.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_switch (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * dir = 0;
  int quiet = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      if ((argc > 1) && !str_cmp ("--", argv[1]))
        break;

      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_switch_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;


        case opt_dir:
          {
            lim_free (0, dir);
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_quiet:
          {
            quiet = 1;
            break;
          }

        }
    }

  {
    arch_project_tree_t * tree;
    t_uchar * target_revision;
    arch_patch_id * patch_id;
    t_uchar * tree_revision;
    struct arch_archive * arch = 0;
    int result;


    tree = arch_project_tree_new (talloc_context, dir);

    if (!tree->root)
      {
        safe_printfmt (2, "%s: not in a project tree\n  dir: %s\n",
                       argv[0], dir);
        exit (2);
      }

    if (argc != 2)
      arch_cmd_fail (argv[0], "no target version or revision provided.\n");


    tree_revision = arch_project_tree_revision (argv[0], tree->root);
    

    /* TODO: fix -d parameters lack of use */
    target_revision = arch_determine_fqrevision(&arch, tree->archive, argv[1], argv[0]);
    patch_id = arch_patch_id_new (target_revision);

    arch_tree_ensure_no_conflicts (tree);

    /* remember that call_cmd never returns, so we have to set the 
     * tree version now, before we move on to appylying the delta
     */

    arch_set_tree_version (tree->root, patch_id);

    result = arch_call_cmd (arch_cmd_apply_delta, argv[0], 
                          "-A", tree->archive,
                          "-d", tree->root, tree_revision,
                          target_revision, NULL);
    arch_project_tree_delete (tree);
    talloc_free (patch_id);
    return result;
  }
  return 0;
}
