/* cachedrevs.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "libarch/pristines.h"
#include "libarch/build-revision.h"
#include "commands/cmd.h"
#include "commands/cachedrevs.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [version]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit."))

t_uchar arch_cmd_cachedrevs_help[] = N_("list cached revisions in an archive\n"
                                      "Report which revisions of VERSION have been cached\n"
                                      "as whole trees in the archive.\n"
                                      "\n"
                                      "See also \"baz cacherev -H\".\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_cachedrevs (t_uchar *program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_cachedrevs_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

       }
    }

  if (argc > 2)
    goto usage_error;

  default_archive = arch_my_default_archive (default_archive);

  {
    t_uchar * version_spec;
    t_uchar * version = 0;
    t_uchar * package = 0;
    struct arch_archive * arch = 0;

    if (argc == 2)
      version_spec = str_save(0, argv[1]);
    else
      version_spec = arch_try_tree_version(program_name);

    arch = arch_archive_connect_branch (version_spec, &package);
    if (!arch)
      {
          safe_printfmt (2, "%s is not a valid version (or url).\n", version_spec);
          exit (1);
      }
    if (!arch_valid_package_name (package, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, "%s: invalid version spec (%s)\n",
                       argv[0], package);
        exit (2);
      }

    version = arch_parse_package_name (arch_ret_non_archive, 0, package);

    {
      rel_table revisions = 0;
      int x;

      revisions = arch_archive_revisions (arch, version, 2);

      for (x = 0; x < rel_n_records (revisions); ++x)
        {
          int is_cached;
          int has_ancestry;
          arch_patch_id * revision = arch_patch_id_new_archive (arch->official_name, revisions[x][0]);

          arch_revision_type (0, &is_cached, &has_ancestry, arch, revision);

          if (is_cached)
            safe_printfmt (1, "%s\n", revisions[x][0]);
          talloc_free (revision);
        }

      rel_free_table (revisions);
    }

    arch_archive_close (arch);

    lim_free (0, version_spec);
    lim_free (0, package);
    lim_free (0, version);
  }

  lim_free (0, default_archive);
  return 0;
}




/* tag: Tom Lord Thu May 29 23:36:53 2003 (cachedrevs.c)
 */
