;;;
;;; tm-bbdb.el --- tm shared module for BBDB
;;;
;;; Copyright (C) 1995 KOBAYASHI Shuhei <shuhei@cmpt01.phys.tohoku.ac.jp>
;;;
;;; Author: KOBAYASHI Shuhei <shuhei@cmpt01.phys.tohoku.ac.jp>
;;; Maintainer: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Version: $Id: tm-bbdb.el,v 4.0 1996/01/26 08:20:29 morioka Exp $
;;; Keywords: mail, news, MIME, multimedia, multilingual, BBDB
;;;
;;; This file is part of tm (Tools for MIME).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Code:

(require 'bbdb)
(require 'tl-822)
(require 'tm-ew-d)


;;; @ mail-extr
;;;

(defvar tm-bbdb/use-mail-extr t)

(defun tm-bbdb/extract-address-components (str)
  (let* ((ret     (rfc822/extract-address-components str))
         (phrase  (car ret))
         (address (cdr ret))
         (methods tm-bbdb/canonicalize-full-name-methods))
    (while (and phrase methods)
      (setq phrase  (funcall (car methods) phrase)
            methods (cdr methods)))
    (if (string= phrase "")
        (setq phrase nil))
    (cons phrase address)
    ))

(or tm-bbdb/use-mail-extr
    (progn
      (fset 'mail-extract-address-components
	    (symbol-function 'tm-bbdb/extract-address-components))
      (provide 'mail-extr)
      ))


;;; @ bbdb-extract-field-value
;;;

(or (fboundp 'tm:bbdb-extract-field-value)
    (progn
      ;; (require 'bbdb-hooks) ; not provided.
      ;; (or (fboundp 'bbdb-extract-field-value) ; defined as autoload
      (or (fboundp 'bbdb-header-start)
          (load "bbdb-hooks"))
      (fset 'tm:bbdb-extract-field-value
            (symbol-function 'bbdb-extract-field-value))
      (defun bbdb-extract-field-value (field)
        (let ((value (rfc822/get-field-body field)))
          (and value
               (mime-eword/decode-string value))))
      ))


;;; @ full-name canonicalization methods
;;;

(defun tm-bbdb/canonicalize-spaces (str)
  (let (dest)
    (while (string-match "\\s +" str)
      (setq dest (cons (substring str 0 (match-beginning 0)) dest))
      (setq str (substring str (match-end 0)))
      )
    (or (string= str "")
        (setq dest (cons str dest)))
    (setq dest (nreverse dest))
    (mapconcat 'identity dest " ")
    ))

(defun tm-bbdb/canonicalize-dots (str)
  (let (dest)
    (while (string-match "\\." str)
      (setq dest (cons (substring str 0 (match-end 0)) dest))
      (setq str (substring str (match-end 0)))
      )
    (or (string= str "")
        (setq dest (cons str dest)))
    (setq dest (nreverse dest))
    (mapconcat 'identity dest " ")
    ))

(defvar tm-bbdb/canonicalize-full-name-methods
  '(mime-eword/decode-string
    tm-bbdb/canonicalize-dots
    tm-bbdb/canonicalize-spaces))


;;; @ BBDB functions for mime/viewer-mode
;;;

(defvar tm-bbdb/auto-create-p nil)

(defun tm-bbdb/update-record (&optional offer-to-create)
  "Return the record corresponding to the current MIME previewing message.
Creating or modifying it as necessary. A record will be created if
tm-bbdb/auto-create-p is non-nil, or if OFFER-TO-CREATE is non-nil and
the user confirms the creation."
  (save-excursion
    (if (and mime::article/preview-buffer
             (get-buffer mime::article/preview-buffer))
        (set-buffer mime::article/preview-buffer))
    (if bbdb-use-pop-up
        (tm-bbdb/pop-up-bbdb-buffer offer-to-create)
    (let ((from (rfc822/get-field-body "From")))
      (if (or (null from)
              (string-match (bbdb-user-mail-names)
                            (car
                             (cdr
                              ;; (tm-bbdb/extract-address-components from)
                              (mail-extract-address-components from)
                              ))))
          (setq from (or (rfc822/get-field-body "To")
                         from)))
      (if from
          (bbdb-annotate-message-sender
           from t
           (or (bbdb-invoke-hook-for-value tm-bbdb/auto-create-p)
               offer-to-create)
           offer-to-create))
      ))))

(defun tm-bbdb/annotate-sender (string)
  "Add a line to the end of the Notes field of the BBDB record 
corresponding to the sender of this message."
  (interactive
   (list (if bbdb-readonly-p
             (error "The Insidious Big Brother Database is read-only.")
           (read-string "Comments: "))))
  (bbdb-annotate-notes (tm-bbdb/update-record t) string))

(defun tm-bbdb/edit-notes (&optional arg)
  "Edit the notes field or (with a prefix arg) a user-defined field
of the BBDB record corresponding to the sender of this message."
  (interactive "P")
  (let ((record (or (tm-bbdb/update-record t)
                    (error ""))))
    (bbdb-display-records (list record))
    (if arg
	(bbdb-record-edit-property record nil t)
      (bbdb-record-edit-notes record t))))

(defun tm-bbdb/show-sender ()
  "Display the contents of the BBDB for the sender of this message.
This buffer will be in bbdb-mode, with associated keybindings."
  (interactive)
  (let ((record (tm-bbdb/update-record t)))
    (if record
	(bbdb-display-records (list record))
	(error "unperson"))))

(defun tm-bbdb/pop-up-bbdb-buffer (&optional offer-to-create)
  "Make the *BBDB* buffer be displayed along with the MIME preview window(s),
displaying the record corresponding to the sender of the current message."
  (bbdb-pop-up-bbdb-buffer
    (function (lambda (w)
      (let ((b (current-buffer)))
	(set-buffer (window-buffer w))
	(prog1 (eq major-mode 'mime/viewer-mode)
	  (set-buffer b))))))
  (let ((bbdb-gag-messages t)
	(bbdb-use-pop-up nil)
	(bbdb-electric-p nil))
    (let ((record (tm-bbdb/update-record offer-to-create))
	  (bbdb-elided-display (bbdb-pop-up-elided-display))
	  (b (current-buffer)))
      (bbdb-display-records (if record (list record) nil))
      (set-buffer b)
      record)))


;;; @ end
;;;

(provide 'tm-bbdb)

(run-hooks 'tm-bbdb-load-hook)

;;; end of tm-bbdb.el
