/*
 * dnode.c - Ultrix node functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dnode.c,v 1.4 98/03/18 16:23:49 abe Exp $";
#endif


#include "lsof.h"

#define	KERNEL
#include <sys/fs_types.h>
#undef	KERNEL


/*
 * Local definitions
 */

struct l_lock {				/* local lock info */
    struct flock set;			/* lock data */
    struct l_lock *next;
};

struct l_flinfo {			/* local file lock info */
    dev_t dev;				/* device of lock */
    ino_t ino;				/* inode number of lock */
    struct l_lock *lp;			/* lock information */
    struct l_flinfo *next;
};

#define	L_FLINFO_HSZ	256		/* local file lock information hash
					 * table size (must be a power of 2) */
#define L_FLINFO_HASH(d, i)	(((((int)(d) + (int)i) *31415) >> 5) & (L_FLINFO_HSZ - 1))


/*
 * Local static variables
 */

static struct l_flinfo **Flinfo = (struct l_flinfo **)NULL;
					/* local file lock hash buckets */
static int FlinfoSt = 0;		/* Flinfo[] load status */


/*
 * Local function prototypes
 */

_PROTOTYPE(static char isglocked,(void));
_PROTOTYPE(static int load_flinfo,(void));


/*
 * clr_flinfo() - clear local file lock table information
 */

void
clr_flinfo()
{
	struct l_lock *lf, *lfn;
	int i;
	struct l_flinfo *fi, *fin;

	if (!Flinfo && !FlinfoSt)
	    return;
	for (i = 0; i < L_FLINFO_HSZ; i++) {
	    if (!(fi = Flinfo[i]))
		continue;
	    do {
		if ((lf = fi->lp)) {
		    do {
			lfn = lf->next;
			(void) free((FREE_P *)lf);
		    } while ((lf = lfn));
		}
		fin = fi->next;
		(void) free((FREE_P *)fi);
	    } while ((fi = fin));
	}
	FlinfoSt = 0;
}


/*
 * isglocked() - is gnode locked?
 */

static char
isglocked()
{
	struct l_flinfo *fp;
	int i, l;
	struct l_lock *lp;

	if (!Lf->dev_def || Lf->inp_ty != 1)
	    return(' ');
	if (!Flinfo || !FlinfoSt) {
	    if (!load_flinfo())
		return(' ');
	}
/*
 * Hash the device and inode number and see if there's a local file lock
 * information structure for it.
 */
	i = L_FLINFO_HASH(Lf->dev, Lf->inode);
	for (fp = Flinfo[i]; fp; fp = fp->next) {
	    if (fp->dev == (dev_t)Lf->dev && fp->ino == (ino_t)Lf->inode)
		break;
	}
	if (!fp)
	    return(' ');
/*
 * Search the device and inode number lock list for one held by this process.
 */
	for (lp = fp->lp; lp; lp = lp->next) {
	    if (lp->set.l_pid != (pid_t)Lp->pid)
		continue;
	    if (lp->set.l_whence == 0 && lp->set.l_start == 0
	    &&  lp->set.l_len == 0x7fffffff)
		l = 1;
	    else
		l = 0;
	    if (lp->set.l_type == F_WRLCK)
		return(l ? 'W' : 'w');
	    else if (lp->set.l_type == F_RDLCK)
		return(l ? 'R' : 'r');
	    return(' ');
	}
	return(' ');
}


/*
 * load_flinfo() - load local file lock information
 */

static int
load_flinfo()
{
	struct flino fi;
	struct filock fl;
	KA_T fif, fip, flf, flp;
	int i;
	struct l_flinfo *lfi;
	struct l_lock *ll;
	KA_T v;

	if (Flinfo && FlinfoSt)
	    return(1);
/*
 * Get kernel fids chain pointer.
 */
	if (get_Nl_value("fids", Drive_Nl, &v) < 0 || !v
	||  kread(v, (char *)&fip, sizeof(fip)))
	    return(0);
/*
 * Define local hash buckets, if necessary.
 */
	if (!Flinfo) {
	    if ((Flinfo = (struct l_flinfo **)calloc(sizeof(struct flinfo *),
						     L_FLINFO_HSZ))
	    == (struct l_flinfo **)NULL) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d byte local lock hash buckets\n",
		    Pn, L_FLINFO_HSZ * sizeof(struct l_flinfo *));
		Exit(1);
	    }
	}
/*
 * Follow the fids chain.
 */
	if (!(fif = fip))
	    return(1);
    /*
     * Follow the filock chain for this fid entry.
     * Duplicate it via the lock file lock information hash buckets.
     */
	do {
	    if (kread(fip, (char *)&fi, sizeof(fi)))
		return(0);
	    if (!(flf = (KA_T)fi.fl_flck))
		continue;
	/*
	 * Allocate a local file lock information structure for this fid.
	 */
	    if (!(lfi = (struct l_flinfo *)malloc(sizeof(struct l_flinfo)))) {
		(void) fprintf(stderr,
		    "%s: no space for local vnode lock info struct\n", Pn);
		Exit(1);
	    }
	    lfi->dev = fi.fl_dev;
	    lfi->ino = fi.fl_number;
	    lfi->lp = (struct l_lock *)NULL;
	    lfi->next = (struct l_flinfo *)NULL;
	/*
	 * Follow the flino's filock chain, duplicating it locally.
	 */
	    flp = flf;
	    do {
		if (kread(flp, (char *)&fl, sizeof(fl)))
		    break;
	    /*
	     * Allocate a local lock information structure and link it
	     * to the chain for its vnode.
	     */
		if (!(ll = (struct l_lock *)malloc(sizeof(struct l_lock)))) {
		    (void) fprintf(stderr,
			"%s: no space for local lock struct\n", Pn);
		    Exit(1);
		}
		ll->set = fl.set;
		ll->next = lfi->lp;
		lfi->lp = ll;
	    } while ((flp = (KA_T)fl.next) && flp != flf);
	/*
	 * Link the local file lock information structure to its hash bucket.
	 */
	    i = L_FLINFO_HASH(lfi->dev, lfi->ino);
	    lfi->next = Flinfo[i];
	    Flinfo[i] = lfi;
	} while ((fip = (KA_T)fi.next) && fip != fif);
	FlinfoSt = 1;
	return(1);
}


/*
 * process_node() - process gnode
 */

void
process_node(ga)
	KA_T ga;			/* gnode kernel space address */
{
	static KA_T cgops = (KA_T)0;
	struct fs_data f;
	static unsigned long fgops;
	static int ft = 1;
	struct gnode g;
	struct mount m;
	struct mntinfo *mi;
	static KA_T nfsgops = (KA_T)0;
	char tbuf[32], *tn;
	enum vtype type;
	static KA_T ugops = (KA_T)0;
	struct v_fs_data *vf;
	struct vnode *v = (struct vnode *)NULL;

#if	ULTRIXV>=40200
	struct fifonode fn;
#endif	/* ULTRIXV>=40200 */

/*
 * Do first-time only operations.
 */
	if (ft) {
	    if (get_Nl_value("cgops", Drive_Nl, &cgops) < 0 || !cgops)
		cgops = (KA_T)0;
	    if (get_Nl_value("fgops", Drive_Nl, &fgops) < 0 || !fgops)
		fgops = (KA_T)0;
	    if (get_Nl_value("nfsgops", Drive_Nl, &nfsgops) < 0 || !nfsgops)
		nfsgops = (KA_T)0;
	    if (get_Nl_value("ugops", Drive_Nl, &ugops) < 0 || !ugops)
		ugops = (KA_T)0;
	    ft = 0;
	}
/*
 * Read the gnode.
 */
	if (!ga) {
	    enter_nm("no gnode address");
	    return;
	}
	if (readgnode(ga, &g)) {
	    enter_nm(Namech);
	    return;
	}

#if	defined(HASNCACHE)
	Lf->id = (unsigned long)g.g_id;
	Lf->na = ga;
#endif	/* defined(HASNCACHE) */

/*
 * Determine the node type from the operations switch.
 */
	if (g.g_ops) {
	    if (nfsgops && nfsgops == (KA_T)g.g_ops)
		Ntype = N_NFS;

#if	ULTRIXV>=40200
	    else if (cgops && cgops == (KA_T)g.g_ops)
		Ntype = N_CDFS;
	    else if (fgops && fgops == (KA_T)g.g_ops)
		Ntype = N_FIFO;
	    else if (ugops && ugops == (KA_T)g.g_ops)
		Ntype = N_UFS;
#endif	/* ULTRIXV>=40200 */

	}
/*
 * Get the mount structure and its fs_data structure.
 * (Ultrix 4.2 and above N_FIFO nodes might not have a mount structure.)
 */
	if (Ntype != N_FIFO || (Ntype == N_FIFO && g.g_mp)) {
	    if (!g.g_mp
	    ||  kread((KA_T)g.g_mp, (char *)&m, sizeof(m))) {
		(void) sprintf(Namech, "bad mount for %s at %s",
		    print_kptr(ga, tbuf),
		    print_kptr((KA_T)g.g_mp, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	    if (!m.m_fs_data
	    ||  kread((KA_T)m.m_fs_data, (char *)&f, sizeof(f))) {
		(void) sprintf(Namech, "bad fsdata (%s) for mount at %s",
		    print_kptr((KA_T)m.m_fs_data, tbuf),
		    print_kptr((KA_T)g.g_mp, (char *)NULL));
		enter_nm(Namech);
		return;
	    }
	} else
	    f.fd_path[0] = f.fd_devname[0] = '\0';
/*
 * If there is a further node, access it.
 */
	switch (Ntype) {

#if	ULTRIXV>=40200
	case N_FIFO:
	    if (!g.g_fifo
	    ||  kread((KA_T)g.g_fifo, (char *)&fn, sizeof(fn))) {
		(void) sprintf(Namech, "bad fifonode address (%s)",
		    print_kptr((KA_T)g.g_fifo, (char *)NULL));
		(void) strcpy(Lf->type, ((g.g_mode & GFMT) == GFPIPE) ? "PIPE"
								      : "FIFO");
		enter_nm(Namech);
	    }
	    break;
#endif	/* ULTRIXV>=40200 */

	case N_NFS:
	    v = (struct vnode *)&g;
	    vf = (struct v_fs_data *)&f;
	    mi = (struct mntinfo *)&vf->fd_un.gvfs.mi;
	}
/*
 * Set the device and type for printing.
 */
	Lf->dev = g.g_dev;
	Lf->dev_def = 1;
	switch (Ntype) {
	case N_NFS:
	    type = v->v_type;
	    switch (v->v_type) {
	    case VNON:
		tn = "VNON";
		break;
	    case VREG:
		tn = "VREG";
		break;
	    case VDIR:
		tn = "VDIR";
		break;
	    case VBLK:
		tn = "VBLK";
		Lf->dev = g.g_rdev;
		Ntype = N_BLK;
		break;
	    case VCHR:
		tn = "VCHR";
		Lf->dev = g.g_rdev;
		Ntype = N_CHR;
		break;
	    case VLNK:
		tn = "VLNK";
		break;
	    case VSOCK:
		tn = "VSOCK";
		break;
	    case VBAD:
		tn = "VBAD";
		break;
	    case VFIFO:
		tn = "VFIFO";
		break;
	    default:
		tn = "VNON";
		break;
	    }
	    break;
	default:
	    switch (g.g_mode & GFMT) {
	    case GFPORT:
		tn = "PORT";
		type = VFIFO;
		break;
	    case GFCHR:
		tn = "GCHR";
		type = VCHR;
		Lf->dev = g.g_rdev;
		Ntype = N_CHR;
		break;
	    case GFDIR:
		tn = "GDIR";
		type = VDIR;
		break;
	    case GFBLK:
		tn = "GBLK";
		type = VBLK;
		Lf->dev = g.g_rdev;
		Ntype = N_BLK;
		break;
	    case GFREG:
		tn = "GREG";
		type = VREG;
		break;
	    case GFLNK:
		tn = "GLNK";
		type = VLNK;
		break;
	    case GFSOCK:
		tn = "SOCK";
		type = VSOCK;
		break;

#if	ULTRIXV>=40200
	    case GFPIPE:
		Lf->dev_def = 0;
		enter_dev_ch(print_kptr((KA_T)g.g_fifo, (char *)NULL));
		tn = "PIPE";
		type = VFIFO;
		break;
#endif	/* ULTRIXV>=40200 */

	    default:
		tn = "GNON";
		type = VNON;
	    }
	    break;
	}
	(void) strcpy(Lf->type, tn);
	Lf->ntype = Ntype;
/*
 * Obtain the inode number.
 */

	if (type != VBLK) {
	    if ((Lf->inode = (unsigned long)g.g_number))
		Lf->inp_ty = 1;
	}
/*
 * Record lock information.
 */
	if (FILEPTR && (FILEPTR->f_flag & FSHLOCK))
	    Lf->lock = 'R';
	else if (FILEPTR && (FILEPTR->f_flag & FEXLOCK))
	    Lf->lock = 'W';
	else
	    Lf->lock = isglocked();
/*
 * Obtain the file size.
 */
	if (Foffset)
	    Lf->off_def = 1;
	else {

#if	ULTRIXV>=40200
	    if (type == N_FIFO) {
		Lf->sz = (SZOFFTYPE) (Lf->access == 'r') ? fn.fn_rptr
							 : fn.fn_wptr;
		Lf->sz_def = 1;
	    } else
#endif	/* ULTRIXV>=40200 */

	    {
		if (type == VREG || type == VDIR) {
			Lf->sz = (SZOFFTYPE)g.g_size;
			Lf->sz_def = 1;
		} else if ((type == VCHR || type == VBLK) && !Fsize)
			Lf->off_def = 1;
	    
	    }
	}
/*
 * Record an NFS file selection.
 */
	if (Ntype == N_NFS && Fnfs)
	    Lf->sf |= SELNFS;
/*
 * Save the file system names.
 */
	if (f.fd_path[0] && strlen(f.fd_path) > 0) {
	    if (!(Lf->fsdir = mkstrcpy(f.fd_path, (MALLOC_S *)NULL))) {
		(void) fprintf(stderr,
		    "%s: no fs_data path name space, PID %d\n",
		    Pn, Lp->pid);
		Exit(1);
	    }
	} else
	    Lf->fsdir = (char *)NULL;
	if (f.fd_devname[0] && strlen(f.fd_devname) > 0) {
	    if (!(Lf->fsdev = mkstrcpy(f.fd_devname, (MALLOC_S *)NULL))) {
		(void) fprintf(stderr,
		    "%s: no fs_data path name space, PID %d\n",
		    Pn, Lp->pid);
		Exit(1);
	    }
	} else
	    Lf->fsdev = (char *)NULL;

#if	defined(HASBLKDEV)
/*
 * If this is a VBLK file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VBLK && Lf->dev_def)
	    find_bl_ino();
#endif	/* defined(HASBLKDEV) */

/*
 * If this is a VCHR file and it's missing an inode number, try to
 * supply one.
 */
	if (Lf->inp_ty == 0 && type == VCHR && Lf->dev_def)
	    find_ch_ino();
/*
 * Test for specified file.
 */
	if (Sfile && is_file_named(NULL, type))
	    Lf->sf |= SELNM;
/*
 * Enter name characters.
 */
	if (Namech[0])
	    enter_nm(Namech);
}
