/*
 * dmnt.c - AIX mount support functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.6 98/07/17 08:48:47 abe Exp $";
#endif


#include "lsof.h"


/*
 * Local static definitions
 */

static struct mounts *Lmi = (struct mounts *)NULL;	/* local mount info */
static int Lmist = 0;					/* Lmi status */


/*
 * readmnt() - read mount table
 */

struct mounts *
readmnt()
{
	char *dir, *fs, *h, *ln, *ty;
	char *dn = (char *)NULL;
	struct mounts *mtp;
	int nm;
	struct stat sb;
	unsigned sz;
	struct vmount *v;
	struct vmount *vt = (struct vmount *)NULL;

	if (Lmi || Lmist)
	    return(Lmi);
/*
 * Read the table of vmount structures.
 */
	for (sz = sizeof(struct vmount);;) {
	    if (!(vt = (struct vmount *)malloc(sz))) {
		(void) fprintf(stderr, "%s: no space for vmount table\n", Pn);
		return(0);
	    }
	    nm = mntctl(MCTL_QUERY, sz, vt);
	    if (nm > 0) {
		if (vt->vmt_revision != VMT_REVISION) {
		    (void) fprintf(stderr,
			"%s: stale file system, rev %d != %d\n",
			Pn, vt->vmt_revision, VMT_REVISION);
		    return(0);
		}
		break;
	    }
	    if (nm == 0) {
		sz = (unsigned)vt->vmt_revision;
		(void) free((FREE_P *)vt);
	    } else {
		(void) fprintf(stderr, "%s: mntctl error: %s\n",
		    Pn, strerror(errno));
		return(0);
	    }
	}
/*
 * Scan the vmount structures and build Lmi.
 */
	for (v = vt; nm--; v = (struct vmount *)((char *)v + v->vmt_length)) {
	    dir = (char *)vmt2dataptr(v, VMT_STUB);
	    fs = (char *)vmt2dataptr(v, VMT_OBJECT);
	    h = (char *)vmt2dataptr(v, VMT_HOST);
            if (statsafely(dir, &sb)) {
		if (!Fwarn) {

		/*
		 * Issue stat() failure warning.
		 */
		    switch(v->vmt_gfstype) {

#if	defined(HAS_AFS)
		    case MNT_AFS:
			ty = "afs";
			break;
#endif	/* defined(HAS_AFS) */

		    case MNT_AIX:
			ty = "oaix";
			break;
		    case MNT_NFS:
			ty = "nfs";
			break;
		    case MNT_JFS:
			ty = "jfs";
			break;
		    case MNT_CDROM:
			ty = "cdrom";
			break;
		    default:
			ty = "unknown";
		    }
		    (void) fprintf(stderr,
			"%s: WARNING: can't stat() %s file system %s\n",
			Pn, ty, dir);
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
	    /*
	     * Assemble alternate device number and mode flags.
	     */
		(void) bzero((char *)&sb, sizeof(sb));
		if (v->vmt_flags & MNT_REMOTE)
		    sb.st_dev = (dev_t)(SDEV_REMOTE | v->vmt_vfsnumber);
		else {

#if	defined(HAS_AFS)
		    if (v->vmt_gfstype == MNT_AFS)
			sb.st_dev = AFSDEV;
		    else
#endif	/* defined(HAS_AFS) */

			sb.st_dev = (dev_t)v->vmt_fsid.val[0];
		}
		if (!Fwarn)
		    (void) fprintf(stderr,
			"      assuming \"dev=%#lx\" from mount table\n",
			sb.st_dev);
		sb.st_mode = S_IFDIR | 0777;
	    }
	/*
	 * Allocate space for the directory (mounted on) and resolve
	 * any symbolic links.
	 */
	    if (dn)
		(void) free((FREE_P *)dn);
	    if (!(dn = mkstrcpy(dir, (MALLOC_S *)NULL))) {

no_space_for_mount:

		(void) fprintf(stderr, "%s: no space for mount at %s (%s)\n",
		    Pn, fs, dir);
		Exit(1);
	    }
	    if (!(ln = Readlink(dn))) {
		if (!Fwarn) {
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
		continue;
	    }
	    if (ln != dn) {
		(void) free((FREE_P *)dn);
		dn = ln;
	    }
	/*
	 * Allocate a local mounts structure and fill the directory information.
	 */
	    if (!(mtp = (struct mounts *)malloc(sizeof(struct mounts))))
		goto no_space_for_mount;
	    mtp->dir = dn;
	    dn = (char *)NULL;
	    mtp->dev = sb.st_dev;
	    mtp->inode = sb.st_ino;
	    mtp->mode = sb.st_mode;
	    mtp->rdev = sb.st_rdev;
	    mtp->next = Lmi;
	/*
	 * Form the file system (mounted-on) device name.  Resolve any
	 * symbolic links.  Allocate space for the result and store it in
	 * the local mounts structure.
	 */
	    if (h && (v->vmt_flags & MNT_REMOTE)) {
		if (!(dn = mkstrcat(h, -1, ":", 1, fs, -1, (MALLOC_S *)NULL)))
		    goto no_space_for_mount;
	    } else {
		if (!(dn = mkstrcpy(fs, (MALLOC_S *)NULL)))
		    goto no_space_for_mount;
	    }
	    mtp->fsname = dn;
	    ln = Readlink(dn);
	    dn = (char *)NULL;
	/*
	 * Stat the file system (mounted-on) device name to get its modes.
	 * Set the modes to zero if the stat fails.  Add file system
	 * (mounted-on) device information to the local mountsstructure.
	 */
	    if (!ln || statsafely(ln, &sb))
		sb.st_mode = 0;
	    mtp->fsnmres = ln;
	    mtp->fs_mode = sb.st_mode;
	    Lmi = mtp;
        }
/*
 * Clean up and return local mount info table address.
 */
	if (dn)
	    (void) free((FREE_P *)dn);
	if (vt)
	    (void) free((FREE_P *)vt);
	Lmist = 1;
	return(Lmi);
}


/*
 * readvfs() - read vfs structure
 */

struct l_vfs *
readvfs(vn)
	struct vnode *vn;		/* vnode */
{
	struct gfs g;
	void *mp;
	char *s1, *s2;
	u_long ul;
	struct vfs v;
	struct vmount *vm;
	struct l_vfs *vp;

	if (!vn->v_vfsp)
	    return((struct l_vfs *)NULL);
	for (vp = Lvfs; vp; vp = vp->next) {
	    if ((KA_T)vn->v_vfsp == vp->addr)
		return(vp);
	}
	if (!(vp = (struct l_vfs *)malloc(sizeof(struct l_vfs)))) {
	    (void) fprintf(stderr, "%s: PID %d, no space for vfs\n",
		Pn, Lp->pid);
	    Exit(1);
	}
	vp->dir = (char *)NULL;
	vp->fsname = (char *)NULL;
/*
 * Read the vfs structure.
 */
	if (kread((KA_T)vn->v_vfsp, (char *)&v, sizeof(v))) {

vfs_exit:
	    (void) free((FREE_P *)vp);
	    return((struct l_vfs *)NULL);
	}
/*
 * Locate AIX mount information.
 */
	if (!v.vfs_gfs || kread((KA_T)v.vfs_gfs, (char *)&g, sizeof(g)))
	    goto vfs_exit;
	if (!v.vfs_mdata
	||  kread((KA_T)v.vfs_mdata + (KA_T)sizeof(u_long), (char *)&ul,
		  sizeof(ul))
	)
	    goto vfs_exit;
	if (!(mp = (void *)malloc((size_t)ul))) {
	    (void) fprintf(stderr, "%s: PID %d, no space for mount data\n",
		Pn, Lp->pid);
	    Exit(1);
	}
	if (kread((KA_T)v.vfs_mdata, (char *)mp, (int)ul)) {
	    (void) free((FREE_P *)mp);
	    goto vfs_exit;
	}
	vm = (struct vmount *)mp;
	vp->vmt_flags = vm->vmt_flags;
	vp->vmt_gfstype = vm->vmt_gfstype;

#if	AIXV>=3200
	if (vp->vmt_flags & MNT_REMOTE)
	    vp->dev = 0x80000000 | vm->vmt_vfsnumber;
	else
#endif	/* AIXV>=3200 */

#if	defined(HAS_AFS)
	    if (vm->vmt_gfstype == MNT_AFS)
		vp->dev = AFSDEV;
	    else
#endif	/* defined(HAS_AFS) */

		vp->dev = (dev_t)vm->vmt_fsid.fsid_dev;
	if ((s1 = vmt2dataptr(vm, VMT_STUB))) {
	    if (!(vp->dir = mkstrcpy(s1, (MALLOC_S *)NULL))) {

readvfs_aix1:
		(void) fprintf(stderr, "%s: PID %d, readvfs, no space\n",
		    Pn, Lp->pid);
		Exit(1);
	    }
	} else
	    vp->dir = (char *)NULL;
	s1 = vmt2dataptr(vm, VMT_HOST);
	if (!(s2 = vmt2dataptr(vm, VMT_OBJECT)) || *s1 == '\0')
	    s2 = g.gfs_name;
	if (!s1 && !s2)
	    vp->fsname = (char *)NULL;
	else {
	    if (vm->vmt_flags & MNT_REMOTE) {
		if (!(vp->fsname = mkstrcat(s1 ? s1 : "", -1, ":", 1, s2, -1,
				   (MALLOC_S *)NULL)))
		    goto readvfs_aix1;
	    } else {
		if (!(vp->fsname = mkstrcpy(s2, (MALLOC_S *)NULL)))
		    goto readvfs_aix1;
	    }
	}
	(void) free((FREE_P *)mp);
	vp->next = Lvfs;
	vp->addr = (KA_T)vn->v_vfsp;

#if	defined(HAS_AFS)
	if (!AFSVfsp && vm->vmt_gfstype == MNT_AFS)
	    AFSVfsp = (KA_T)vn->v_vfsp;
#endif	/* defined(HAS_AFS) */

	Lvfs = vp;
	return(vp);
}
