/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#if !output_h
#define output_h 1

#include "str.h"

typedef struct outp_driver outp_driver;
typedef struct outp_class outp_class;

/* FORMAT OF TAGGED STRINGS:

   Tagged strings are normal strings except:

   0. They are lengthed strings. (See str.h for possible formats.)

   1. `\0' must be represented as `\0\0\0'.

   2. `\0' introduces several sequences of the form `\0XXK', where `XX'
   is the (little-endian) length of the sequence (including `\0XX'),
   and `K' is a single-byte key.  This header is followed by a value of
   appropriate length.

   3. Possible tag types for tagged strings are TAG_* below. */

/* Tag types for tagged strings; see TODO.

   These values must not be changed. */
enum
  {
    TAG_FONT_BY_NAME,		/* Takes null-terminated font name. */
    TAG_FONT_BY_FAMILY,		/* Takes null-terminated font family. */
    TAG_FONT_BY_POSITION,	/* Takes single-byte position number. */
    TAG_COLOR,			/* Takes r,g,b each as 2-byte numbers. */
    TAG_NO_OP,			/* No arguments. */
    TAG_NEWLINE			/* Inserts a newline; no arguments. */
  };

/* A rectangle. */
typedef struct
  {
    int x1, y1;			/* Upper left. */
    int x2, y2;			/* Lower right, not part of the rectangle. */
  }
rect;

#if __GNUC__ > 1 && defined(__OPTIMIZE__)
extern inline int width (rect r) __attribute__ ((const));
extern inline int height (rect r) __attribute__ ((const));

extern inline int
width (rect r)
{
  return r.x2 - r.x1 + 1;
}

extern inline int 
height (rect r)
{
  return r.y2 - r.y1 + 1;
}
#else /* !__GNUC__ */
#define width(R) 				\
	((R).x2 - (R).x1 + 1)
#define height(R) 				\
	((R).y2 - (R).y1 + 1)
#endif /* !__GNUC__ */

/* Color descriptor. */
typedef struct
  {
    int flags;			/* 0=normal, 1=transparent (ignore r,g,b). */
    int r;			/* Red component, 0-65535. */
    int g;			/* Green component, 0-65535. */
    int b;			/* Blue component, 0-65535. */
  }
color;

/* Mount positions for the four basic fonts.  Do not change the values. */
enum
  {
    OUTP_F_R,			/* Roman font. */
    OUTP_F_I,			/* Italic font. */
    OUTP_F_B,			/* Bold font. */
    OUTP_F_BI			/* Bold-italic font. */
  };

/* Line styles.  These must match:
   som.h:SLIN_*
   ascii.c:ascii_line_*() 
   postscript.c:ps_line_*() */
enum
  {
    OUTP_L_NONE = 0,		/* No line. */
    OUTP_L_SINGLE = 1,		/* Single line. */
    OUTP_L_DOUBLE = 2,		/* Double line. */
    OUTP_L_SPECIAL = 3,		/* Special line of driver-defined style. */

    OUTP_L_COUNT		/* Number of line styles. */
  };

/* Contains a line style for each part of an intersection. */
typedef struct
  {
    int l;			/* left */
    int t;			/* top */
    int r;			/* right */
    int b;			/* bottom */
  }
outp_styles;

/* A value for a driver option. */
typedef struct
  {
    int type;			/* `a' for a string; '{' for braced. */
    a_string string;		/* Value string. */
  }
outp_value;

/* Text display options. */
enum
  {
    OUTP_T_NONE = 0,

    /* Must match tab.h:TAB_*. */
    OUTP_T_JUST_MASK = 00003,	/* Justification mask. */
    OUTP_T_JUST_RIGHT = 00000,	/* Right justification. */
    OUTP_T_JUST_LEFT = 00001,	/* Left justification. */
    OUTP_T_JUST_CENTER = 00002,	/* Center justification. */

    /* Must match tab.h:TAB_RICH. */
    OUTP_T_FANCY = 00004,	/* Allow escapes in the string. */

    OUTP_T_HORZ = 00010,	/* Horizontal size is specified. */
    OUTP_T_VERT = 00020,	/* (Max) vertical size is specified. */

    OUTP_T_0 = 00140,		/* Normal orientation. */
    OUTP_T_CC90 = 00040,	/* 90 degrees counterclockwise. */
    OUTP_T_CC180 = 00100,	/* 180 degrees counterclockwise. */
    OUTP_T_CC270 = 00140,	/* 270 degrees counterclockwise. */
    OUTP_T_C90 = 00140,		/* 90 degrees clockwise. */
    OUTP_T_C180 = 00100,	/* 180 degrees clockwise. */
    OUTP_T_C270 = 00040,	/* 270 degrees clockwise. */

    /* Internal use by drivers only. */
    OUTP_T_INTERNAL_DRAW = 01000	/* 1=Draw the text, 0=Metrics only. */
  };

/* Describes text output. */
typedef struct
  {
    /* Public. */
    int options;		/* What is specified. */
    a_string s;			/* String. */
    int h, v;			/* Horizontal, vertical size. */
    int x, y;			/* Position. */

    /* Internal use only. */
    int w, l;			/* Width, length. */
  }
outp_text;

/* Exported by som.h. */
struct som_table;

/* Defines a class of output driver. */
struct outp_class
  {
    /* Basic class information. */
    const char *name;		/* Name of this driver class. */
    int magic;			/* Driver-specific constant. */
    int special;		/* Boolean value. */

    /* Static member functions. */
    int (*open_global) (outp_class *);
    int (*close_global) (outp_class *);
    int *(*font_sizes) (outp_class *, int *n_valid_sizes);

    /* Virtual member functions. */
    int (*preopen_driver) (outp_driver *);
    void (*option) (outp_driver *, const char *key, const outp_value *value);
    int (*postopen_driver) (outp_driver *);
    int (*close_driver) (outp_driver *);

    int (*open_page) (outp_driver *);
    int (*close_page) (outp_driver *);

    /* special != 0: Used to submit tables for output. */
    void (*submit) (outp_driver *, struct som_table *);
    
    /* special != 0: Methods below need not be defined. */
    
    /* Line methods. */
    void (*line_horz) (outp_driver *, const rect *, const color *, int style);
    void (*line_vert) (outp_driver *, const rect *, const color *, int style);
    void (*line_intersection) (outp_driver *, const rect *, const color *,
			       const outp_styles *style);

    /* Drawing methods. */
    void (*box) (outp_driver *, const rect *, const color *bord,
		 const color *fill);
    void (*polyline_begin) (outp_driver *, const color *);
    void (*polyline_point) (outp_driver *, int, int);
    void (*polyline_end) (outp_driver *);

    /* Text methods. */
    void (*text_set_font_by_name) (outp_driver *, const char *s);
    void (*text_set_font_by_position) (outp_driver *, int);
    void (*text_set_font_family) (outp_driver *, const char *s);
    const char *(*text_get_font_name) (outp_driver *);
    const char *(*text_get_font_family) (outp_driver *);
    int (*text_set_size) (outp_driver *, int);
    int (*text_get_size) (outp_driver *, int *em_width);
    void (*text_metrics) (outp_driver *, outp_text *);
    void (*text_draw) (outp_driver *, outp_text *);
  };

/* Device types. */
enum
  {
    OUTP_DEV_NONE = 0,		/* None of the below. */
    OUTP_DEV_LISTING = 001,	/* Listing device. */
    OUTP_DEV_SCREEN = 002,	/* Screen device. */
    OUTP_DEV_PRINTER = 004,	/* Printer device. */
    OUTP_DEV_DISABLED = 010	/* Broken device. */
  };

/* Defines the configuration of an output driver. */
struct outp_driver
  {
    outp_class *class;		/* Driver class. */
    char *name;			/* Name of this driver. */
    int driver_open;		/* 1=driver is open, 0=driver is closed. */
    int page_open;		/* 1=page is open, 0=page is closed. */

    outp_driver *next, *prev;	/* Next, previous output driver in list. */

    int device;			/* Zero or more of OUTP_DEV_*. */
    int res, horiz, vert;	/* Device resolution. */
    int width, length;		/* Page size. */

    int cp_x, cp_y;		/* Current position. */
    int font_height;		/* Default font character height. */
    int prop_em_width;		/* Proportional font em width. */
    int fixed_width;		/* Fixed-pitch font character width. */
    int horiz_line_width[OUTP_L_COUNT];	/* Width of horizontal lines. */
    int vert_line_width[OUTP_L_COUNT];	/* Width of vertical lines. */
    int horiz_line_spacing[1 << OUTP_L_COUNT];
    int vert_line_spacing[1 << OUTP_L_COUNT];

    void *ext;			/* Private extension record. */
    void *prc;			/* Per-procedure extension record. */
  };

/* Option structure for the keyword recognizer. */
typedef struct
  {
    const char *keyword;	/* Keyword name. */
    int cat;			/* Category. */
    int subcat;			/* Subcategory. */
  }
outp_option;

/* Information structure for the keyword recognizer. */
typedef struct
  {
    char *initial;		/* Initial characters. */
    outp_option **options;	/* Search starting points. */
  }
outp_option_info;

/* A list of driver classes. */
typedef struct outp_driver_class_list
  {
    int ref_count;
    outp_class *class;
    struct outp_driver_class_list *next;
  }
outp_driver_class_list;

/* List of known output driver classes. */
extern outp_driver_class_list *outp_class_list;

/* List of configured output drivers. */
extern outp_driver *outp_driver_list;

/* Title, subtitle. */
extern char *outp_title;
extern char *outp_subtitle;

int outp_init (void);
int outp_read_devices (void);
int outp_done (void);

void outp_configure_clear (void);
void outp_configure_add (char *);
void outp_configure_macro (char *);

void outp_list_classes (void);

void outp_enable_device (int enable, int device);
outp_driver *outp_drivers (outp_driver *);

int outp_match_keyword (const char *, outp_option *,
			outp_option_info *, int *);

int outp_evaluate_dimension (char *, char **);
int outp_get_paper_size (char *, int *h, int *v);

int outp_eject_page (outp_driver *);

int outp_string_width (outp_driver *, const char *);

/* Imported from som-frnt.c. */
void som_destroy_driver (outp_driver *);

#endif /* output_h */
