/*
 * Grace - Graphics for Exploratory Data Analysis
 * 
 * Home page: http://plasma-gate.weizmann.ac.il/Grace/
 * 
 * Copyright (c) 1991-95 Paul J Turner, Portland, OR
 * Copyright (c) 1996-98 GRACE Development Team
 * 
 * Maintained by Evgeny Stambulchik <fnevgeny@plasma-gate.weizmann.ac.il>
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 *
 * utilities for graphs
 *
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>

#include "globals.h"
#include "draw.h"
#include "utils.h"
#include "protos.h"

extern char print_file[];

static void auto_ticks(int gno, int axis);

/*
 * Count the number of active sets in graph gno
 */
int nactive(int gno)
{
    int i, cnt = 0;
    for (i = 0; i < g[gno].maxplot; i++) {
	if (isactive_set(gno, i)) {
	    cnt++;
	}
    }
    return cnt;
}

int isxreversed(int gno)
{
    return (g[gno].xinvert);
}

int isyreversed(int gno)
{
    return (g[gno].yinvert);
}

int islogx(int gno)
{
    return (g[gno].xscale == SCALE_LOG);
}

int islogy(int gno)
{
    return (g[gno].yscale == SCALE_LOG);
}

char *graph_types(int it)
{
    static char s[16];

    switch (it) {
    case GRAPH_XY:
	strcpy(s, "XY");
	break;
    case GRAPH_CHART:
	strcpy(s, "Chart");
	break;
    case GRAPH_POLAR:
	strcpy(s, "Polar");
	break;
    case GRAPH_SMITH:
	strcpy(s, "Smith");
	break;
    default:
        strcpy(s, "Unknown");
	break;
   }
    return s;
}

char *scale_types(int it)
{
    static char s[16];

    switch (it) {
    case SCALE_NORMAL:
	strcpy(s, "Normal");
	break;
    case SCALE_LOG:
	strcpy(s, "Logarithmic");
	break;
    case SCALE_REC:
	strcpy(s, "Reciprocal");
	break;
    default:
        strcpy(s, "Unknown");
	break;
    }
    
    return s;
}

int get_format_index(int f)
{
    int i = 0;

    while (f != format_types[i] && format_types[i] != FORMAT_INVALID) {
	i++;
    }
    return i;
}

char *get_format_types(int f)
{
    static char s[128];

    strcpy(s, "decimal");
    switch (f) {
    case FORMAT_DECIMAL:
	strcpy(s, "decimal");
	break;
    case FORMAT_EXPONENTIAL:
	strcpy(s, "exponential");
	break;
    case FORMAT_GENERAL:
	strcpy(s, "general");
	break;
    case FORMAT_POWER:
	strcpy(s, "power");
	break;
    case FORMAT_SCIENTIFIC:
	strcpy(s, "scientific");
	break;
    case FORMAT_ENGINEERING:
	strcpy(s, "engineering");
	break;
    case FORMAT_DDMMYY:
	strcpy(s, "ddmmyy");
	break;
    case FORMAT_MMDDYY:
	strcpy(s, "mmddyy");
	break;
    case FORMAT_MMYY:
	strcpy(s, "mmyy");
	break;
    case FORMAT_MMDD:
	strcpy(s, "mmdd");
	break;
    case FORMAT_MONTHDAY:
	strcpy(s, "monthday");
	break;
    case FORMAT_DAYMONTH:
	strcpy(s, "daymonth");
	break;
    case FORMAT_MONTHS:
	strcpy(s, "months");
	break;
    case FORMAT_MONTHSY:
	strcpy(s, "monthsy");
	break;
    case FORMAT_MONTHL:
	strcpy(s, "monthl");
	break;
    case FORMAT_DAYOFWEEKS:
	strcpy(s, "dayofweeks");
	break;
    case FORMAT_DAYOFWEEKL:
	strcpy(s, "dayofweekl");
	break;
    case FORMAT_DAYOFYEAR:
	strcpy(s, "dayofyear");
	break;
    case FORMAT_HMS:
	strcpy(s, "hms");
	break;
    case FORMAT_MMDDHMS:
	strcpy(s, "mmddhms");
	break;
    case FORMAT_MMDDYYHMS:
	strcpy(s, "mmddyyhms");
	break;
    case FORMAT_DEGREESLON:
	strcpy(s, "degreeslon");
	break;
    case FORMAT_DEGREESMMLON:
	strcpy(s, "degreesmmlon");
	break;
    case FORMAT_DEGREESMMSSLON:
	strcpy(s, "degreesmmsslon");
	break;
    case FORMAT_MMSSLON:
	strcpy(s, "mmsslon");
	break;
    case FORMAT_DEGREESLAT:
	strcpy(s, "degreeslat");
	break;
    case FORMAT_DEGREESMMLAT:
	strcpy(s, "degreesmmlat");
	break;
    case FORMAT_DEGREESMMSSLAT:
	strcpy(s, "degreesmmsslat");
	break;
    case FORMAT_MMSSLAT:
	strcpy(s, "mmsslat");
	break;
    }
    return s;
}

void kill_graph(int gno)
{
    int i, j;

    if (gno == maxgraph) {
	for (i = 0; i < maxgraph; i++) {
	    for (j = 0; j < g[i].maxplot; j++) {
		killset(i, j);
	    }
	    set_default_graph(i);
	}
    } else {
	for (i = 0; i < g[gno].maxplot; i++) {
	    killset(gno, i);
	}
	set_default_graph(gno);
    }
}

void copy_graph(int from, int to)
{
    int i, j;
    plotarr *p;
    kill_graph(to);
    p = g[to].p;
    memcpy(&g[to], &g[from], sizeof(graph));

    g[to].labs.title = copy_plotstr(g[from].labs.title);
    g[to].labs.stitle = copy_plotstr(g[from].labs.stitle);

    for (j = 0; j < MAXAXES; j++) {
	g[to].t[j].label = copy_plotstr(g[from].t[j].label);
	for (i = 0; i < MAX_TICKS; i++) {
	    if (g[from].t[j].tloc[i].label != NULL) {
                g[to].t[j].tloc[i].label = copy_string(g[to].t[j].tloc[i].label,
                                                g[from].t[j].tloc[i].label);
	    }
	}
    }

    g[to].p = p;
    set_graph_active(to);	/* TODO compare maxplots */
    for (i = 0; i < g[from].maxplot; i++) {
	for (j = 0; j < MAX_SET_COLS; j++) {
	    g[to].p[i].ex[j] = NULL;
	}
	g[to].p[i].active = FALSE;
	if (isactive_set(from, i)) {
	    do_copyset(from, i, to, i);
	}
    }
}

void copy_graph_sets_only(int from, int to)
{
    int i, j;
    kill_graph(to);
    set_graph_active(to);	/* TODO compare maxplots */
    for (i = 0; i < g[from].maxplot; i++) {
	for (j = 0; j < MAX_SET_COLS; j++) {
	    g[to].p[i].ex[j] = NULL;
	}
	g[to].p[i].active = FALSE;
	if (isactive_set(from, i)) {
	    do_copyset(from, i, to, i);
	}
    }
}

void swap_graph(int from, int to)
{
    graph gtmp;
    memcpy(&gtmp, &g[from], sizeof(graph));
    memcpy(&g[from], &g[to], sizeof(graph));
    memcpy(&g[to], &gtmp, sizeof(graph));
    set_dirtystate();
}

void get_graph_box(int i, boxtype * b)
{
    memcpy(b, &boxes[i], sizeof(boxtype));
}

void get_graph_ellipse(int i, ellipsetype * b)
{
    memcpy(b, &ellip[i], sizeof(ellipsetype));
}

void get_graph_line(int i, linetype * l)
{
    memcpy(l, &lines[i], sizeof(linetype));
}

void get_graph_string(int i, plotstr * s)
{
    memcpy(s, &pstr[i], sizeof(plotstr));
}

void get_graph_framep(int gno, framep * f)
{
    memcpy(f, &g[gno].f, sizeof(framep));
}

void get_graph_world(int gno, world * w)
{
    memcpy(w, &g[gno].w, sizeof(world));
}

void get_graph_view(int gno, view * v)
{
    memcpy(v, &g[gno].v, sizeof(view));
}

void get_graph_labels(int gno, labels * labs)
{
    memcpy(labs, &g[gno].labs, sizeof(labels));
}

void get_graph_plotarr(int gno, int i, plotarr * p)
{
    memcpy(p, &g[gno].p[i], sizeof(plotarr));
}

void get_graph_tickmarks(int gno, tickmarks * t, int a)
{
/*
 *     int i;
 */
    
    memcpy(t, &g[gno].t[a], sizeof(tickmarks));
/*
 *     for (i = 0; i < MAX_TICKS; i++) {
 *         t->tloc[i].label = 
 *                     copy_string(t->tloc[i].label, g[gno].t[a].tloc[i].label);
 *     }
 */
}

void get_graph_legend(int gno, legend * leg)
{
    memcpy(leg, &g[gno].l, sizeof(legend));
}

/*
 *
 * set graph props
 *
 */

void set_graph_box(int i, boxtype * b)
{
    memcpy(&boxes[i], b, sizeof(boxtype));
}

void set_graph_line(int i, linetype * l)
{
    memcpy(&lines[i], l, sizeof(linetype));
}

void set_graph_string(int i, plotstr * s)
{
    memcpy(&pstr[i], s, sizeof(plotstr));
}

void set_graph_active(int gno)
{

    g[gno].active = TRUE;
}

void set_graph_framep(int gno, framep * f)
{
    memcpy(&g[gno].f, f, sizeof(framep));
}

void set_graph_world(int gno, world * w)
{
    memcpy(&g[gno].w, w, sizeof(world));
}

void set_graph_view(int gno, view * v)
{
    memcpy(&g[gno].v, v, sizeof(view));
}

void set_graph_viewport(int gno, double vx1, double vy1, double vx2, double vy2)
{
    g[gno].v.xv1 = vx1;
    g[gno].v.xv2 = vx2;
    g[gno].v.yv1 = vy1;
    g[gno].v.yv2 = vy2;
    
    set_dirtystate();
}

void set_graph_labels(int gno, labels * labs)
{
    memcpy(&g[gno].labs, labs, sizeof(labels));
}

void set_graph_plotarr(int gno, int i, plotarr * p)
{
    memcpy(&g[gno].p[i], p, sizeof(plotarr));
}

void set_graph_tickmarks(int gno, tickmarks * t, int a)
{
/*
 *     int i;
 */
    
    memcpy(&g[gno].t[a], t, sizeof(tickmarks));
/*
 *     for (i = 0; i < MAX_TICKS; i++) {
 *         g[gno].t[a].tloc[i].label = 
 *                     copy_string(g[gno].t[a].tloc[i].label, t->tloc[i].label);
 *     }
 */
}

void set_graph_legend(int gno, legend * leg)
{
    memcpy(&g[gno].l, leg, sizeof(legend));
}



int wipeout(void)
{
    if (!noask && is_dirtystate()) {
        if (!yesno("Abandon unsaved project?", NULL, NULL, NULL)) {
            return 1;
        }
    }
    kill_graph(maxgraph);
    do_clear_lines();
    do_clear_boxes();
    do_clear_ellipses();
    do_clear_text();
    cg = 0;
    strcpy(docname, NONAME);
    description[0] = '\0';
    print_file[0] = '\0';
    clear_dirtystate();
    return 0;
}


/* The following routines determine default axis range and tickmarks */

static void autoscale_axis_byset(int gno, int axistype, int setno);
static double nicenum(double x, int round);

#define NICE_FLOOR  0
#define NICE_CEIL   1
#define NICE_ROUND  2

int axis_type(int axis)
{
    int atype;
    
    switch (axis) {
    case ALL_AXES:
        atype = AXIS_TYPE_ANY;
        break;
    case ALL_X_AXES:
    case X_AXIS:
    case ZX_AXIS:
        atype = AXIS_TYPE_X;
        break;
    case ALL_Y_AXES:
    case Y_AXIS:
    case ZY_AXIS:
        atype = AXIS_TYPE_Y;
        break;
    default:
        atype = AXIS_TYPE_BAD;
        break;
    }
    
    return atype;
}

void autotick_axis(int gno, int axis)
{
    switch (axis) {
    case ALL_AXES:
        auto_ticks(gno, X_AXIS);
        auto_ticks(gno, ZX_AXIS);
        auto_ticks(gno, Y_AXIS);
        auto_ticks(gno, ZY_AXIS);
        break;
    case ALL_X_AXES:
        auto_ticks(gno, X_AXIS);
        auto_ticks(gno, ZX_AXIS);
        break;
    case ALL_Y_AXES:
        auto_ticks(gno, Y_AXIS);
        auto_ticks(gno, ZY_AXIS);
        break;
    default:
        auto_ticks(gno, axis);
        break;
    }
}

void autoscale_set(int gno, int setno, int axis)
{
    if (setno == ALL_SETS || isactive_set(gno, setno)) {
	autoscale_axis_byset(gno, axis_type(axis), setno);
	autotick_axis(gno, axis);
#ifndef NONE_GUI
        update_ticks(gno);
#endif
    }
}

static void round_axis_limits(double *amin, double *amax, int scale)
{
/*
 *     double extra_range;
 */
    
    if (*amin == *amax) {
        switch (sign(*amin)) {
        case 0:
            *amin = -1.0;
            *amax = +1.0;
            break;
        case 1:
            *amin /= 2.0;
            *amax *= 2.0;
            break;
        case -1:
            *amin *= 2.0;
            *amax /= 2.0;
            break;
        }
    } 
/*
 *     else {
 *         extra_range = 0.05 * fabs(*amin);
 *         *amin -= extra_range;
 *         *amax += extra_range;
 *     }
 */
    *amin = nicenum(*amin, NICE_FLOOR);
    *amax = nicenum(*amax, NICE_CEIL);
    
    if (scale == SCALE_NORMAL && sign(*amin) == sign(*amax)) {
        if ((*amax)/(*amin) > 5.0) {
            *amin = 0.0;
        } else if ((*amin)/(*amax) > 5.0) {
            *amax = 0.0;
        }
    }
}

static void autoscale_axis_byset(int gno, int axistype, int setno)
{
    world w;
    double xmax, xmin, ymax, ymin;
    int scale;

    get_graph_world(gno, &w);
    
    if (g[gno].type == GRAPH_SMITH) {
        if (axistype == AXIS_TYPE_X || axistype == AXIS_TYPE_ANY) {
            w.xg1 = -1.0;
            w.yg1 = -1.0;
        }
        if (axistype == AXIS_TYPE_Y || axistype == AXIS_TYPE_ANY) {
            w.xg2 = 1.0;
            w.yg2 = 1.0;
	}
        set_graph_world(gno, &w);
        return;
    }

    getsetminmax(gno, setno, &xmin, &xmax, &ymin, &ymax);

    if ((finite(xmin) == 0) && (xmin < 0.)) {
      xmin = -MAXNUM;
    }
    if ((finite(xmax) == 0) && (xmax > 0.)) {
      xmax = +MAXNUM;
    }
    if ((finite(ymin) == 0) && (ymin < 0.)) {
      ymin = -MAXNUM;
    }
    if ((finite(ymax) == 0) && (ymax > 0.)) {
      ymax = +MAXNUM;
    }

    if (axistype == AXIS_TYPE_X || axistype == AXIS_TYPE_ANY) {
        scale = g[gno].xscale;
        round_axis_limits(&xmin, &xmax, scale);
        w.xg1 = xmin;
        w.xg2 = xmax;
    }

    if (axistype == AXIS_TYPE_Y || axistype == AXIS_TYPE_ANY) {
        scale = g[gno].yscale;
        round_axis_limits(&ymin, &ymax, scale);
        w.yg1 = ymin;
        w.yg2 = ymax;
    }
   
/*
 *     
 *     if (g[gno].type == GRAPH_CHART) {
 * 	if (axistype == AXIS_TYPE_X || axistype == AXIS_TYPE_ANY) {
 *             w.xg1 -= (w.xg2 - w.xg1) * 0.05;
 * 	    w.xg2 += (w.xg2 - w.xg1) * 0.05;
 * 	}
 *         if (axistype == AXIS_TYPE_Y || axistype == AXIS_TYPE_ANY) {
 *             if (w.yg1 > 0.0) {
 * 	        w.yg1 = 0.0;
 * 	    }
 *         }
 *     }
 */

    set_graph_world(gno, &w);
}

static void auto_ticks(int gno, int axis)
{
    tickmarks t;
    double range, d, tmpmax, tmpmin;
    int axis_scale;

    get_graph_tickmarks(gno, &t, axis);

    if (is_xaxis(axis)) {
        tmpmin = g[gno].w.xg1;
        tmpmax = g[gno].w.xg2;
        axis_scale = g[gno].xscale;
    } else {
        tmpmin = g[gno].w.yg1;
        tmpmax = g[gno].w.yg2;
        axis_scale = g[gno].yscale;
    }

    if (axis_scale == SCALE_LOG) {
	if (t.tmajor <= 1.0) {
            t.tmajor = 10.0;
        }
        tmpmax = log10(tmpmax)/log10(t.tmajor);
	tmpmin = log10(tmpmin)/log10(t.tmajor);
    } else if (t.tmajor <= 0.0) {
        t.tmajor = 1.0;
    }
    if (t.nminor < 0) {
	t.nminor = 1;
    }
    
    range = tmpmax - tmpmin;
    if (axis_scale != SCALE_LOG) {
        d = nicenum(range/(t.t_autonum - 1), NICE_ROUND);
	t.tmajor = d;
    } else {
        d = ceil(range/(t.t_autonum - 1));
	t.tmajor = pow(t.tmajor, d);
    }
    
    set_graph_tickmarks(gno, &t, axis);
}

/*
 * nicenum: find a "nice" number approximately equal to x
 */

static double nicenum(double x, int round)
{
    int xsign;
    double f, y, exp, nnres, smallx;
    double maxf, maxexp;

    if (x == 0.0) {
        return(0.0);
    }

    maxf  = MAXNUM/pow(10.0,floor(log10(MAXNUM)));
    maxexp = floor(log10(MAXNUM));

    xsign = sign(x);
    x = fabs(x);
    exp = floor(log10(x));
    smallx = pow(10.0, exp);
    f = x/smallx;	/* fraction between 1 and 10 */
    if ((round == NICE_FLOOR && xsign == +1) ||
        (round == NICE_CEIL  && xsign == -1)) {
	if (f < 2.0)
	    y = 1.;
	else if (f < 5.0)
	    y = 2.;
	else if (f < 10.0)
	    y = 5.;
	else
	    y = 10.;
    } else if ((round == NICE_FLOOR && xsign == -1) ||
               (round == NICE_CEIL  && xsign == +1)) {
        if (f <= 1.)
            y = 1.;
        else if (f <= 2.)
            y = 2.;
        else if (f <= 5.)
            y = 5.;
        else
            y = 10.;
    } else {
	if (f < 1.5)
	    y = 1.;
	else if (f < 3.)
	    y = 2.;
	else if (f < 7.)
	    y = 5.;
	else
	    y = 10.;
    }
    if (exp==maxexp)
       nnres = (y > maxf) ? xsign*floor(maxf)*smallx : xsign*y*smallx;
    else
       nnres = xsign*y*smallx;
    return (nnres);
}
