/*--------------------------------------------------------------------------
  ----- File:        t1load.c 
  ----- Author:      Rainer Menzner (rmz@neuroinformatik.ruhr-uni-bochum.de)
  ----- Date:        06/15/1998
  ----- Description: This file is part of the t1-library. It contains
                     functions for loading fonts  and for managing size
		     dependent data.
  ----- Copyright:   t1lib is copyrighted (c) Rainer Menzner, 1996-1998. 
                     As of version 0.5, t1lib is distributed under the
		     GNU General Public Library Lincense. The
		     conditions can be found in the files LICENSE and
		     LGPL, which should reside in the toplevel
		     directory of the distribution.  Please note that 
		     there are parts of t1lib that are subject to
		     other licenses:
		     The parseAFM-package is copyrighted by Adobe Systems
		     Inc.
		     The type1 rasterizer is copyrighted by IBM and the
		     X11-consortium.
  ----- Warranties:  Of course, there's NO WARRANTY OF ANY KIND :-)
  ----- Credits:     I want to thank IBM and the X11-consortium for making
                     their rasterizer freely available.
		     Also thanks to Piet Tutelaers for his ps2pk, from
		     which I took the rasterizer sources in a format
		     independent from X11.
                     Thanks to all people who make free software living!
--------------------------------------------------------------------------*/
  
#define T1LOAD_C

#define ANSI_REALLOC_VM  

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "../type1/ffilest.h"
#include "../type1/types.h"
#include "parseAFM.h" 
#include "../type1/objects.h"
#include "../type1/spaces.h"
#include "../type1/util.h"
#include "../type1/fontfcn.h"
#include "../type1/blues.h"
#include "../type1/regions.h"


#include "t1types.h"
#include "t1extern.h"
#include "t1load.h"
#include "t1env.h"
#include "t1base.h"
#include "t1finfo.h"
#include "t1afmtool.h"


extern char *vm_base; /* from fontfcn.c in initfont()! */
extern char *vm_used; /* from fontfcn.c in fontfcnA()! */

int fontfcnA( char *env, int *mode, psfont *Font_Ptr);



/* T1_LoadFont(FontID): Loads a Type1 font into memory and allocates all
   memory, necessary for this. */

int T1_LoadFont( int FontID)
{
  int i, j, k;
  char *FileName, *FileNamePath;
  int mode;  /* This is used by the type1-library for error reporting */   
  char *charname;
  
  /* The following vars are used for reallocation of VM */
  long tmp_size;
  float ascender;
#ifdef ANSI_REALLOC_VM
  unsigned long shift;
  unsigned long ldummy;
  char *tmp_ptr;
#endif

  extern psobj *StdEncArrayP;       /* For checking of a fonts encoding */

  /* These are for constructing the kerning lookup table: */
  PairKernData *pkd;
  METRICS_ENTRY *kern_tbl;
  int char1, char2;
  
  
  if (CheckForInit()){
    T1_errno=T1ERR_OP_NOT_PERMITTED;
    return(-1);
  }
  

  i=CheckForFontID(FontID);
  if (i==1)
    return(0);      /* Font already loaded */
  if (i==-1){
    T1_errno=T1ERR_INVALID_FONTID;
    return(-1);     /* illegal FontID */
  }
  
  /* Allocate memory for ps_font structure: */
  if ((pFontBase->pFontArray[FontID].pType1Data=(psfont *)malloc(sizeof(psfont)))==NULL){
    sprintf( err_warn_msg_buf,
	     "Failed to allocate memory for psfont-struct (FontID=%d)", FontID);
    print_msg("T1_LoadFont()", err_warn_msg_buf);
    T1_errno=T1ERR_ALLOC_MEM;
    return(-1);
  }

  /* Check for valid filename */
  if ((FileName=T1_GetFontFileName(FontID))==NULL){
    sprintf( err_warn_msg_buf, "No font file name for font %d",
	     FontID);
    print_msg( "T1_LoadFont()", err_warn_msg_buf);
    return(-1);
  }
  
  /* Fetch the full path of type1 font file */
  if ((FileNamePath=Env_GetCompletePath( FileName,
					 T1_PFAB_ptr))==NULL){
    sprintf( err_warn_msg_buf, "Couldn't locate font file for font %d in %s",
	     FontID, T1_PFAB_ptr);
    print_msg( "T1_LoadFont()", err_warn_msg_buf);
    T1_errno=T1ERR_FILE_OPEN_ERR;
    return(-1);
  }
  
  /* And load all PostScript information into memory */
  if (fontfcnA( FileNamePath, &mode,
		pFontBase->pFontArray[FontID].pType1Data) == FALSE){
    sprintf( err_warn_msg_buf,
	     "Loading font with ID = %d failed! (mode = %d)",
	    FontID, mode);
    print_msg( "T1_LoadFont()", err_warn_msg_buf);
    free(FileNamePath);
    pFontBase->pFontArray[FontID].pType1Data=NULL;
    T1_errno=mode;
    return(-1);
  }
  free(FileNamePath);

  
  /* Store the base address of virtual memory and realloc in order not
     to waste too much memory: */
  pFontBase->pFontArray[FontID].vm_base=vm_base; 
#ifdef ANSI_REALLOC_VM
  /* We first get the size of pointers on the current system */
  /* Get size of VM, ... */
  tmp_size=((unsigned long)vm_used - (unsigned long)vm_base); 
  /* ... realloc to that size ... */
  tmp_ptr=(char *)realloc(vm_base,  tmp_size); 
  /* ... and shift all pointers refering to that area */
  if (tmp_ptr > vm_base){
    shift= (unsigned long)tmp_ptr - (unsigned long)vm_base;
    sprintf( err_warn_msg_buf,
	     "Old VM at 0x%lX, new VM at 0x%lX, shifting up by %lu",
	     (unsigned long)vm_base, (unsigned long)tmp_ptr, tmp_size);
    T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
    
    /* We start by shifting the topmost pointers: */
    pFontBase->pFontArray[FontID].vm_base=tmp_ptr;
    
    ldummy=(long)(pFontBase->pFontArray[FontID].pType1Data->vm_start);
    ldummy +=shift;
    pFontBase->pFontArray[FontID].pType1Data->vm_start=(char *)ldummy;
    
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP;
    ldummy +=shift;
    pFontBase->pFontArray[FontID].pType1Data->CharStringsP=(psdict *)ldummy;
    
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Private;
    ldummy +=shift;
    pFontBase->pFontArray[FontID].pType1Data->Private=(psdict *)ldummy;
    
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP;
    ldummy +=shift;
    pFontBase->pFontArray[FontID].pType1Data->fontInfoP=(psdict *)ldummy;
    
    ldummy=(long)(pFontBase->pFontArray[FontID].pType1Data->BluesP);
    ldummy +=shift;
    pFontBase->pFontArray[FontID].pType1Data->BluesP=(struct blues_struct *)ldummy;
    
    /* We now have to care for correcting all pointers which are in the VM
       and refer to some place in the VM! Note: Instead of selecting the
       appropriate pointer-elements of the union we simply shift the
       unspecified pointer "valueP".
       Note: The filename entry does not need to be modified since it does not
       need to be shifted since it points to memory managed by t1lib.
       */
    /* FontInfo-dictionary: All name-pointers and the pointers to all array
       types have to be shifted: */
    i=pFontBase->pFontArray[FontID].pType1Data->fontInfoP[0].key.len;
    for (j=1; j<=i; j++){
      if ((pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_ARRAY) ||
	  (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_STRING) ||
	  (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_NAME) ||
	  (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_FILE)){
	ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP;
	ldummy +=shift;
	pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP=(char *)ldummy;
      }
      /* The encoding needs special treatment: */
      if (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_ENCODING){
	/* If a builtin encoding is used, it is sufficient to shift the pointer
	   to the Encoding since the character-namestrings of builtin encodings
	   are static and thus located on the heap.
	   For font-specific encoding, character-namestrings reside in VM and
	   thus each entry has to be shifted. 
	   Caution: We still have to shift the builtin encoding-pointer, since
	   they also point to are located in VM: */
	ldummy=(long)StdEncArrayP;
	ldummy +=shift;
	StdEncArrayP=(psobj *)ldummy;
	ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP;
	ldummy +=shift;
	pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP=(char *)ldummy;
	if (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.arrayP
	    == StdEncArrayP){ /* Font uses builtin standard encoding */
	  ;
	} 
	else{ /* Font-specific encoding */ 
	  for (k=0; k<256; k++){
	    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.arrayP[k].data.arrayP;
	    ldummy +=shift;
	    pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.arrayP[k].data.arrayP=(struct ps_obj *)ldummy;
	  }
	}
      } /* end of encoding-handling */
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].key.data.valueP;
      ldummy +=shift;
      pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].key.data.valueP=(char *)ldummy;
    } /* fontinfo-dict done */
    
    /* Private-dictionary: All name-pointers and the pointers to all array
       types have to be shifted: */
    i=pFontBase->pFontArray[FontID].pType1Data->Private[0].key.len;
    for (j=1; j<=i; j++){
      if ((pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_ARRAY) ||
	  (pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_STRING) ||
	  (pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_NAME) ||
	  (pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_FILE)){
	ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Private[j].value.data.valueP;
	ldummy +=shift;
	pFontBase->pFontArray[FontID].pType1Data->Private[j].value.data.valueP=(char *)ldummy;
      }
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Private[j].key.data.valueP;
      ldummy +=shift;
      pFontBase->pFontArray[FontID].pType1Data->Private[j].key.data.valueP=(char *)ldummy;
    }
    
    /* BluesP: The entry "next" is the only pointer in blues_struct. Although it is
       not used anywhere we should shift it for correctness reasons (in case its not
       NULL)! */
    if (pFontBase->pFontArray[FontID].pType1Data->BluesP->next != NULL){
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->BluesP->next;
      ldummy +=shift;
      pFontBase->pFontArray[FontID].pType1Data->BluesP->next=(struct blues_struct *)ldummy;
    }
    
    /* The CharStrings-dictionary: Every namepointer and its corresponding
       charstring has to be shifted: */
    i=pFontBase->pFontArray[FontID].pType1Data->CharStringsP[0].key.len;
    for (j=1; j<=i; j++){
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].value.data.valueP;
      ldummy +=shift;
      pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].value.data.valueP=(char *)ldummy;
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].key.data.valueP;
      ldummy +=shift;
      pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].key.data.valueP=(char *)ldummy;
    }
    
    /* The Subroutines have also to be reorganized: */
    i=pFontBase->pFontArray[FontID].pType1Data->Subrs.len;
    /* First, shift pointer to array-start and after that the pointers to
       each command string: */
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP;
    ldummy +=shift;
    pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP=(struct ps_obj *)ldummy;
    for (j=1; j<=i; j++) {
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP[j].data.valueP;
      ldummy +=shift;
      pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP[j].data.valueP=(char *)ldummy;
    }
  } /* end of if( tmp_ptr > vm_base ) */
  else if ( vm_base > tmp_ptr){
    shift= (unsigned long)vm_base - (unsigned long)tmp_ptr;
    sprintf( err_warn_msg_buf,
	     "Old VM at 0x%lX, new VM at 0x%lX, shifting down by %lu",
	     (unsigned long)vm_base, (unsigned long)tmp_ptr, tmp_size);
    T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
    
    /* We start by shifting the topmost pointers: */
    pFontBase->pFontArray[FontID].vm_base=tmp_ptr;
    
    ldummy=(long)(pFontBase->pFontArray[FontID].pType1Data->vm_start);
    ldummy -=shift;
    pFontBase->pFontArray[FontID].pType1Data->vm_start=(char *)ldummy;
    
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP;
    ldummy -=shift;
    pFontBase->pFontArray[FontID].pType1Data->CharStringsP=(psdict *)ldummy;
    
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Private;
    ldummy -=shift;
    pFontBase->pFontArray[FontID].pType1Data->Private=(psdict *)ldummy;
    
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP;
    ldummy -=shift;
    pFontBase->pFontArray[FontID].pType1Data->fontInfoP=(psdict *)ldummy;
    
    ldummy=(long)(pFontBase->pFontArray[FontID].pType1Data->BluesP);
    ldummy -=shift;
    pFontBase->pFontArray[FontID].pType1Data->BluesP=(struct blues_struct *)ldummy;
    
    /* We now have to care for correcting all pointers which are in the VM
       and refer to some place in the VM! Note: Instead of selecting the
       appropriate pointer-elements of the union we simply shift the
       unspecified pointer "valueP".
       Note: The filename entry does not need to be modified since it does not
       need to be shifted since it points to memory managed by t1lib.
       */
    /* FontInfo-dictionary: All name-pointers and the pointers to all array
       types have to be shifted: */
    i=pFontBase->pFontArray[FontID].pType1Data->fontInfoP[0].key.len;
    for (j=1; j<=i; j++){
      if ((pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_ARRAY) ||
	  (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_STRING) ||
	  (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_NAME) ||
	  (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_FILE)){
	ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP;
	ldummy -=shift;
	pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP=(char *)ldummy;
      }
      /* The encoding needs special treatment: */
      if (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.type==OBJ_ENCODING){
	/* If a builtin encoding is used, it is sufficient to shift the pointer
	   to the Encoding since the character-namestrings of builtin encodings
	   are static and thus located on the heap.
	   For font-specific encoding, character-namestrings reside in VM and
	   thus each entry has to be shifted. 
	   Caution: We still have to shift the builtin encoding-pointer, since
	   they also point to are located in VM: */
	ldummy=(long)StdEncArrayP;
	ldummy -=shift;
	StdEncArrayP=(psobj *)ldummy;
	ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP;
	ldummy -=shift;
	pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.valueP=(char *)ldummy;
	if (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.arrayP
	     == StdEncArrayP){ /* Font uses builtin encoding */
	  ;
	} 
	else{ /* Font-specific encoding */ 
	  for (k=0; k<256; k++){
	    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.arrayP[k].data.arrayP;
	    ldummy -=shift;
	    pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].value.data.arrayP[k].data.arrayP=(struct ps_obj *)ldummy;
	  }
	}
      } /* end of encoding-handling */
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].key.data.valueP;
      ldummy -=shift;
      pFontBase->pFontArray[FontID].pType1Data->fontInfoP[j].key.data.valueP=(char *)ldummy;
    } /* fontinfo-dict done */
    
    /* Private-dictionary: All name-pointers and the pointers to all array
       types have to be shifted: */
    i=pFontBase->pFontArray[FontID].pType1Data->Private[0].key.len;
    for (j=1; j<=i; j++){
      if ((pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_ARRAY) ||
	  (pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_STRING) ||
	  (pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_NAME) ||
	  (pFontBase->pFontArray[FontID].pType1Data->Private[j].value.type==OBJ_FILE)){
	ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Private[j].value.data.valueP;
	ldummy -=shift;
	pFontBase->pFontArray[FontID].pType1Data->Private[j].value.data.valueP=(char *)ldummy;
      }
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Private[j].key.data.valueP;
      ldummy -=shift;
      pFontBase->pFontArray[FontID].pType1Data->Private[j].key.data.valueP=(char *)ldummy;
    }
    
    /* BluesP: The entry "next" is the only pointer in blues_struct. Although it is
       not used anywhere we should shift it for correctness reasons (in case its not
       NULL)! */
    if (pFontBase->pFontArray[FontID].pType1Data->BluesP->next != NULL){
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->BluesP->next;
      ldummy -=shift;
      pFontBase->pFontArray[FontID].pType1Data->BluesP->next=(struct blues_struct *)ldummy;
    }
    
    /* The CharStrings-dictionary: Every namepointer and its corresponding
       charstring has to be shifted: */
    i=pFontBase->pFontArray[FontID].pType1Data->CharStringsP[0].key.len;
    for (j=1; j<=i; j++){
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].value.data.valueP;
      ldummy -=shift;
      pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].value.data.valueP=(char *)ldummy;
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].key.data.valueP;
      ldummy -=shift;
      pFontBase->pFontArray[FontID].pType1Data->CharStringsP[j].key.data.valueP=(char *)ldummy;
    }
    
    /* The Subroutines have also to be reorganized: */
    i=pFontBase->pFontArray[FontID].pType1Data->Subrs.len;
    /* First, shift pointer to array-start and after that the pointers to
       each command string: */
    ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP;
    ldummy -=shift;
    pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP=(struct ps_obj *)ldummy;
    for (j=1; j<=i; j++) {
      ldummy=(long)pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP[j].data.valueP;
      ldummy -=shift;
      pFontBase->pFontArray[FontID].pType1Data->Subrs.data.arrayP[j].data.valueP=(char *)ldummy;
    }
  } /* end of if( vm_base > tmp_ptr ) */
  else{ /* VM addess has not changed during reallocation */
    sprintf( err_warn_msg_buf,
	     "Old VM and new VM at 0x%lX, no pointer-shifting",
	     (unsigned long)vm_base);
    T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  }
#endif
    
  /* Generate a message how much VM the current font consumes */
  sprintf( err_warn_msg_buf,
	   "VM for Font %d: %d bytes", FontID, (int) tmp_size);
  T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_STATISTIC);

  
  /* Set the matrix for common transformations to "no transformations" */
  pFontBase->pFontArray[FontID].FontTransform[0]=1.0;
  pFontBase->pFontArray[FontID].FontTransform[1]=0.0;
  pFontBase->pFontArray[FontID].FontTransform[2]=0.0;
  pFontBase->pFontArray[FontID].FontTransform[3]=1.0;

  /* Now, that the font has been loaded into memory, try to find the
     FontMatrix in the font info dictionary. If it exists, load it into
     our local fontmatrix, otherwise use a default matrix which scales to
     1/1000 (since font outlines  are defined in a 1000 point space)
     and does no further transformations. */
  if (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[FONTMATRIX].value.data.arrayP == NULL){
    pFontBase->pFontArray[FontID].FontMatrix[0]=0.001;
    pFontBase->pFontArray[FontID].FontMatrix[1]=0.0;
    pFontBase->pFontArray[FontID].FontMatrix[2]=0.0;
    pFontBase->pFontArray[FontID].FontMatrix[3]=0.001;
  }
  else{
    pFontBase->pFontArray[FontID].FontMatrix[0]= (double)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[FONTMATRIX].value.data.arrayP[0].data.real;
    pFontBase->pFontArray[FontID].FontMatrix[1]= (double)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[FONTMATRIX].value.data.arrayP[1].data.real;
    pFontBase->pFontArray[FontID].FontMatrix[2]= (double)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[FONTMATRIX].value.data.arrayP[2].data.real;
    pFontBase->pFontArray[FontID].FontMatrix[3]= (double)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[FONTMATRIX].value.data.arrayP[3].data.real;
  }

  /* Set the default values for transformation: */
  pFontBase->pFontArray[FontID].slant=0.0;
  pFontBase->pFontArray[FontID].extend=1.0;

  
  /* Now try to load afm-structures from corresponding .afm-file. */
  if ((i=openFontMetricsFile( FontID))){
    /* Try a fallback function: */
    if ((i=openFontMetricsFileSloppy( FontID))){
      sprintf( err_warn_msg_buf,
	       "Alert: Error (%d) sloppy-processing afm-file for Font %d!",
	       i ,FontID);
      T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_STATISTIC);
      if ((pFontBase->pFontArray[FontID].pAFMData=
	   T1_GenerateAFMFallbackInfo(FontID))==NULL){
	sprintf( err_warn_msg_buf,
		 "Ultimately failed to generate metrics information Font %d!",
		 FontID);
	T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_WARNING);
	T1_PrintLog( "T1_LoadFont()", "Segmentation fault might result",T1LOG_WARNING);
      }
      else{
	pFontBase->pFontArray[FontID].info_flags |=AFM_SELFGEN_SUCCESS;
	T1_PrintLog( "T1_LoadFont()",
		     "Generating AFM-information from fontfile successful!",
		     T1LOG_STATISTIC);
      }
    }
    else{
      pFontBase->pFontArray[FontID].info_flags |=AFM_SLOPPY_SUCCESS;
      sprintf( err_warn_msg_buf,
	       "Alert: Limited afm-information for Font %d",FontID);
      T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_STATISTIC);
    }
  }
  else
    pFontBase->pFontArray[FontID].info_flags |=AFM_SUCCESS;
  
  /* Now, set Encodingvector entry to default if the font's
     internal encoding is "StandardEncoding".
     */
  if (pFontBase->pFontArray[FontID].pType1Data->fontInfoP[ENCODING].value.data.arrayP
      == StdEncArrayP){
    pFontBase->pFontArray[FontID].info_flags |=USES_STANDARD_ENCODING;
    pFontBase->pFontArray[FontID].pFontEnc=pFontBase->default_enc;
    sprintf( err_warn_msg_buf,
	     "Font %d reencoded to default",FontID);
    T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  }
  else {
    sprintf( err_warn_msg_buf,
	     "Font %d not reencoded to default",FontID);
    T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  }

  
  /* If AFM-Info available we try to speed up some things: */
  if (pFontBase->pFontArray[FontID].pAFMData != NULL){
    /* We have to fill the array that maps the current encodings' indices to the
       indices used in afm file */
    if ((pFontBase->pFontArray[FontID].pEncMap=
	 (int *)calloc(256,sizeof(int)))==NULL){
      sprintf( err_warn_msg_buf, "Error allocating memory for encoding map (FontID=%d)",
	       FontID);
      T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf,
		 T1LOG_WARNING);
      T1_errno=T1ERR_ALLOC_MEM;
      return(-1);
    }
    for (i=0; i<256; i++)
      pFontBase->pFontArray[FontID].pEncMap[i]=-1;
    for (i=0; i<256; i++){
      charname=T1_GetCharName( FontID, i);
      for ( j=0; j<pFontBase->pFontArray[FontID].pAFMData->numOfChars; j++){
	if (strcmp( charname,
		    pFontBase->pFontArray[FontID].pAFMData->cmi[j].name)==0){
	  pFontBase->pFontArray[FontID].pEncMap[i]=j;
	}
      }
    }
    
    /* We now introduce an "index map" which gives quickly access to what metric 
       features a character provides. */
    if ((pFontBase->pFontArray[FontID].pIndexMap= 
	 (int *)malloc(256*sizeof(int)))==NULL){ 
      sprintf( err_warn_msg_buf, "Error allocating memory for index map (FontID=%d)",
	       FontID);
      T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf,
		 T1LOG_WARNING);
      T1_errno=T1ERR_ALLOC_MEM;
      return(-1); 
    }
    for (i=0; i<256; i++){
      pFontBase->pFontArray[FontID].pIndexMap[i]=-1;
    }
    
    /* We now create an encoding-specific kerning table which will speed up
       looking for kerning pairs! */
    /* First, get number of defined kerning pairs: */
    k=pFontBase->pFontArray[FontID].pAFMData->numOfPairs;
    if (k>0){ /* i.e., there are any pairs */ 
      if ((pFontBase->pFontArray[FontID].pKernMap=
	   (METRICS_ENTRY *)malloc( k*sizeof( METRICS_ENTRY)))==NULL){
	sprintf( err_warn_msg_buf, "Error allocating memory for metrics map (FontID=%d)",
		 FontID);
	T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf,
		     T1LOG_WARNING);
	T1_errno=T1ERR_ALLOC_MEM;
	return(-1);
      }
      kern_tbl=pFontBase->pFontArray[FontID].pKernMap;
      pkd=pFontBase->pFontArray[FontID].pAFMData->pkd;
      j=0;
      for ( i=0; i<k; i++){
	if ((char1=T1_GetEncodingIndex( FontID, pkd[i].name1))==-1){
	  /* pair is not relevant in current encoding */
	  continue;
	}
	if ((char2=T1_GetEncodingIndex( FontID, pkd[i].name2))==-1){
	  /* pair is not relevant in current encoding */
	  continue;
	}

	/* Since we get here we have a relevant pair -->
	   Put char1 in higher byte and char2 in LSB:
	   */
	kern_tbl[j].chars=(char1 << 8) | char2;

	/* We save the index into metrics array */ 
	if ( pFontBase->pFontArray[FontID].pIndexMap[char1] == -1 )
	  pFontBase->pFontArray[FontID].pIndexMap[char1]=j;

	/* We only make use of horizontal kerning */
	kern_tbl[j].hkern=pkd[i].xamt;
	j++;
	/* Set the kerning flag for this char1: */
	pFontBase->pFontArray[FontID].pIndexMap[char1] |=KERN_FLAG;
      }
      /* Reset the remaining of the table in case there were
	 irrelevant pairs: */
      for ( ; j<k; j++){
	kern_tbl[j].chars=0;
	kern_tbl[j].hkern=0;
      }
    }
    else
      pFontBase->pFontArray[FontID].pKernMap=NULL;
  } /* End of "if (AFM-info ..)" */
  
  
  /* We have just loaded a physical font into memory, thus .... */
  pFontBase->pFontArray[FontID].physical=1;

  /* Set reference-counter to 1: */
  pFontBase->pFontArray[FontID].refcount=1;

  /* Get the index into encoding vector where the space character is
     found. If not encoded, set space_position to -1. */
  pFontBase->pFontArray[FontID].space_position=-1;
  i=0;
  if (pFontBase->pFontArray[FontID].pFontEnc){ /* external default encoding */
    while (i<256){
      if (strcmp( (char *)pFontBase->pFontArray[FontID].pFontEnc[i],
		  "space")==0){
	/* space found at position i: */
	pFontBase->pFontArray[FontID].space_position=i;
	break;
      }
      i++;
    }
  }
  else{ /* internal encoding */
    while (i<256){
      if (strcmp( (char *)pFontBase->pFontArray[FontID].pType1Data->fontInfoP[ENCODING].value.data.arrayP[i].data.arrayP,
		  "space")==0){
	/* space found at position i: */
	pFontBase->pFontArray[FontID].space_position=i;
	break;
      }
      i++;
    }
  }
  

  /* Set the lining rule parameters to default values */
  pFontBase->pFontArray[FontID].UndrLnPos=
    pFontBase->pFontArray[FontID].pType1Data->fontInfoP[UNDERLINEPOSITION].value.data.real;
  pFontBase->pFontArray[FontID].UndrLnThick=
    pFontBase->pFontArray[FontID].pType1Data->fontInfoP[UNDERLINETHICKNESS].value.data.real;
  /* We have to guess a value for the typographic ascender. Since this
     dimension is not part of the Type 1 font specification, this value
     should  explicitly be set later by the user! */
  ascender=(float) T1_GetCharBBox( FontID, T1_GetEncodingIndex( FontID, "d")).ury;
  pFontBase->pFontArray[FontID].OvrLnPos=ascender
    + (float) abs( (double)pFontBase->pFontArray[FontID].UndrLnPos);
  pFontBase->pFontArray[FontID].OvrStrkPos=ascender / 2.0;
  pFontBase->pFontArray[FontID].OvrLnThick=pFontBase->pFontArray[FontID].UndrLnThick;
  pFontBase->pFontArray[FontID].OvrStrkThick=pFontBase->pFontArray[FontID].UndrLnThick;
  
  
  /* Finally, set the font size dependencies pointer to NULL since we can
     assume, that at load time of a font, no size specific data of this
     font is available.
     */
  
  pFontBase->pFontArray[FontID].pFontSizeDeps=NULL;

  /* If wanted, some debugging information is put into logfile */
  sprintf( err_warn_msg_buf, "Pointer vm_base: 0x%lX",
	   (long)pFontBase->pFontArray[FontID].vm_base);
  T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  sprintf( err_warn_msg_buf, "Pointer vm_start: 0x%lX",
	   (long)pFontBase->pFontArray[FontID].pType1Data->vm_start);
  T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  sprintf( err_warn_msg_buf, "Pointer CharStringsP: 0x%lX",
	   (long)pFontBase->pFontArray[FontID].pType1Data->CharStringsP);
  T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  sprintf( err_warn_msg_buf, "Pointer Private: 0x%lX",
	   (long)pFontBase->pFontArray[FontID].pType1Data->Private);
  T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);
  sprintf( err_warn_msg_buf, "Pointer fontInfoP: 0x%lX",
	   (long)pFontBase->pFontArray[FontID].pType1Data->fontInfoP);
  T1_PrintLog( "T1_LoadFont()", err_warn_msg_buf, T1LOG_DEBUG);

  return(0);
}
  

/* openFontMetricsFile( FontID): Gets the fontfilename corresponding to
   FontID, opens the corresponding afm-file and fills the data structures.
   return-value is the value returned by the T1lib_parseFile() function: */

int openFontMetricsFile( int FontID)
{
  char *FontFileName;
  char *AFMFileName;
  char *AFMFileNamePath;
  
  int i;
  FILE *metricsfile;
  

  FontFileName=T1_GetFontFileName( FontID);
  i=strlen(FontFileName);
  AFMFileName=(char *)malloc( i+1);
  strcpy( AFMFileName, FontFileName);
  /* We now change the extension pfa or pfb to afm: */
  AFMFileName[i-3]='a';
  AFMFileName[i-2]='f';
  AFMFileName[i-1]='m';

  /* Get full path of the afm file */
  AFMFileNamePath=Env_GetCompletePath( AFMFileName, T1_AFM_ptr);
  free( AFMFileName);
  
  /* open afm-file: */
  if ((metricsfile=fopen(AFMFileNamePath,"r"))==NULL){
    if (AFMFileNamePath!=NULL)
      free(AFMFileNamePath);
    return(-4);
  }
  free(AFMFileNamePath);
  
  /* Call procedure to read afm-file and store the data formatted.
     Flags used here: P_M  All Metrics Information
                      P_P  Pair Kerning Information
     The P_G flag to get global font information should not be used
     if not absolutely needed. When parsing an unknown keyword, which
     may be harmless, the T1lib_parseFile function returns the error code
     -1 (parseError). On the other hand, all other really relevant
     data may habe been parsed and stored correctly. In such a case,
     There's no way to make a serious decision whether an error has
     occured or not.
     */
  i=T1lib_parseFile( (FILE *) metricsfile,
		     (FontInfo **) &(FontBase.pFontArray[FontID].pAFMData),
		     P_M | P_P );
  fclose(metricsfile);
  return(i);
}



/* openFontMetricsFileSloppy( FontID): Gets the fontfilename corresponding to
   FontID, opens the corresponding afm-file and tries to fill the data
   structures only with metric data. This can be considered a fallback!
   */
int openFontMetricsFileSloppy( int FontID)
{
  char *FontFileName;
  char *AFMFileName;
  char *AFMFileNamePath;
  int i;
  FILE *metricsfile;
  

  FontFileName=T1_GetFontFileName( FontID);
  i=strlen(FontFileName);
  AFMFileName=(char *)malloc( i+1);
  strcpy( AFMFileName, FontFileName);
  /* We now change the extension pfa or pfb to afm: */
  AFMFileName[i-3]='a';
  AFMFileName[i-2]='f';
  AFMFileName[i-1]='m';

  /* Get full path of the afm file */
  AFMFileNamePath=Env_GetCompletePath( AFMFileName, T1_AFM_ptr);
  free( AFMFileName);
  
  /* open afm-file: */
  if ((metricsfile=fopen(AFMFileNamePath,"r"))==NULL){
    /* patch #1 */
    if (AFMFileNamePath!=NULL)
      free(AFMFileNamePath);
    return(-4);
  }
  free(AFMFileNamePath);
  
  /* Call procedure to read afm-file and store the data formatted.
     Flags used here: P_M  All Metrics Information
     */
  i=T1lib_parseFile( (FILE *) metricsfile,
	       (FontInfo **) &(FontBase.pFontArray[FontID].pAFMData),
	       P_M );
  fclose(metricsfile);
  return(i);
}




/* CreateNewFontSize( FontID, size): Create a new size "size" of font
   "FontID" and allocate all data necessary for this. The data
   structure is connected to the linked list of FontSizeDeps for this
   font. Returns a pointer to the newly created FontSizeDeps-struct
   if all went correct and NULL otherwise.
   Since of version 0.3 a member antialias has been added to the
   FONTSIZEDEPS structure!
   */
FONTSIZEDEPS *CreateNewFontSize( int FontID, float size, int aa)
{

  FONTSIZEDEPS *pFontSizeDeps, *pPrev;
  

  /* First, get to the last font size in the linked list for this font.
     The following routine returns the address of the last struct in the
     linked list of FONTSIZEDEPS or NULL if none exists. */
  pFontSizeDeps=GetLastFontSize( FontID);
  pPrev=pFontSizeDeps;
  
  
  if (pFontSizeDeps==NULL){
    /* Allocate memory for first FontSizeDeps-structure: */
    if ((pFontBase->pFontArray[FontID].pFontSizeDeps=(FONTSIZEDEPS *)malloc(sizeof(FONTSIZEDEPS)))==NULL){
      T1_errno=T1ERR_ALLOC_MEM;
      return(NULL);
    }
    pFontSizeDeps=pFontBase->pFontArray[FontID].pFontSizeDeps;
  }
  else{
    /* A valid address of an existing structure was found */
    if ((pFontSizeDeps->pNextFontSizeDeps=(FONTSIZEDEPS *)malloc(sizeof(FONTSIZEDEPS)))==NULL){
      T1_errno=T1ERR_ALLOC_MEM;
      return(NULL);
    }
    pFontSizeDeps=pFontSizeDeps->pNextFontSizeDeps;
  }

  /* The pointer to the previous struct */
  pFontSizeDeps->pPrevFontSizeDeps=pPrev;
  /* Put the size into this structure */
  pFontSizeDeps->size=size;
  /* Set the antialias mark: */
  if (aa)
    pFontSizeDeps->antialias=1;
  else
    pFontSizeDeps->antialias=0;
  
  /* Just the current becomes now the last item in the linked list: */
  pFontSizeDeps->pNextFontSizeDeps=NULL;
  /* Setup CharSpaceMatrix for this font: */
  pFontSizeDeps->pCharSpaceLocal=(struct XYspace *) IDENTITY;
  /* Apply transformation with font matrix: */
  pFontSizeDeps->pCharSpaceLocal=(struct XYspace *)
    Transform(pFontSizeDeps->pCharSpaceLocal,
	      pFontBase->pFontArray[FontID].FontMatrix[0],
	      pFontBase->pFontArray[FontID].FontMatrix[1],
	      pFontBase->pFontArray[FontID].FontMatrix[2],
	      pFontBase->pFontArray[FontID].FontMatrix[3]);
  /* Apply a further transformation (optionally): */
  pFontSizeDeps->pCharSpaceLocal=(struct XYspace *)
    Transform(pFontSizeDeps->pCharSpaceLocal,
	      pFontBase->pFontArray[FontID].FontTransform[0],
	      pFontBase->pFontArray[FontID].FontTransform[1],
	      pFontBase->pFontArray[FontID].FontTransform[2],
	      pFontBase->pFontArray[FontID].FontTransform[3]);
  /* Apply device specific scaling */
  pFontSizeDeps->pCharSpaceLocal=(struct XYspace *)
    Scale(pFontSizeDeps->pCharSpaceLocal,
	  DeviceSpecifics.scale_x,
	  - DeviceSpecifics.scale_y);
  /* Apply desired scaling factor, and make it Permanent */
  pFontSizeDeps->pCharSpaceLocal=(struct XYspace *) Permanent
    (Scale(pFontSizeDeps->pCharSpaceLocal, size, size));
							 
  /* We should now allocate memory for the glyph area of the font
     cache: */
  if ((pFontSizeDeps->pFontCache=(GLYPH *)calloc(256,sizeof(GLYPH)))
      ==NULL)
    return(NULL);

  sprintf( err_warn_msg_buf, "New Size %f created for FontID %d (antialias=%d)",
	   pFontSizeDeps->size, FontID, pFontSizeDeps->antialias);
  T1_PrintLog( "CreateNewFontSize()", err_warn_msg_buf, T1LOG_STATISTIC);
  /* We are done */
  return(pFontSizeDeps);
  
}

  


/* QueryFontSize( FontID, size, aa): Search if a requested size of font
   FontID is already existing. If so, it returns a pointer to the
   respective FontSizeDeps-structure,  otherwise NULL is returned: */
FONTSIZEDEPS *QueryFontSize( int FontID, float size, int aa)
{
  
  FONTSIZEDEPS *link_ptr;


  /* There's not yet one size: */
  if (pFontBase->pFontArray[FontID].pFontSizeDeps == NULL)
    return(pFontBase->pFontArray[FontID].pFontSizeDeps);
  

  /* There's already existing one or more size */
  link_ptr=pFontBase->pFontArray[FontID].pFontSizeDeps;
  
  while (((link_ptr->size != size)||(link_ptr->antialias != aa))
	 &&(link_ptr->pNextFontSizeDeps != NULL))
    link_ptr=link_ptr->pNextFontSizeDeps;
  
  if ((link_ptr->size != size)||(link_ptr->antialias != aa))
    return( NULL);     /* requested size/aa-combination  was not found */
  else
    return(link_ptr); /* return pointer to requested struct */
  
}

/* FONTSIZEDEPS *GetLastFontSize( FontID): Get the address of the
   last struct in the linked list of FontSizeDeps or NULL if there is
   no existing size dependent data. */
FONTSIZEDEPS *GetLastFontSize( int FontID)
{
  FONTSIZEDEPS *link_ptr, *result_ptr;
  

  /* There's not yet one size: */
  if (pFontBase->pFontArray[FontID].pFontSizeDeps == NULL)
    return((FONTSIZEDEPS *) (pFontBase->pFontArray[FontID].pFontSizeDeps));
  
  
  /* There's already existing one or more size */
  link_ptr=pFontBase->pFontArray[FontID].pFontSizeDeps;
  
  while (link_ptr != NULL){
    result_ptr=link_ptr;
    link_ptr=link_ptr->pNextFontSizeDeps;
  }

  return((FONTSIZEDEPS *)(result_ptr));
}

