/* mailcheck.c
 *
 * Copyright 1996, 1997, 1998 Jefferson E. Noxon <jeff@planetfall.com>
 *
 * This file may be copied under the terms of the GNU Public License
 * version 2, incorporated herein by reference.
 */

/* Command line parameters:
 * -l: login mode; program exits quietly if ~/.hushlogin exists.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>

/* Global variables */
char *Homedir;			/* Home directory pathname */
int Login_mode;			/* TRUE if the '-l' switch is set */

/* Open an rc file.  Return NULL if we can't find one. */
FILE *
open_rcfile (void)
{
  char namebuf[256];

  snprintf (namebuf, sizeof (namebuf), "%s/.mailcheckrc", Homedir);
  if (!access (namebuf, R_OK))
    return fopen (namebuf, "r");

  return fopen ("/etc/mailcheckrc", "r");
}


/* Expand environment variables.  Input buffer must be long enough
 * to hold output.  Nested $()'s don't work.
 */
char *
expand_envstr (char *path)
{
  char *srcptr, *envptr, *tmpptr, *dup, *envname;
  int len;

  srcptr = strstr(path, "$(");
  if (!srcptr)
    return path;

  len = strlen(path);
    if (!len)
      return path;

  dup = malloc(len+1);
    if (!dup)
      return path;

  strcpy(dup, path);
  envptr = strstr(dup, "$(") + 2;
  tmpptr = strchr(envptr, ')');
  if (!tmpptr)
    {
      free (dup);
      return path;
    }

  *tmpptr = 0;
  *srcptr = 0;
  envname = getenv(envptr);
  if (envname)
    strcat(srcptr, envname);
  strcat(srcptr, tmpptr+1);

  free (dup);
  return expand_envstr(path);
}

int
count_entries (char *path)
{
  DIR *mdir;
  struct dirent *entry;
  int count=0;

  mdir = opendir(path);
  if (!mdir)
    return -1;

  while ((entry=readdir(mdir)))
    ++count;

  return count-2;
}

void
check_for_mail (char *tmppath)
{
  struct stat st;
  char *mailpath = expand_envstr (tmppath);
  char maildir[2048];
  int new=0, cur=0;

  if (!stat (mailpath, &st))
    {
      /* Is it a maildir? */
      if (S_ISDIR(st.st_mode))
        {
          sprintf(maildir,"%s/cur",mailpath);
          cur = count_entries(maildir);
          sprintf(maildir,"%s/new",mailpath);
          new = count_entries(maildir);

          if (cur && new)
            printf ("You have %d new and %d saved messages in %s\n",
                    new, cur, mailpath);
          else if (cur)
            printf ("You have %d saved messages in %s\n", cur, mailpath);
          else if (new)
            printf ("You have %d new messages in %s\n", new, mailpath);
        }

      /* It's an mbox. */
      else
        if (st.st_size != 0)
          printf ("You have %smail in %s\n",
	    (st.st_mtime > st.st_atime) ? "new " : "", mailpath);
    }

}

/* Process command-line options */
void
process_options (int argc, char *argv[])
{
  int result;

  while (1)
    {
      result = getopt (argc, argv, "l");

      switch (result)
	{
	case EOF:
	  return;
	case 'l':
	  Login_mode = 1;
	  break;
	}
    }
}

int
main (int argc, char *argv[])
{
  char buf[1024], *ptr;
  FILE *rcfile;
  struct stat st;

  Homedir = getenv ("HOME");
  if (!Homedir)
    {
      fprintf (stderr, "%s: couldn't read environment variable HOME.\n",
	       argv[0]);
      return 1;
    }

  process_options (argc, argv);

  if (Login_mode)
    {
      /* If we can stat .hushlogin successfully, we should exit. */
      snprintf (buf, sizeof (buf), "%s/.hushlogin", Homedir);
      if (!stat (buf, &st))
	return 0;
    }

  rcfile = open_rcfile ();
  if (!rcfile)
    {
      fprintf (stderr, "%s: couldn't open /etc/mailcheckrc "
	       "or %s/.mailcheckrc.\n", argv[0], Homedir);
      return 1;
    }

  while (fgets (buf, sizeof (buf), rcfile))
    {
      /* eliminate newline */
      ptr = strchr (buf, '\n');
      if (ptr)
	*ptr = 0;

      /* If it's not a blank line or comment, look for mail in it */
      if (strlen (buf) && (*buf != '#'))
	check_for_mail (buf);
    }

  return 0;
}
