// Fl_win32.C

// fltk (Fast Light Tool Kit) version 0.99
// Copyright (C) 1998 Bill Spitzak

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.

// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.

// Written by Bill Spitzak spitzak@d2.com

// This file contains win32-specific code for fltk which is always linked
// in.  Search other files for "WIN32" or filenames ending in _win32.C
// for other system-specific code.

#include <FL/Fl.H>
#include <FL/win32.H>
#include <FL/Fl_Window.H>
#include <string.h>

////////////////////////////////////////////////////////////////
// interface to poll/select call:

// fd's are not yet implemented.
// On NT these are probably only used for network stuff, so this may
// talk to a package that Wonko has proposed writing to make the network
// interface system independent.

#define POLLIN 1
#define POLLOUT 4
#define POLLERR 8

void Fl::add_fd(int n, int events, void (*cb)(int, void*), void *v) {}

void Fl::add_fd(int fd, void (*cb)(int, void*), void* v) {
  Fl::add_fd(fd,POLLIN,cb,v);
}

void Fl::remove_fd(int n) {}

// Timeouts are insert-sorted into order.  This works good if there
// are only a small number:

#define MAXTIMEOUT 8

static int numtimeouts;
static struct {
  double time;
  void (*cb)(void*);
  void* arg;
} timeout[MAXTIMEOUT+1];

void Fl::add_timeout(double t, void (*cb)(void *), void *v) {
  int i;
  if (numtimeouts<MAXTIMEOUT) numtimeouts++;
  for (i=0; i<numtimeouts-1; i++) {
    if (timeout[i].time > t) {
      for (int j=numtimeouts-1; j>i; j--) timeout[j] = timeout[j-1];
      break;
    }
  }
  timeout[i].time = t;
  timeout[i].cb = cb;
  timeout[i].arg = v;
}

void Fl::remove_timeout(void (*cb)(void *), void *v) {
  int i,j;
  for (i=j=0; i<numtimeouts; i++) {
    if (timeout[i].cb == cb && timeout[i].arg==v) ;
    else {if (j<i) timeout[j]=timeout[i]; j++;}
  }
  numtimeouts = j;
}

void (*Fl::idle)();

// I have attempted to write these to not call reset() unnecessarily,
// since on some systems this results in a time-wasting context switch
// to the system when Clock() is called.  This means that only
// Fl::reset() and Fl::wait(n) are guaranteed to reset the clock that
// timeouts are measured from.  Fl::wait(), Fl::check(), and
// Fl::ready() may or may not change the clock.

MSG fl_msg;

static int initclock; // if false we didn't call reset() last time

double Fl::reset() {

  // This should be a value that increases uniformly over real time:
  unsigned long newclock = fl_msg.time; // NOT YET IMPLEMENTED!
  const int TICKS_PER_SECOND = 1000; // divisor of the value to get seconds

  static unsigned long prevclock;
  if (!initclock) {prevclock = newclock; initclock = 1; return 0;}
  double t = double(newclock-prevclock)/TICKS_PER_SECOND;
  prevclock = newclock;

  // elapse the pending timeouts:
  for (int i=0; i<numtimeouts; i++) timeout[i].time -= t;
  return t;
}

void call_timeouts() {
  if (!numtimeouts || timeout[0].time > 0) return;
  struct {
    void (*cb)(void *);
    void *arg;
  } temp[MAXTIMEOUT];
  int i,j,k;
  // copy all expired timeouts to temp array:
  for (i=j=0; j<numtimeouts && timeout[j].time <= 0; i++,j++) {
    temp[i].cb = timeout[j].cb;
    temp[i].arg= timeout[j].arg;
  }
  // remove them from source array:
  for (k=0; j<numtimeouts;) timeout[k++] = timeout[j++];
  numtimeouts = k;
  // and then call them:
  for (k=0; k<i; k++) temp[k].cb(temp[k].arg);
}

static double innards(int timeout_flag, double time) {
  int have_message;
  // get the first message by waiting the correct amount of time:
  if (!timeout_flag) {
    initclock = 0; // don't waste time calling system for current time
    GetMessage(&fl_msg, NULL, 0, 0);
    have_message = 1;
  } else {
    if (time > 0) time -= Fl::reset();
    if (time >= 0.001) {
      int timerid = SetTimer(NULL, 0, int(time*1000), NULL);
      GetMessage(&fl_msg, NULL, 0, 0);
      KillTimer(NULL, timerid);
      have_message = 1;
    } else {
      have_message = PeekMessage(&fl_msg, NULL, 0, 0, PM_REMOVE);
      if (have_message && !time)
	time = -.00000001; // so check() can return non-zero
    }
  }
  // execute it, them execute any other messages that become ready during it:
  while (have_message) {
    DispatchMessage(&fl_msg);
    have_message = PeekMessage(&fl_msg, NULL, 0, 0, PM_REMOVE);
  }
  return time;
}

int Fl::wait() {
  flush();
  if (!numtimeouts && !idle) {
    innards(0, 0.0);
  } else {
    innards(1, idle ? 0.0 : timeout[0].time);
    reset();	// expire timeouts during poll call
    call_timeouts();
    if (idle) idle();
  }
  return 1;
}

double Fl::wait(double time) {
  flush();
  double wait_time = idle ? 0.0 : time;
  if (numtimeouts && timeout[0].time < time) wait_time = timeout[0].time;
  double remaining_wait = innards(1, wait_time);
  time = (time-wait_time+remaining_wait) - reset();
  call_timeouts();
  if (idle) idle();
  return time;
}

int Fl::check() {
  flush();
  double t = innards(1, 0.0);
  if (numtimeouts) {reset(); call_timeouts();}
//if (idle) idle(); // ??? should it do this ???
  return t < 0.0;
}

int Fl::ready() {
//if (idle) return 1; // ??? should it do this ???
  if (numtimeouts) {reset(); if (timeout[0].time <= 0) return 1;}
  return PeekMessage(&fl_msg, NULL, 0, 0, PM_NOREMOVE);
}

////////////////////////////////////////////////////////////////

Fl_Window *Fl_Window::current_;

int Fl::h() {return GetSystemMetrics(SM_CYSCREEN);}

int Fl::w() {return GetSystemMetrics(SM_CXSCREEN);}

void Fl::get_mouse(int &x, int &y) {
  // not yet implemented (is this impossible?  Seems hard to believe
  // as it would not fit in with Windoze design principles...
  // Fake version returns most-recent event
  // but if no window, guess that mouse is in middle of screen
  x = event_x_root();
  y = event_y_root();
  if (!x && !y && !Fl_X::first) {
    x = w()/2;
    y = h()/2;
  }
}

////////////////////////////////////////////////////////////////

Fl_X* Fl_X::first;

static Fl_X* freelist;

Fl_Window* fl_find(HWND xid) {
  Fl_X *window;
  for (Fl_X **pp = &Fl_X::first; (window = *pp); pp = &window->next)
    if (window->xid == xid) {
      if (window != Fl_X::first && !Fl::modal()) {
	// make this window be first to speed up searches
	// this is not done if modal is true to avoid messing up modal stack
	*pp = window->next;
	window->next = Fl_X::first;
	Fl_X::first = window;
      }
      return window->w;
    }
  return 0;
}

////////////////////////////////////////////////////////////////

extern Fl_Window *fl_xfocus;	// in Fl.C
extern Fl_Window *fl_xmousewin; // in Fl.C
void fl_fix_focus(); // in Fl.C

// call this to free a selection (or change the owner):
// (under X this had to be communicated to the system)
void Fl::selection_owner(Fl_Widget *owner) {
  if (selection_owner_ && owner != selection_owner_)
    selection_owner_->handle(FL_SELECTIONCLEAR);
  selection_owner_ = owner;
}

////////////////////////////////////////////////////////////////

// event processors in Fl.C:
void fl_send_event(int event, Fl_Widget *widget);
void fl_send_push(Fl_Window*);
void fl_send_move(Fl_Window*);
void fl_send_release(Fl_Window*);
void fl_send_keyboard(Fl_Window*);

static void mouse_event(Fl_Window *window, int what, int button,
			WPARAM wParam, LPARAM lParam)
{
  static int px, py, pmx, pmy;
  POINT pt; pt.x = LOWORD(lParam); pt.y = HIWORD(lParam);
  Fl::e_x = pt.x & 0x8000 ? pt.x-0x10000 : pt.x;
  Fl::e_y = pt.y & 0x8000 ? pt.y-0x10000 : pt.y;
  ClientToScreen(fl_xid(window), &pt);
  Fl::e_x_root = pt.x & 0x8000 ? pt.x-0x10000 : pt.x;
  Fl::e_y_root = pt.y & 0x8000 ? pt.y-0x10000 : pt.y;
#if 0
  if (window->parent()) {
    do window = window->window(); while (window->parent());
    ScreenToClient(fl_xid(window), &pt);
    Fl::e_x = pt.x & 0x8000 ? pt.x-0x10000 : pt.x;
    Fl::e_y = pt.y & 0x8000 ? pt.y-0x10000 : pt.y;
  }
#endif
  ulong state = 0;
  // if GetKeyState is expensive we might want to comment some of these out:
  if (wParam & MK_SHIFT) state |= FL_SHIFT;
  if (GetKeyState(VK_CAPITAL)) state |= FL_CAPS_LOCK;
  if (wParam & MK_CONTROL) state |= FL_CTRL;
  if (GetKeyState(VK_MENU)&~1) state |= FL_ALT;
  if (GetKeyState(VK_NUMLOCK)) state |= FL_NUM_LOCK;
  if (GetKeyState(VK_LWIN)&~1 || GetKeyState(VK_RWIN)&~1) state |= FL_META;
  if (GetKeyState(VK_SCROLL)) state |= FL_SCROLL_LOCK;
  if (wParam & MK_LBUTTON) state |= FL_BUTTON1;
  if (wParam & MK_MBUTTON) state |= FL_BUTTON2;
  if (wParam & MK_RBUTTON) state |= FL_BUTTON3;
  Fl::e_state = state;

  switch (what) {
  case 1: // double-click
    if (Fl::e_is_click) {Fl::e_clicks++; goto J1;}
  case 0: // single-click
    Fl::e_clicks = 0;
  J1:
    SetCapture(fl_xid(window));
    Fl::e_keysym = FL_Button + button;
    Fl::e_is_click = 1;
    px = pmx = Fl::e_x_root; py = pmy = Fl::e_y_root;
    fl_send_push(window);
    break;

  case 2: // release:
    ReleaseCapture();
    Fl::e_keysym = FL_Button + button;
    fl_send_release(window);
    break;

  case 3: // move:
    // windoze produces extra events even if mouse does not move, ignore em:
    if (Fl::e_x_root == pmx && Fl::e_y_root == pmy) return;
    pmx = Fl::e_x_root; pmy = Fl::e_y_root;
    if (abs(Fl::e_x_root-px)>5 || abs(Fl::e_y_root-py)>5) Fl::e_is_click = 0;
    fl_send_move(window);
    break;

  }
}

// convert a Micro$oft VK_x to an Fltk (X) Keysym:
// See also the inverse converter in Fl_get_key_win32.C
// This table is in numeric order by VK:
static const struct {unsigned short vk, fltk;} vktab[] = {
  {VK_BACK,	FL_BackSpace},
  {VK_TAB,	FL_Tab},
  {VK_CLEAR,	FL_KP+'5'},
  {VK_RETURN,	FL_Enter},
  {VK_SHIFT,	FL_Shift_L},
  {VK_CONTROL,	FL_Control_L},
  {VK_MENU,	FL_Alt_L},
  {VK_PAUSE,	FL_Pause},
  {VK_CAPITAL,	FL_Caps_Lock},
  {VK_ESCAPE,	FL_Escape},
  {VK_SPACE,	' '},
  {VK_PRIOR,	FL_Page_Up},
  {VK_NEXT,	FL_Page_Down},
  {VK_END,	FL_End},
  {VK_HOME,	FL_Home},
  {VK_LEFT,	FL_Left},
  {VK_UP,	FL_Up},
  {VK_RIGHT,	FL_Right},
  {VK_DOWN,	FL_Down},
  {VK_SNAPSHOT,	FL_Print},	// does not work on NT
  {VK_INSERT,	FL_Insert},
  {VK_DELETE,	FL_Delete},
  {VK_LWIN,	FL_Meta_L},
  {VK_RWIN,	FL_Meta_R},
  {VK_APPS,	FL_Menu},
  {VK_MULTIPLY,	FL_KP+'*'},
  {VK_ADD,	FL_KP+'+'},
  {VK_SUBTRACT,	FL_KP+'-'},
  {VK_DECIMAL,	FL_KP+'.'},
  {VK_DIVIDE,	FL_KP+'/'},
  {VK_NUMLOCK,	FL_Num_Lock},
  {VK_SCROLL,	FL_Scroll_Lock},
  {0xba,	';'},
  {0xbb,	'='},
  {0xbc,	','},
  {0xbd,	'-'},
  {0xbe,	'.'},
  {0xbf,	'/'},
  {0xc0,	'`'},
  {0xdb,	'['},
  {0xdc,	'\\'},
  {0xdd,	']'},
  {0xde,	'\''}
};
static int ms2fltk(int vk, int extended) {
  static unsigned short vklut[256];
  if (!vklut[1]) { // init the table
    int i;
    for (i = 0; i < 256; i++) vklut[i] = tolower(i);
    for (i=VK_F1; i<=VK_F16; i++) vklut[i] = i+(FL_F-(VK_F1-1));
    for (i=VK_NUMPAD0; i<=VK_NUMPAD9; i++) vklut[i] = i+(FL_KP+'0'-VK_NUMPAD0);
    for (i = 0; i < sizeof(vktab)/sizeof(*vktab); i++)
      vklut[vktab[i].vk] = vktab[i].fltk;
  }
  if (extended) switch (vk) {
    case VK_CONTROL : return FL_Control_R;
    case VK_MENU: return FL_Alt_R;
    case VK_RETURN: return FL_KP_Enter;
  }
  return vklut[vk];
}

static char call_DefWindowProc;
static Fl_Window* fl_resize_bug_fix;
char fl_direct_paint;

HPALETTE fl_colormap;
extern HPALETTE fl_make_palette(); // in fl_color_win32.C

static LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam) 
{
  static char buffer[2];

  fl_msg.message = uMsg;

  Fl_Window *window = fl_find(hWnd);
  if (!window) goto DEFAULT;

MS_IS_FUCKED_UP:
  switch (uMsg) {

  case WM_QUIT: // this should not happen?
    Fl::fatal("WM_QUIT message");

  // case WM_CREATE: // we don't care about this
  // case WM_DESTROY: // we don't care about this

  case WM_CLOSE: // user clicked close box
    if (Fl::modal()) {
      if (window != Fl::modal()) break;
    } else {
      Fl::atclose(window, window->user_data());
    }
    window->do_callback();
    break;

  case WM_PAINT: {
    // some "internal" repaints must be ignored?
    // RECT rect;
    // if (!GetUpdateRect(hWnd, &rect, FALSE)) break;
    // Stoopid windoze really wants you to paint the window *now*,
    // but has already set the clip region!  Fltk no like this,
    // since it wants to draw it's own damage, which may be outside
    // this area.  I kludge around this, grep for fl_direct_paint
    // to find the kludges...
    if (!window->damage()) fl_direct_paint = 1;
    PAINTSTRUCT ps;
    fl_gc = BeginPaint(hWnd, &ps);
    window->expose(2, ps.rcPaint.left, ps.rcPaint.top,
		   ps.rcPaint.right-ps.rcPaint.left,
		   ps.rcPaint.bottom-ps.rcPaint.top);
    if (!fl_direct_paint) EndPaint(hWnd, &ps);
    Fl_X::i(window)->flush();
    window->clear_damage();
    Fl_X::i(window)->region = 0;
    if (fl_direct_paint) {EndPaint(hWnd, &ps); fl_direct_paint = 0;}
    goto DEFWINDOWPROC; // will this fix it?
    }

  case WM_LBUTTONDOWN:  mouse_event(window, 0, 1, wParam, lParam); break;
  case WM_LBUTTONDBLCLK:mouse_event(window, 1, 1, wParam, lParam); break;
  case WM_LBUTTONUP:    mouse_event(window, 2, 1, wParam, lParam); break;
  case WM_MBUTTONDOWN:  mouse_event(window, 0, 2, wParam, lParam); break;
  case WM_MBUTTONDBLCLK:mouse_event(window, 1, 2, wParam, lParam); break;
  case WM_MBUTTONUP:    mouse_event(window, 2, 2, wParam, lParam); break;
  case WM_RBUTTONDOWN:  mouse_event(window, 0, 3, wParam, lParam); break;
  case WM_RBUTTONDBLCLK:mouse_event(window, 1, 3, wParam, lParam); break;
  case WM_RBUTTONUP:    mouse_event(window, 2, 3, wParam, lParam); break;
  case WM_MOUSEMOVE:    mouse_event(window, 3, 0, wParam, lParam); break;

  case WM_SETFOCUS:
    Fl::e_keysym = 0; // make sure it is not confused with navigation key
    fl_xfocus = window;
    fl_fix_focus();
    goto DEFWINDOWPROC;

  case WM_KILLFOCUS:
    Fl::e_keysym = 0; // make sure it is not confused with navigation key
    fl_xfocus = 0;
    fl_fix_focus();
    Fl::flush(); // it never returns to main loop when deactivated...
    goto DEFWINDOWPROC;

  case WM_SHOWWINDOW:
    if (wParam) {
      window->set_visible();
      window->handle(FL_SHOW);
    } else {
      window->clear_visible();
      window->handle(FL_HIDE);
    }
    goto DEFWINDOWPROC;

  case WM_KEYDOWN:
  case WM_SYSKEYDOWN:
    // save the keysym until we figure out the characters:
    Fl::e_keysym = ms2fltk(wParam,lParam&(1<<24));
  case WM_KEYUP:
  case WM_SYSKEYUP:
    TranslateMessage(&fl_msg); // always returns 1!!!
    // A bit of Windoze horror: this seems to be the only way to
    // figure out if this keystroke is going to produce a damn
    // character :-( :
    if (PeekMessage(&fl_msg, hWnd, WM_CHAR, WM_SYSDEADCHAR, 1)) {
      uMsg = fl_msg.message;
      wParam = fl_msg.wParam;
      lParam = fl_msg.lParam;
      goto MS_IS_FUCKED_UP;
    }
    // otherwise use it as a 0-character key...
  case WM_DEADCHAR:
  case WM_SYSDEADCHAR:
    buffer[0] = 0;
    Fl::e_text = buffer;
    Fl::e_length = 0;
    goto GETSTATE;
  case WM_CHAR:
  case WM_SYSCHAR:
    buffer[0] = char(wParam);
    Fl::e_text = buffer;
    Fl::e_length = 1;
  GETSTATE:
    {ulong state = Fl::e_state & 0xff000000; // keep the mouse button state
     // if GetKeyState is expensive we might want to comment some of these out:
      if (GetKeyState(VK_SHIFT)&~1) state |= FL_SHIFT;
      if (GetKeyState(VK_CAPITAL)) state |= FL_CAPS_LOCK;
      if (GetKeyState(VK_CONTROL)&~1) state |= FL_CTRL;
      if (lParam & (1<<29)) state |= FL_ALT;
      if (GetKeyState(VK_NUMLOCK)) state |= FL_NUM_LOCK;
      if (GetKeyState(VK_LWIN)&~1 || GetKeyState(VK_RWIN)&~1) state |= FL_META;
      if (GetKeyState(VK_SCROLL)) state |= FL_SCROLL_LOCK;
      Fl::e_state = state;}
    if (lParam & (1<<31)) goto DEFAULT; // ignore up events (after fixing shift)
    /*for (int i = lParam&0xff; i--;)*/ fl_send_keyboard(window);
    break;

  case WM_GETMINMAXINFO:
    Fl_X::i(window)->set_minmax((LPMINMAXINFO)lParam);
    break;

  case WM_SIZE:
    if (wParam == SIZEICONIC || wParam == SIZEZOOMHIDE) {
      window->clear_visible();
      window->handle(FL_HIDE);
    } else {
      window->set_visible();
      window->handle(FL_SHOW);
      int W = LOWORD(lParam);
      int H = HIWORD(lParam);
      fl_resize_bug_fix = window;
      window->size(W, H);
      fl_resize_bug_fix = 0;
    }
    goto DEFWINDOWPROC;

  case WM_MOVE:
    if (window->visible()) {
      int X = LOWORD(lParam);
      int Y = HIWORD(lParam);
      fl_resize_bug_fix = window;
      window->position(X, Y);
      fl_resize_bug_fix = 0;
    }
    goto DEFWINDOWPROC;

  case WM_SETCURSOR:
    if (LOWORD(lParam) != HTCLIENT)
      goto DEFWINDOWPROC;
    else
      SetCursor(Fl_X::i(window)->cursor);
    break;

  case WM_QUERYNEWPALETTE :
    // Realize the color palette if necessary...
    if (fl_colormap) {
      SelectPalette(fl_gc, fl_colormap, FALSE);
      RealizePalette(fl_gc);
      InvalidateRect(hWnd, NULL, FALSE);
      return (TRUE);
    }
    break;
       
  case WM_PALETTECHANGED:
    // Reselect our color palette if necessary...
    if (fl_colormap && (HWND)wParam != hWnd) {
      SelectPalette(fl_gc, fl_colormap, FALSE);
      RealizePalette(fl_gc);
      UpdateColors(fl_gc);
    }
    break;

  case WM_CREATE :
    // Make a color palette if necessary
    fl_colormap = fl_make_palette();
    if (fl_colormap) {
      SelectPalette(fl_gc, fl_colormap, FALSE);
      RealizePalette(fl_gc);
    }
    goto DEFAULT;

  default:
  DEFAULT:
    fl_send_event(0, 0);
    break;
  }

  if (!call_DefWindowProc) return 0;
  // flag must be turned off now.  Otherwise it may cause
  // an unwanted call to DefWindowProc when a callback that
  // processed events returns.
  call_DefWindowProc = 0;

DEFWINDOWPROC:
  return DefWindowProc(hWnd, uMsg, wParam, lParam);
}

// if event unwanted by any Fl::add_handler functions, this is called:
int fl_backstop_handle(int event) {
  if (event == FL_PUSH) {
    // raise windows that are clicked on:
    Fl_X::first->w->show();
  } else if (event == FL_SHORTCUT && Fl::event_key()==FL_Escape) {
    // Escape key closes windows:
    Fl_Window* w = Fl::modal();
    if (!w) {w = Fl_X::first->w; Fl::atclose(w, w->user_data());}
    w->do_callback();
  }
  // If we don't want the event, tell winProc to let windoze do it
  call_DefWindowProc = 1;
  return 1;
}

////////////////////////////////////////////////////////////////

void Fl_Window::resize(int X,int Y,int W,int H) {
  if (shown() && this != fl_resize_bug_fix) {
    if (X==x() && Y==y() && W==w() && H==h()) return;
    MoveWindow(i->xid, X, Y, W, H, TRUE);
    if (!parent()) redraw();
  } else {
    // This is to avoid echoing resize events back to window manager:
    fl_resize_bug_fix = 0;
  }
  if (X != x() || Y != y()) set_flag(FL_FORCE_POSITION);
  if (W != w() || H != h()) Fl_Group::resize(X,Y,W,H); else {x(X); y(Y);}
  // Notice that this does *not* set any redraw bits.  I assumme
  // I will receive damage for the whole window from X.  I think
  // that "ForgetGravity" forces the expose event for the entire
  // window, but this may not be true on some implementations.
}

////////////////////////////////////////////////////////////////

char fl_show_iconic;	// hack for Fl_Window::iconic()
// int fl_background_pixel = -1; // color to use for background
HCURSOR fl_default_cursor;

Fl_X* Fl_X::make(Fl_Window* w) {
  Fl_Group::current(0); // get rid of very common user bug: forgot end()
  w->clear_damage(); // wait for expose events

  static char* class_name;
  if (!class_name) {	// create a single Windoze class used for everything:
    class_name = "FLTK";
    WNDCLASSEX wc;
    wc.style = CS_HREDRAW | CS_VREDRAW | CS_CLASSDC | CS_DBLCLKS;
    wc.lpfnWndProc = (WNDPROC)WndProc;
    wc.cbClsExtra = wc.cbWndExtra = 0;
    wc.hInstance = fl_display;
    wc.hIcon = wc.hIconSm = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = fl_default_cursor = LoadCursor(NULL, IDC_ARROW);
    //uchar r,g,b; Fl::get_color(FL_GRAY,r,g,b);
    //wc.hbrBackground = (HBRUSH)CreateSolidBrush(RGB(r,g,b));
    wc.hbrBackground = NULL;
    wc.lpszMenuName = NULL;
    wc.lpszClassName = class_name;
    wc.cbSize = sizeof(WNDCLASSEX);
    RegisterClassEx(&wc);
  }

  HWND parent;
  DWORD style;
  int xp = w->x();
  int yp = w->y();
  int wp = w->w();
  int hp = w->h();

  if (w->parent()) {
    style = WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
    parent = fl_xid(w->window());
  } else {
    if (!w->size_range_set) {
      if (w->resizable()) {
	Fl_Widget *o = w->resizable();
	int minw = o->w(); if (minw > 100) minw = 100;
	int minh = o->h(); if (minh > 100) minh = 100;
	w->size_range(w->w() - o->w() + minw, w->h() - o->h() + minh, 0, 0);
      } else {
	w->size_range(w->w(), w->h(), w->w(), w->h());
      }
    }
    if (w->border()) {
      style = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU
	| WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
      if (w->maxw != w->minw || w->maxh != w->minh)
	style |= WS_THICKFRAME | WS_MAXIMIZEBOX;
      if (!w->modal()) style |= WS_MINIMIZEBOX;
      xp -= GetSystemMetrics(SM_CXFRAME);
      yp -= GetSystemMetrics(SM_CYFRAME)+GetSystemMetrics(SM_CYCAPTION);
      wp += 2*GetSystemMetrics(SM_CXFRAME);
      hp += 2*GetSystemMetrics(SM_CYFRAME)+GetSystemMetrics(SM_CYCAPTION);
    } else {
      style = WS_POPUP | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
    }
    if (!(w->flags() & Fl_Window::FL_FORCE_POSITION)) {
      xp = yp = CW_USEDEFAULT;
    }
    parent = 0;
    if (w->non_modal()) { // find some other window to be "transient for":
      Fl_X* y = Fl_X::first;
      while (y && (y->w->parent() || y->w->non_modal())) y = y->next;
      if (y) parent = y->xid;
    }
  }

  Fl_X* x = freelist;
  if (x) freelist = x->next; else x = new Fl_X;
  x->other_xid = 0;
  x->setwindow(w);
  x->region = 0;
  x->private_dc = 0;
  x->cursor = fl_default_cursor;
  x->xid = CreateWindow(
    class_name, w->label(), style,
    xp, yp, wp, hp,
    parent,
    NULL, // menu
    fl_display,
    NULL // creation parameters
    );
  x->next = Fl_X::first;
  Fl_X::first = x;

  // use w->xclass() to set the icon...
  if (fl_show_iconic)
    ShowWindow(x->xid, SW_MINIMIZE);
  else {
    ShowWindow(x->xid, SW_SHOW);
    ((Fl_Group*)w)->show();
    fl_fix_focus();
  }
  return x;
}

void Fl::flush() {
  Fl_Group::current(0); // get rid of very common user bug: forgot end()
  if (damage()) {
    damage_ = 0;
    for (Fl_X* x = Fl_X::first; x; x = x->next) {
      if (x->w->damage() && x->w->visible()) {
	x->flush();
	x->w->clear_damage();
	x->region = 0;
      }
    }
  }
}

void Fl::redraw() {
  for (Fl_X* x = Fl_X::first; x; x = x->next) x->w->clear_damage(~0);
}

Fl_Window* Fl::first_window() {Fl_X* x = Fl_X::first; return x ? x->w : 0;}

Fl_Window* Fl::next_window(const Fl_Window* w) {
  Fl_X* x = Fl_X::i(w)->next; return x ? x->w : 0;}

////////////////////////////////////////////////////////////////

HINSTANCE fl_display;

int Fl_WinMain(HINSTANCE hInstance, LPSTR lpCmdLine, int nCmdShow,
	       int (*mainp)(int, char**)) {
  fl_display = hInstance;

  int argc;
  char **argv;
  // test version for now:
  argc = 1; char* testargv[] = {"test", 0}; argv = testargv;

  return mainp(argc, argv);
}

////////////////////////////////////////////////////////////////
// hide() destroys the X window:

void Fl_Window::hide() {
  if (!shown()) return;

  // remove from the list of windows:
  Fl_X* x = i;
  Fl_X** pp = &Fl_X::first;
  for (; *pp != x; pp = &(*pp)->next) if (!*pp) return;
  *pp = x->next;

  // recursively remove any subwindows:
  for (Fl_X *w = Fl_X::first; w;) {
    if (w->w->window() == this) {w->w->hide(); w = Fl_X::first;}
    else w = w->next;
  }
  if (x->private_dc) ReleaseDC(x->xid,x->private_dc);

  DestroyWindow(x->xid);
  i = 0;
  x->next = freelist; freelist = x;
  set_flag(INVISIBLE);
  handle(FL_HIDE);

  // Make sure no events are sent to this window:
  if (contains(Fl::pushed())) Fl::pushed(Fl_X::first->w);
  if (this == fl_xmousewin) fl_xmousewin = 0;
  if (this == fl_xfocus) fl_xfocus = 0;
  fl_fix_focus();

}

Fl_Window::~Fl_Window() {
  hide();
}

////////////////////////////////////////////////////////////////

void Fl_Window::size_range_() {
  size_range_set = 1;
}

void Fl_X::set_minmax(LPMINMAXINFO minmax)
{
  int wd, hd;
  if (w->border()) {
    wd = 2*GetSystemMetrics(SM_CXFRAME);
    hd = 2*GetSystemMetrics(SM_CYFRAME) + GetSystemMetrics(SM_CYCAPTION);
  } else {
    wd = hd = 0;
  }
  minmax->ptMinTrackSize.x = w->minw + wd;
  minmax->ptMinTrackSize.y = w->minh + hd;
  if (w->maxw) {
    minmax->ptMaxTrackSize.x = w->maxw + wd;
    minmax->ptMaxSize.x = w->maxw + wd;
  }
  if (w->maxh) {
    minmax->ptMaxTrackSize.y = w->maxh + hd;
    minmax->ptMaxSize.y = w->maxh + hd;
  }
}

////////////////////////////////////////////////////////////////

// returns pointer to the filename, or null if name ends with '/'
const char *filename_name(const char *name) {
  const char *p,*q;
  for (p=q=name; *p; p++) if (*p == '/' || *p == '\\' || *p == ':') q = p+1;
  return q;
}

void Fl_Window::label(const char *name,const char *iname) {
  Fl_Widget::label(name);
  iconlabel_ = iname;
  if (shown() && !parent()) {
    if (!name) name = "";
    SetWindowText(i->xid, name);
    // if (!iname) iname = filename_name(name);
    // should do something with iname here...
  }
}

void Fl_Window::label(const char *name) {
  label(name, iconlabel());
}

void Fl_Window::iconlabel(const char *iname) {
  label(label(), iname);
}

////////////////////////////////////////////////////////////////
// Implement the virtual functions for the base Fl_Window class:

// If the box is a filled rectangle, we can make the redisplay *look*
// faster by using X's background pixel erasing.  We can make it
// actually *be* faster by drawing the frame only, this is done by
// setting fl_boxcheat, which is seen by code in fl_drawbox.C:
// For WIN32 it looks like all windows share a background color, so
// I use FL_GRAY for this and only do this cheat for windows that are
// that color.

Fl_Widget *fl_boxcheat;
//static inline int can_boxcheat(uchar b) {return (b==1 || (b&2) && b<=15);}

void Fl_Window::show() {
  if (!shown()) {
    // if (can_boxcheat(box())) fl_background_pixel = fl_xpixel(color());
    Fl_X::make(this);
    // fl_background_pixel = -1;
  } else {
    ShowWindow(i->xid, SW_RESTORE);
    SetActiveWindow(i->xid);
  }
}

HDC fl_gc = 0; // the current context

// make X drawing go into this window (called by subclass flush() impl.)
void Fl_Window::make_current() {
  if (!fl_direct_paint) fl_gc = GetDC(i->xid);
  // calling GetDC seems to always reset these: (?)
  SetTextAlign(fl_gc, TA_BASELINE|TA_LEFT);
  SetBkMode(fl_gc, TRANSPARENT);
  current_ = this;
}

// WM_PAINT events and cropped damage call this:
void Fl_Window::expose(uchar flags,int X,int Y,int W,int H) {
  if (i) {
    if (!i->region.r) {
      i->region.x = X;
      i->region.y = Y;
      i->region.r = X+W;
      i->region.b = Y+H;
    } else {
      if (X < i->region.x) i->region.x = X;
      if (Y < i->region.y) i->region.y = Y;
      if (X+W > i->region.r) i->region.r = X+W;
      if (Y+H > i->region.b) i->region.b = Y+H;
    }
  }
  damage(flags);
}

#include <FL/fl_draw.H>

void Fl_Window::flush() {
  make_current();
  if (damage() & ~6) {
    draw();
  } else {
    fl_clip_region(i->region);
    draw();
    fl_pop_clip();
  }
}

// End of Fl_win32.C //
