// Fl_Widget.C

// fltk (Fast Light Tool Kit) version 0.99
// Copyright (C) 1998 Bill Spitzak

#include <FL/Fl.H>
#include <FL/Fl_Widget.H>
#include <FL/Fl_Group.H>

////////////////////////////////////////////////////////////////
// for compatability with Forms, all widgets without callbacks are
// inserted into a "queue" when they are activated, and the forms
// compatability interaction functions (fl_do_events, etc) will
// read one widget at a time from this queue and return it:

const int QUEUE_SIZE = 20;

static Fl_Widget *obj_queue[QUEUE_SIZE];
static int obj_head, obj_tail;

void Fl_Widget::default_callback(Fl_Widget *o, void * /*v*/) {
#if 0
  // This is necessary for strict forms compatability but is confusing.
  // Use the parent's callback if this widget does not have one.
  for (Fl_Widget *p = o->parent(); p; p = p->parent())
    if (p->callback() != default_callback) {
      p->do_callback(o,v);
      return;
    }
#endif
  obj_queue[obj_head++] = o;
  if (obj_head >= QUEUE_SIZE) obj_head = 0;
  if (obj_head == obj_tail) {
    obj_tail++;
    if (obj_tail >= QUEUE_SIZE) obj_tail = 0;
  }
}

Fl_Widget *Fl::readqueue() {
  if (obj_tail==obj_head) return 0;
  Fl_Widget *o = obj_queue[obj_tail++];
  if (obj_tail >= QUEUE_SIZE) obj_tail = 0;
  return o;
}
    
////////////////////////////////////////////////////////////////

int Fl_Widget::handle(int) {return 0;}

Fl_Widget::Fl_Widget(int X, int Y, int W, int H, const char* L) {

  x_ = ix_ = X; y_ = iy_ = Y; w_ = iw_ = W; h_ = ih_ = H;

  label_.value	= L;
  label_.type	= FL_NORMAL_LABEL;
  label_.font	= FL_HELVETICA;
  label_.size	= FL_NORMAL_SIZE;
  label_.color	= FL_BLACK;
  callback_	= default_callback;
  user_data_ 	= 0;
  type_		= 0;
  flags_	= 0;
  damage_	= 0;
  box_		= FL_NO_BOX;
  color_	= FL_GRAY;
  color2_	= FL_GRAY;
  align_	= FL_ALIGN_CENTER;
  when_		= FL_WHEN_RELEASE;

  parent_ = 0;
  if (Fl_Group::current()) Fl_Group::current()->add(this);
}

void Fl_Widget::resize(int X, int Y, int W, int H) {
  x_ = X; y_ = Y; w_ = W; h_ = H;
}

void Fl_Widget::init_size(int X, int Y, int W, int H) {
  resize(X,Y,W,H);
  ix_ = X; iy_ = Y; iw_ = W; ih_ = H;
}

int Fl_Widget::take_focus() {
  if (!activevisible()) return 0;
  if (!handle(FL_FOCUS)) return 0; // see if it wants it
  if (contains(Fl::focus())) return 1; // it called Fl::focus for us
  Fl::focus(this);
  return 1;
}

extern void fl_throw_focus(Fl_Widget*); // in Fl_x.C

// Destruction does not remove from any parent group!  And groups when
// destroyed destroy all their children.  This is convienent and fast.
// However, it is only legal to destroy a "root" such as an Fl_Window,
// and automatic destructors may be called.
Fl_Widget::~Fl_Widget() {
  parent_ = 0; // kludge to prevent ~Fl_Group from destroying again
  fl_throw_focus(this);
}

void Fl_Widget::activate() {
  if (active()) return;
  clear_flag(INACTIVE);
  handle(FL_ACTIVATE);
  if (inside(Fl::focus())) Fl::focus()->take_focus();
}

void Fl_Widget::deactivate() {
  if (!active()) return;
  set_flag(INACTIVE);
  handle(FL_DEACTIVATE);
  fl_throw_focus(this);
}

static void redraw_background(Fl_Widget* w) {
  for (Fl_Widget *p = w->parent(); p; p = p->parent())
    if (p->box() || !p->parent()) {p->redraw(); break;}
}

void Fl_Widget::show() {
  if (visible()) return;
  clear_flag(INVISIBLE);
  handle(FL_SHOW);
  if (inside(Fl::focus())) Fl::focus()->take_focus();
  if (!align() || (align()&FL_ALIGN_INSIDE))
    redraw();
  else
    redraw_background(this);
}

void Fl_Widget::hide() {
  if (!visible()) return;
  set_flag(INVISIBLE);
  handle(FL_HIDE);
  fl_throw_focus(this);
  redraw_background(this);
}

// return true if widget is inside (or equal to) this:
// Returns false for null widgets.
int Fl_Widget::contains(const Fl_Widget *o) const {
  for (; o; o = o->parent_) if (o == this) return 1;
  return 0;
}

////////////////////////////////////////////////////////////////

#include <FL/Fl_Window.H>

void Fl_Widget::redraw() {damage(~0);}

void Fl_Widget::damage(uchar flags) {
  Fl_Widget* o = this;
  for (;;) {
    if (!o->visible()) return;
    o->damage_ |= flags;
    // mark all the parents as having children bad but not bad themselves:
    if (o->type() >= FL_WINDOW) {
      Fl::damage(1); return;
    } else {
      o = o->parent();
      if (!o) return;
      flags = 1;
    }
  }
}

// damage a region
void Fl_Widget::damage(uchar flags, int X, int Y, int W, int H) {
  Fl_Widget* o = this;
  for (;;) {
    if (!o->visible()) return;
    if (o->type() >= FL_WINDOW) {
      ((Fl_Window*)o)->expose(6,X,Y,W,H); return;
    } else {  
      o->damage_ |= flags;
      o = o->parent();
      if (!o) return;
      flags = 1;
    }
  }
}
