// Fl_Scroll.C

// A scrolling area of child widgets.  You can also subclass this so
// that you can draw into the area not occupied by children, so this
// is also an arbitrary "canvas".

// This uses XCopyArea, but only in the easy case where there is no
// damage other than scrolling.  If any portion of the widget is damaged
// then no XCopyArea is done.  This is because it is difficult/impossible
// to determine if the damage is in the source region of the copy.

// This also makes the (incorrect?) assumption that you will not scroll
// and damage a child at the same time.

#include <FL/Fl.H>
#include <FL/Fl_Scroll.H>
#include <FL/fl_draw.H>
#include <FL/x.H>

void Fl_Scroll::draw() {
  fix_slider_order();
  int tw = w() - (vslider.visible() ? vslider.w() : 0);
  int th = h() - (hslider.visible() ? hslider.h() : 0);
  if (damage()==2 || damage()==3) {
    int src_x, src_w, dest_x, clip_x, clip_w;
    if (xposition_ < oldx) {
      src_x = 0;
      dest_x = oldx-xposition_;
      src_w = tw-dest_x;
      clip_x = 0;
      clip_w = dest_x;
    } else {
      src_x = xposition_-oldx;
      dest_x = 0;
      src_w = tw-src_x;
      clip_x = src_w;
      clip_w = tw-clip_x;
    }
    int src_y, src_h, dest_y, clip_y, clip_h;
    if (yposition_ < oldy) {
      src_y = 0;
      dest_y = oldy-yposition_;
      src_h = th-dest_y;
      clip_y = 0;
      clip_h = dest_y;
    } else {
      src_y = yposition_-oldy;
      dest_y = 0;
      src_h = th-src_y;
      clip_y = src_h;
      clip_h = th-clip_y;
    }
    if (src_x == dest_x && src_y == dest_y) goto REDRAW_NONE;
    if (src_w <= 0 || src_h <= 0) goto REDRAW_ALL;
    if (src_x == dest_x) {clip_x = 0; clip_w = tw;}
    else if (src_y == dest_y) {clip_y = 0; clip_h = th;}
    else goto REDRAW_ALL; // we can only do scroll in one direction
    dest_x += x(); src_x += x(); clip_x += x();
    dest_y += y(); src_y += y(); clip_y += y();
#ifdef WIN32
    BitBlt(fl_gc, dest_x, dest_y, src_w, src_h, fl_gc, src_x, src_y, SRCCOPY);
    // need to redraw areas that the source of BitBlt was bad due to
    // overlapped windows (hope it's not as ugly as X):
#else
    XCopyArea(fl_display, fl_window, fl_window, fl_gc,
	      src_x, src_y, src_w, src_h, dest_x, dest_y);
    // we have to sync the display and get the GraphicsExpose events! (sigh)
    for (;;) {
      XEvent e; XWindowEvent(fl_display, fl_window, ExposureMask, &e);
      if (e.type == NoExpose) break;
      // otherwise assumme it is a GraphicsExpose event:
      draw_clip(e.xexpose.x, e.xexpose.y, e.xexpose.width, e.xexpose.height);
      if (!e.xgraphicsexpose.count) break;
    }
#endif
    // now draw the area we know we need to (the new area scrolled in):
    draw_clip(clip_x, clip_y, clip_w, clip_h);
  } else {
  REDRAW_ALL:
    draw_clip(x(),y(),tw,th);
  }
 REDRAW_NONE:
  // because scrollbars were clipped, we need to draw them:
  if (damage() & 128) { // redraw the entire thing:
    draw_child(hslider);
    draw_child(vslider);
  } else {	// only redraw the children that need it:
    update_child(hslider);
    update_child(vslider);
  }
  oldx = xposition_;
  oldy = yposition_;
}

void Fl_Scroll::draw_clip(int X, int Y, int W, int H) {
  fl_clip(X,Y,W,H);
  if (!box()) { // clear area to right and bottom:
    int tx = x()+xposition_+width_;
    if (tx < X+W) {fl_color(color()); fl_rectf(tx,Y,X+W-tx,H);}
    else tx = X+W;
    int ty = y()+yposition_+height_;
    if (ty < Y+H) {fl_color(color()); fl_rectf(X,ty,tx-X,Y+H-ty);}
  }
  Fl_Group::draw();
  fl_pop_clip();
}

void Fl_Scroll::area(int W, int H) {
  static char recurse;
  if (recurse) return;
  recurse = 1;
  if (!(type()&HORIZONTAL)) {W = w()-vslider.w(); xposition_ = 0;}
  if (!(type()&VERTICAL)) {H = h()-hslider.h(); yposition_ = 0;}
  width_ = W;
  height_ = H;
  int tw = w();
  int th = h();
  if (W > tw) {th -= hslider.h();}
  if (H > th) {tw -= vslider.w();}
  if (W > tw) {th = h()-hslider.h();}
  if (tw < w()) vslider.show(); else vslider.hide();
  hslider.resize(x(), y()+h()-hslider.h(), tw, hslider.h());
  if (th < h()) hslider.show(); else hslider.hide();
  if (W < tw)
    position(0,yposition_);
  else if (W-xposition_ < tw)
    position(W-tw, yposition_);
  if (H < th)
    position(xposition_,0);
  else if (H-yposition_ < th)
    position(xposition_, H-th);
  hslider.scrollvalue(xposition(), tw, 0, W);
  vslider.scrollvalue(yposition(), th, 0, H);
  recurse = 0;
}

void Fl_Scroll::resize(int X, int Y, int W, int H) {
  // don't lose the slider sizes:
  int hsliderh = hslider.h();
  int vsliderw = vslider.w();
  // hack, convince Fl_Group to resize in the correct direction:
  int x1, w1;
  if (type() & HORIZONTAL) {
    x1 = X-xposition_;
    w1 = iw();
  } else {
    x1 = X;
    w1 = W;
    if (height_ > H) w1 -= vslider.w();
  }
  int y1, h1;
  if (type() & VERTICAL) {
    y1 = Y-yposition_;
    h1 = ih();
  } else {
    y1 = Y;
    h1 = H;
    if (width_ > W) h1 -= hslider.h();
  }
  Fl_Group::resize(x1,y1,w1,h1);
  // now resize this container to actual area:
  Fl_Widget::resize(X,Y,W,H);
  // and force sliders to correct position:
  hslider.resize(X, Y+H-hsliderh, W, hsliderh);
  vslider.resize(X+W-vsliderw, Y, vsliderw, H);
  area(width_,height_);
}

void Fl_Scroll::position(int X, int Y) {
  int dx = xposition_-X;
  int dy = yposition_-Y;
  if (!dx && !dy) return;
  xposition_ = X;
  yposition_ = Y;
  Fl_Widget*const* a = array();
  for (int i=children(); i--;) {
    Fl_Widget* o = *a++;
    if (o == &hslider || o == &vslider) continue;
    o->position(o->x()+dx, o->y()+dy);
  }
  hslider.scrollvalue(X, w()-(vslider.visible()?vslider.w():0),0, width_);
  vslider.scrollvalue(Y, h()-(hslider.visible()?hslider.h():0),0, height_);
  damage(2);
}

void Fl_Scroll::hslider_cb(Fl_Widget* o, void*) {
  Fl_Scroll* s = (Fl_Scroll*)(o->parent());
  s->position(int(((Fl_Slider*)o)->value()), s->yposition());
}

void Fl_Scroll::vslider_cb(Fl_Widget* o, void*) {
  Fl_Scroll* s = (Fl_Scroll*)(o->parent());
  s->position(s->xposition(), int(((Fl_Slider*)o)->value()));
}

#define SLIDER_SIZE 20

Fl_Scroll::Fl_Scroll(int X,int Y,int W,int H,const char* L)
  : Fl_Group(X,Y,W,H,L), 
    hslider(X,Y+H-SLIDER_SIZE,W,SLIDER_SIZE),
    vslider(X+W-SLIDER_SIZE,Y,SLIDER_SIZE,H) {
  type(BOTH);
  xposition_ = 0;
  yposition_ = 0;
  width_ = 0;
  height_ = 0;
  hslider.type(FL_HORIZONTAL);
  hslider.callback(hslider_cb);
  vslider.callback(vslider_cb);
}

void Fl_Scroll::fix_slider_order() {
  Fl_Widget*const* a = array();
  // initialize by moving sliders to end:
  if (a[children()-1] != &vslider) {
    Fl_Widget** a = (Fl_Widget**)array();
    int i,j; for (i = j = 0; j < children(); j++)
      if (a[j] != &hslider && a[j] != &vslider) a[i++] = a[j];
    a[i++] = &hslider;
    a[i++] = &vslider;
  }
}

void Fl_Scroll::init_area() {
  if (children() < 3) return;
  fix_slider_order();
  Fl_Widget*const* a = array();
  Fl_Widget* o = *a++;
  int l = o->x();
  int r = l+o->w();
  int t = o->y();
  int b = t+o->h();
  for (int i=children()-3; i--;) {
    o = *a++;
    if (o->x() < l) l = o->x();
    if (o->y() < t) t = o->y();
    if (o->x()+o->w() > r) r = o->x()+o->w();
    if (o->y()+o->h() > b) b = o->y()+o->h();
  }
  if (l > x()) l = x();
  xposition_ = -(l-x());
  if (t > y()) t = y();
  yposition_ = -(t-y());
  area(r-l, b-t);
  resize(x(),y(),w(),h()); // turn scrollbars on/off
}

int Fl_Scroll::handle(int event) {
  if (!width_) init_area(); else fix_slider_order();
  return Fl_Group::handle(event);
}

// end of Fl_Scroll.C
