// Fl_Pixmap.C

// Draws X pixmap data, keeping it stashed in a server pixmap so it
// redraws fast.

// See fl_draw_pixmap.C for code used to get the actual data into pixmap.
// Implemented without using the xpm library (which I can't use because
// it interferes with the color cube used by fl_draw_image).

#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include <FL/x.H>
#include <FL/Fl_Widget.H>
#include <FL/Fl_Menu.H>
#include <FL/Fl_Pixmap.H>

void Fl_Pixmap::draw(int X, int Y, int W, int H, int cx,int cy) {
  if (w<0) fl_measure_pixmap(data, w, h);
  if (!w) return;
  // clip the box down to the size of image, quit if empty:
  if (cx < 0) {W += cx; X -= cx; cx = 0;}
  if (cx+W > w) W = w-cx;
  if (W <= 0) return;
  if (cy < 0) {H += cy; Y -= cy; cy = 0;}
  if (cy+H > h) H = h-cy;
  if (H <= 0) return;
  if (!id) {
    id = (ulong)fl_create_offscreen(w, h);
    fl_begin_offscreen((Fl_Offscreen)id);
    fl_draw_pixmap(data, 0, 0, bg);
    fl_end_offscreen();
  }
  fl_copy_offscreen(X, Y, W, H, (Fl_Offscreen)id, cx, cy);
}

Fl_Pixmap::~Fl_Pixmap() {
  if (id) fl_delete_offscreen((Fl_Offscreen)id);
}

static void pixmap_labeltype(
    const Fl_Label* o, int x, int y, int w, int h, uchar a)
{
  Fl_Pixmap* b = (Fl_Pixmap*)(o->value);
  if (b->w<0) fl_measure_pixmap(b->data, b->w, b->h);
  int cx;
  if (a & FL_ALIGN_LEFT) cx = 0;
  else if (a & FL_ALIGN_RIGHT) cx = b->w-w;
  else cx = (b->w-w)/2;
  int cy;
  if (a & FL_ALIGN_TOP) cy = 0;
  else if (a & FL_ALIGN_BOTTOM) cy = b->h-h;
  else cy = (b->h-h)/2;
  b->draw(x,y,w,h,cx,cy);
}

static void pixmap_measure(const Fl_Label* o, int& w, int& h) {
  Fl_Pixmap* b = (Fl_Pixmap*)(o->value);
  if (b->w<0) fl_measure_pixmap(b->data, b->w, b->h);
  w = b->w;
  h = b->h;
}

void Fl_Pixmap::label(Fl_Widget* o) {
  bg = o->color();
  Fl::set_labeltype(_FL_PIXMAP_LABEL, pixmap_labeltype, pixmap_measure);
  o->label(_FL_PIXMAP_LABEL, (const char*)this);
}

void Fl_Pixmap::label(Fl_Menu* o) {
  Fl::set_labeltype(_FL_PIXMAP_LABEL, pixmap_labeltype, pixmap_measure);
  o->label(_FL_PIXMAP_LABEL, (const char*)this);
}
