// Fl_Gl_Overlay.C

// Methods on Fl_Gl_Window that create an overlay window.  Because
// many programs don't need the overlay, this is seperated into this
// source file so it is not linked in if not used.

// Under X this is done by creating another window, of class _Fl_Gl_Overlay
// which is a subclass of Fl_Gl_Window except it uses the overlay planes.
// A pointer to this is stored in the "overlay" pointer of the Fl_Gl_Window.

// Under win32 another GLX context is created to draw into the overlay
// and it is stored in into the "overlay" pointer.

// In both cases if overlay hardware is unavailable, the overlay is
// "faked" by drawing into the main layers.  This is indicated by
// setting overlay == this.

#include <config.h>
#if HAVE_GL

#include <FL/Fl.H>
#include <FL/Fl_Gl_Window.H>
#include <FL/x.H>
#include "Fl_Gl_Choice.H"
#include <stdlib.h>

extern GLXContext fl_first_context;

#if HAVE_GL_OVERLAY
#ifndef WIN32

static int OVERLAY[] = {GLX_LEVEL, 1, None};

class _Fl_Gl_Overlay : public Fl_Gl_Window {
  void draw();
public:
  void show();
  _Fl_Gl_Overlay(int x, int y, int w, int h) :
    Fl_Gl_Window(x,y,w,h) {mode(OVERLAY); deactivate();}
};

void _Fl_Gl_Overlay::draw() {
  glClear(GL_COLOR_BUFFER_BIT);
  Fl_Gl_Window *w = (Fl_Gl_Window *)parent();
  uchar save_valid = w->valid_;
  w->valid_ = valid_;
  w->draw_overlay();
  valid_ = w->valid_;
  w->valid_ = save_valid;
}

void _Fl_Gl_Overlay::show() {
  int newcmap = (!g || !g->colormap);
  fl_background_pixel = 0;
  Fl_Gl_Window::show();
  fl_background_pixel = -1;
  if (newcmap) {
    fl_alloc_color(g->colormap, overlay_color);
    fl_alloc_color(g->colormap, FL_BLACK);
    fl_alloc_color(g->colormap, FL_WHITE);
  }
}

#endif
#endif

////////////////////////////////////////////////////////////////

#if HAVE_GL_OVERLAY
#ifdef WIN32
static int no_overlay_hardware;
int Fl_Gl_Window::can_do_overlay() {
  if (no_overlay_hardware) return 0;
  // need to write a test here...
  return 1;
}
#else // X version:
int Fl_Gl_Window::can_do_overlay() {
  return can_do(OVERLAY);
}
#endif
#else
int Fl_Gl_Window::can_do_overlay() {return 0;}
#endif

void Fl_Gl_Window::make_overlay() {
  if (!overlay) {
#if HAVE_GL_OVERLAY
#ifdef WIN32
    if (!no_overlay_hardware) {
      HDC hdc = fl_private_dc(this, mode_,&g);
      GLXContext context = wglCreateLayerContext(hdc, 1);
      if (!context) { // no overlay hardware
	no_overlay_hardware = 1;
      } else {
	static CONST COLORREF pcr[] = {0x0000ff, 0x000000, 0xffffff};
	wglSetLayerPaletteEntries(hdc, 1, 1, 3, pcr);
	wglRealizeLayerPalette(hdc, 1, TRUE);
	if (fl_first_context) wglShareLists(fl_first_context, context);
	else fl_first_context = context;
	overlay = context;
	valid(0);
	return;
      }
    }
#else
    if (can_do_overlay()) {
      _Fl_Gl_Overlay* o = new _Fl_Gl_Overlay(0,0,w(),h());
      overlay = o;
      add_resizable(*o);
      o->show();
      return;
    }
#endif
#endif
    overlay = this; // fake the overlay
  }
}

void Fl_Gl_Window::redraw_overlay() {
  if (!shown()) return;
  make_overlay();
#ifndef WIN32
  if (overlay != this)
    ((Fl_Gl_Window*)overlay)->redraw();
  else
#endif
    damage(8);
}

void Fl_Gl_Window::make_overlay_current() {
  make_overlay();
#if HAVE_GL_OVERLAY
  if (overlay != this) {
#ifdef WIN32
    wglMakeCurrent(Fl_X::i(this)->private_dc, (GLXContext)overlay);
#else
    Fl_Gl_Window* w = (Fl_Gl_Window*)overlay;
    w->make_current();
#endif
  } else
#endif
    glDrawBuffer(GL_FRONT);
}

void Fl_Gl_Window::hide_overlay() {
#if HAVE_GL_OVERLAY
#ifdef WIN32
  // nothing needs to be done?  Or should it be erased?
#else
  if (overlay && overlay!=this) ((Fl_Gl_Window*)overlay)->hide();
#endif
#endif
}

uchar Fl_Gl_Window::overlay_color = 1;

#endif
