/* 
 * The Python Imaging Library
 * $Id: Convert.c,v 1.6 1996/08/15 13:12:44 fredrik Exp $
 * 
 * convert images
 *
 * history:
 * 95-06-15 fl	created
 * 95-11-28 fl	added some "RGBA" and "CMYK" conversions
 * 96-04-22 fl	added "1" conversions (same as "L")
 * 96-05-05 fl	added palette conversions (hack)
 * 96-07-23 fl	fixed "1" conversions to zero/non-zero convention
 * 96-11-01 fl	fixed "P" to "L" and "RGB" to "1" conversions
 * 96-12-29 fl	set alpha byte in RGB converters
 * 97-05-12 fl	added ImagingConvert2
 * 97-05-30 fl	added floating point support
 * 97-08-27 fl	added "P" to "1" and "P" to "F" conversions
 * 98-01-11 fl	added integer support
 * 98-07-01 fl	added "YCbCr" support
 * 98-07-02 fl	added "RGBX" conversions (sort of)
 * 98-07-04 fl	added floyd-steinberg dithering
 * 98-07-12 fl	changed YCrCb to YCbCr (!)
 *
 * Copyright (c) Secret Labs AB 1997-98.
 * Copyright (c) Fredrik Lundh 1995-97.
 *
 * See the README file for details on usage and redistribution.
 */


#include "Imaging.h"

#define CLIP(v) ((v) <= 0 ? 0 : (v) >= 255 ? 255 : (v))

/* ITU-R Recommendation 601-2 (assuming nonlinear RGB) */
#define L(rgb)\
    ((INT32) (rgb)[0] * 299 + (INT32) (rgb)[1] * 587 + (INT32) (rgb)[2] * 114)

/* ------------------- */
/* 1 (bit) conversions */
/* ------------------- */

static void
bit2l(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++)
	out[x] = (in[x] != 0) ? 255 : 0;
}

static void
bit2rgb(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4) {
	out[0] = out[1] = out[2] = (in[x] != 0) ? 255 : 0;
	out[3] = 255;
    }
}

static void
bit2cmyk(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4)
	out[0] = out[1] = out[2] = 0, (in[x] != 0) ? 0 : 255;
}

static void
bit2ycbcr(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4) {
	out[0] = (in[x] != 0) ? 255 : 0;
        out[1] = out[2] = 128;
        out[3] = 255;
    }
}

/* ----------------- */
/* RGB/L conversions */
/* ----------------- */

static void
l2bit(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++)
	out[x] = (in[x] >= 128) ? 255 : 0;
}

static void
l2rgb(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4) {
	out[0] = out[1] = out[2] = in[x];
	out[3] = 255;
    }
}

static void
rgb2bit(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, in += 4)
	/* ITU-R Recommendation 601-2 (assuming nonlinear RGB) */
	out[x] = (L(in) >= 128000) ? 255 : 0;
}

static void
rgb2l(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, in += 4)
	/* ITU-R Recommendation 601-2 (assuming nonlinear RGB) */
	out[x] = L(in) / 1000;
}

static void
rgb2i(UINT8* out_, const UINT8* in, int xsize)
{
    int x;
    INT32* out = (INT32*) out_;
    for (x = 0; x < xsize; x++, in += 4)
	out[x] = L(in) / 1000;
}

static void
rgb2f(UINT8* out_, const UINT8* in, int xsize)
{
    int x;
    FLOAT32* out = (FLOAT32*) out_;
    for (x = 0; x < xsize; x++, in += 4)
	out[x] = L(in) / 1000.0F;
}


/* ---------------- */
/* RGBA conversions */
/* ---------------- */

static void
rgb2rgba(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, in += 4, out += 4)
	out[0] = in[0], out[1] = in[1], out[2] = in[2], out[3] = 255;
}


static void
rgba2rgb(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, in += 4, out += 4) {
	out[0] = in[0], out[1] = in[1], out[2] = in[2]; /* FIXME: memcpy? */
	out[3] = 255;
    }
}


/* ---------------- */
/* CMYK conversions */
/* ---------------- */

static void
l2cmyk(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4)
	out[0] = out[1] = out[2] = 0, out[3] = ~in[x];
}

static void
rgb2cmyk(UINT8* out, const UINT8* in, int xsize)
{
    int i;
    for (i = 0; i < xsize; i++, in += 4, out += 4)
	/* Note: no undercolour removal */
	out[0] = ~in[0], out[1] = ~in[1], out[2] = ~in[2], out[3] = 0;
}

static void
cmyk2rgb(UINT8* out, const UINT8* in, int xsize)
{
    int i;
    for (i = 0; i < xsize; i++, in += 4, out += 4) {
        out[0] = CLIP(255 - (in[0] + in[3]));
	out[1] = CLIP(255 - (in[1] + in[3]));
	out[2] = CLIP(255 - (in[2] + in[3]));
	out[3] = 255;
    }
}

/* ------------- */
/* I conversions */
/* ------------- */

static void
bit2i(UINT8* out_, const UINT8* in, int xsize)
{
    int x;
    INT32* out = (INT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = (in[x] != 0) ? 255 : 0;
}

static void
l2i(UINT8* out_, const UINT8* in, int xsize)
{
    int x;
    INT32* out = (INT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = (INT32) in[x];
}

static void
i2l(UINT8* out, const UINT8* in_, int xsize)
{
    int x;
    INT32* in = (INT32*) in_;
    for (x = 0; x < xsize; x++) {
        if (in[x] <= 0)
            out[x] = 0;
        else if (in[x] >= 255)
            out[x] = 255;
        else
            out[x] = (UINT8) in[x];
    }
}

static void
i2f(UINT8* out_, const UINT8* in_, int xsize)
{
    int x;
    INT32* in = (INT32*) in_;
    FLOAT32* out = (FLOAT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = (FLOAT32) in[x];
}

/* ------------- */
/* F conversions */
/* ------------- */

static void
bit2f(UINT8* out_, const UINT8* in, int xsize)
{
    int x;
    FLOAT32* out = (FLOAT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = (in[x] != 0) ? 255.0 : 0.0;
}

static void
l2f(UINT8* out_, const UINT8* in, int xsize)
{
    int x;
    FLOAT32* out = (FLOAT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = (FLOAT32) in[x];
}

static void
f2l(UINT8* out, const UINT8* in_, int xsize)
{
    int x;
    FLOAT32* in = (FLOAT32*) in_;
    for (x = 0; x < xsize; x++) {
        if (in[x] <= 0.0)
            out[x] = 0;
        else if (in[x] >= 255.0)
            out[x] = 255;
        else
            out[x] = (UINT8) in[x];
    }
}

static void
f2i(UINT8* out_, const UINT8* in_, int xsize)
{
    int x;
    FLOAT32* in = (FLOAT32*) in_;
    INT32* out = (INT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = (INT32) in[x];
}

/* ----------------- */
/* YCbCr conversions */
/* ----------------- */

/* See ConvertYCbCr.c for RGB/YCbCr tables */

static void
l2ycbcr(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4) {
	out[0] = in[x];
	out[1] = out[2] = 128;
	out[3] = 255;
    }
}

static void
ycbcr2l(UINT8* out, const UINT8* in, int xsize)
{
    int x;
    for (x = 0; x < xsize; x++, in += 4)
	out[x] = in[0];
}

static struct {
    const char* from;
    const char* to;
    ImagingShuffler convert;
} converters[] = {

    { "1", "L", bit2l },
    { "1", "I", bit2i },
    { "1", "F", bit2f },
    { "1", "RGB", bit2rgb },
    { "1", "RGBA", bit2rgb },
    { "1", "RGBX", bit2rgb },
    { "1", "CMYK", bit2cmyk },
    { "1", "YCbCr", bit2ycbcr },

    { "L", "1", l2bit },
    { "L", "I", l2i },
    { "L", "F", l2f },
    { "L", "RGB", l2rgb },
    { "L", "RGBA", l2rgb },
    { "L", "RGBX", l2rgb },
    { "L", "CMYK", l2cmyk },
    { "L", "YCbCr", l2ycbcr },

    { "I",    "L",    i2l },
    { "I",    "F",    i2f },

    { "F",    "L",    f2l },
    { "F",    "I",    f2i },

    { "RGB", "1", rgb2bit },
    { "RGB", "L", rgb2l },
    { "RGB", "I", rgb2i },
    { "RGB", "F", rgb2f },
    { "RGB", "RGBA", rgb2rgba },
    { "RGB", "RGBX", rgb2rgba },
    { "RGB", "CMYK", rgb2cmyk },
    { "RGB", "YCbCr", ImagingConvertRGB2YCbCr },

    { "RGBA", "1", rgb2bit },
    { "RGBA", "L", rgb2l },
    { "RGBA", "I", rgb2i },
    { "RGBA", "F", rgb2f },
    { "RGBA", "RGB", rgba2rgb },
    { "RGBA", "RGBX", rgb2rgba },
    { "RGBA", "CMYK", rgb2cmyk },
    { "RGBA", "YCbCr", ImagingConvertRGB2YCbCr },

    { "RGBX", "1", rgb2bit },
    { "RGBX", "L", rgb2l },
    { "RGBA", "I", rgb2i },
    { "RGBA", "F", rgb2f },
    { "RGBX", "RGB", rgba2rgb },
    { "RGBX", "CMYK", rgb2cmyk },
    { "RGBX", "YCbCr", ImagingConvertRGB2YCbCr },

    { "CMYK", "RGB",  cmyk2rgb },
    { "CMYK", "RGBA", cmyk2rgb },
    { "CMYK", "RGBX", cmyk2rgb },

    { "YCbCr", "L", ycbcr2l },
    { "YCbCr", "RGB", ImagingConvertYCbCr2RGB },

    { NULL }
};


/* ------------------- */
/* Palette conversions */
/* ------------------- */

static void
p2bit(UINT8* out, const UINT8* in, int xsize, const UINT8* palette)
{
    int x;
    /* FIXME: precalculate greyscale palette? */
    for (x = 0; x < xsize; x++)
	out[x] = (L(&palette[in[x]*4]) >= 1000) ? 255 : 0;
}

static void
p2l(UINT8* out, const UINT8* in, int xsize, const UINT8* palette)
{
    int x;
    /* FIXME: precalculate greyscale palette? */
    for (x = 0; x < xsize; x++)
	out[x] = L(&palette[in[x]*4]) / 1000;
}

static void
p2i(UINT8* out_, const UINT8* in, int xsize, const UINT8* palette)
{
    int x;
    INT32* out = (INT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = L(&palette[in[x]*4]) / 1000;
}

static void
p2f(UINT8* out_, const UINT8* in, int xsize, const UINT8* palette)
{
    int x;
    FLOAT32* out = (FLOAT32*) out_;
    for (x = 0; x < xsize; x++)
	out[x] = L(&palette[in[x]*4]) / 1000.0F;
}

static void
p2rgb(UINT8* out, const UINT8* in, int xsize, const UINT8* palette)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4) {
	int v = in[x] * 4;
	out[0] = palette[v];
	out[1] = palette[v+1];
	out[2] = palette[v+2];
	out[3] = 255;
    }
}

static void
p2rgba(UINT8* out, const UINT8* in, int xsize, const UINT8* palette)
{
    int x;
    for (x = 0; x < xsize; x++, out += 4) {
	int v = in[x] * 4;
	out[0] = palette[v];
	out[1] = palette[v+1];
	out[2] = palette[v+2];
	out[3] = palette[v+3];
    }
}

static void
p2cmyk(UINT8* out, const UINT8* in, int xsize, const UINT8* palette)
{
    p2rgb(out, in, xsize, palette);
    rgb2cmyk(out, out, xsize);
}

static void
p2ycbcr(UINT8* out, const UINT8* in, int xsize, const UINT8* palette)
{
    p2rgb(out, in, xsize, palette);
    ImagingConvertRGB2YCbCr(out, out, xsize);
}

static Imaging
frompalette(Imaging imOut, Imaging imIn, const char *mode)
{
    int y;
    void (*convert)(UINT8*, const UINT8*, int, const UINT8*);

    /* Map palette image to L, RGB, RGBA, or CMYK */

    if (!imIn->palette)
	return (Imaging) ImagingError_Argument("No palette");

    if (strcmp(mode, "1") == 0)
	convert = p2bit;
    else if (strcmp(mode, "L") == 0)
	convert = p2l;
    else if (strcmp(mode, "I") == 0)
	convert = p2i;
    else if (strcmp(mode, "F") == 0)
	convert = p2f;
    else if (strcmp(mode, "RGB") == 0)
	convert = p2rgb;
    else if (strcmp(mode, "RGBA") == 0)
	convert = p2rgba;
    else if (strcmp(mode, "RGBX") == 0)
	convert = p2rgba;
    else if (strcmp(mode, "CMYK") == 0)
	convert = p2cmyk;
    else if (strcmp(mode, "YCbCr") == 0)
	convert = p2ycbcr;
    else
	return (Imaging) ImagingError_Argument("Illegal conversion");

    imOut = ImagingNew2(mode, imOut, imIn);
    if (!imOut)
        return NULL;

    for (y = 0; y < imIn->ysize; y++)
	(*convert)((UINT8*) imOut->image[y], (UINT8*) imIn->image[y],
		   imIn->xsize, imIn->palette->palette);

    return imOut;
}

static Imaging
topalette(Imaging imOut, Imaging imIn, ImagingPalette palette, int dither)
{
    int x, y;

    /* Map L or RGB/RGBX/RGBA to palette image */

    if (strcmp(imIn->mode, "L") != 0 && strncmp(imIn->mode, "RGB", 3) != 0)
	return (Imaging) ImagingError_Argument("Illegal conversion");

    /* FIXME: make user configurable */
    if (imIn->bands == 1)
	palette = ImagingPaletteNew("RGB"); /* Initialised to grey ramp */
    else
	palette = ImagingPaletteNewBrowser(); /* Standard colour cube */

    if (!palette)
	return (Imaging) ImagingError_Argument("No palette");

    imOut = ImagingNew2("P", imOut, imIn);
    if (!imOut) {
	ImagingPaletteDelete(palette);
        return NULL;
    }

    imOut->palette = ImagingPaletteDuplicate(palette);

    if (imIn->bands == 1) {
	/* greyscale image */

	/* Greyscale palette: copy data as is */
	for (y = 0; y < imIn->ysize; y++)
	    memcpy(imOut->image[y], imIn->image[y], imIn->linesize);

    } else {
	/* colour image */

	/* Create mapping cache */
	if (ImagingPaletteCachePrepare(palette) < 0) {
	    ImagingDelete(imOut);
	    ImagingPaletteDelete(palette);
	    return NULL;
	}

        if (dither) {
            /* floyd-steinberg dither */

            int* errors;
            errors = calloc(imIn->xsize + 1, sizeof(int) * 3);
            if (!errors) {
                ImagingDelete(imOut);
                ImagingError_NoMemory();
                return NULL;
            }

            /* Map each pixel to the nearest palette entry */
            for (y = 0; y < imIn->ysize; y++) {
                int r, r0, r1, r2;
                int g, g0, g1, g2;
                int b, b0, b1, b2;
                UINT8* in  = (UINT8*) imIn->image[y];
                UINT8* out = imOut->image8[y];
                int* e = errors;

                r = r0 = r1 = 0;
                g = g0 = g1 = 0;
                b = b0 = b1 = 0;

                for (x = 0; x < imIn->xsize; x++, in += 4) {
                    int d2;
                    INT16* cache;

                    r = CLIP(in[0] + (r + e[3+0])/16);
                    g = CLIP(in[1] + (g + e[3+1])/16);
                    b = CLIP(in[2] + (b + e[3+2])/16);

                    /* get closest colour */
                    cache = &ImagingPaletteCache(palette, r, g, b);
                    if (cache[0] == 0x100)
                        ImagingPaletteCacheUpdate(palette, r, g, b);
                    out[x] = cache[0];

                    r -= (int) palette->palette[cache[0]*4];
                    g -= (int) palette->palette[cache[0]*4+1];
                    b -= (int) palette->palette[cache[0]*4+2];

                    /* propagate errors (don't ask ;-)*/
                    r2 = r; d2 = r + r; r += d2; e[0] = r + r0;
                    r += d2; r0 = r + r1; r1 = r2; r += d2;
                    g2 = g; d2 = g + g; g += d2; e[1] = g + g0;
                    g += d2; g0 = g + g1; g1 = g2; g += d2;
                    b2 = b; d2 = b + b; b += d2; e[2] = b + b0;
                    b += d2; b0 = b + b1; b1 = b2; b += d2;

                    e += 3;

                }

                e[0] = b0;
                e[1] = b1;
                e[2] = b2;

            }
            free(errors);

        } else {

            /* closest colour */
            for (y = 0; y < imIn->ysize; y++) {
                int r, g, b;
                UINT8* in  = (UINT8*) imIn->image[y];
                UINT8* out = imOut->image8[y];

                for (x = 0; x < imIn->xsize; x++, in += 4) {
                    INT16* cache;

                    r = in[0]; g = in[1]; b = in[2];

                    /* get closest colour */
                    cache = &ImagingPaletteCache(palette, r, g, b);
                    if (cache[0] == 0x100)
                        ImagingPaletteCacheUpdate(palette, r, g, b);
                    out[x] = cache[0];

                }
            }
        }
	ImagingPaletteCacheDelete(palette);
    }

    ImagingPaletteDelete(palette);

    return imOut;
}

static Imaging
tobilevel(Imaging imOut, Imaging imIn, int dither)
{
    int x, y;
    int* errors;

    /* Map L or RGB to dithered 1 image */
    if (strcmp(imIn->mode, "L") != 0 && strcmp(imIn->mode, "RGB") != 0)
	return (Imaging) ImagingError_Argument("Illegal conversion");

    imOut = ImagingNew2("1", imOut, imIn);
    if (!imOut)
        return NULL;

    errors = calloc(imIn->xsize + 1, sizeof(int));
    if (!errors) {
        ImagingDelete(imOut);
        ImagingError_NoMemory();
        return NULL;
    }

    if (imIn->bands == 1) {

        /* map each pixel to black or white, using error diffusion */
        for (y = 0; y < imIn->ysize; y++) {
            int l, l0, l1, l2, d2;
            UINT8* in  = (UINT8*) imIn->image[y];
            UINT8* out = imOut->image8[y];

            l = l0 = l1 = 0;

            for (x = 0; x < imIn->xsize; x++) {

                /* pick closest colour */
                l = CLIP(in[x] + (l + errors[x+1])/16);
                out[x] = (l > 128) ? 255 : 0;

                /* propagate errors */
                l -= (int) out[x];
                l2 = l; d2 = l + l; l += d2; errors[x] = l + l0;
                l += d2; l0 = l + l1; l1 = l2; l += d2;
            }

            errors[x] = l0;

        }

    } else {

        /* map each pixel to black or white, using error diffusion */
        for (y = 0; y < imIn->ysize; y++) {
            int l, l0, l1, l2, d2;
            UINT8* in  = (UINT8*) imIn->image[y];
            UINT8* out = imOut->image8[y];

            l = l0 = l1 = 0;

            for (x = 0; x < imIn->xsize; x++, in += 4) {

                /* pick closest colour */
                l = CLIP(L(in)/1000 + (l + errors[x+1])/16);
                out[x] = (l > 128) ? 255 : 0;

                /* propagate errors */
                l -= (int) out[x];
                l2 = l; d2 = l + l; l += d2; errors[x] = l + l0;
                l += d2; l0 = l + l1; l1 = l2; l += d2;

            }

            errors[x] = l0;

        }
    }

    free(errors);

    return imOut;
}



Imaging
_convert(Imaging imOut, Imaging imIn, const char *mode,
         ImagingPalette palette, int dither)
{
    int y;
    ImagingShuffler convert;

    if (!imIn)
	return (Imaging) ImagingError_Argument(NULL);

    if (!mode) {
	/* Map palette image to full depth */
	if (!imIn->palette)
	    return (Imaging) ImagingError_Argument("Not a palette image");
	mode = imIn->palette->mode;
    } else
	/* Same mode? */
	if (!strcmp(imIn->mode, mode))
	    return ImagingCopy2(imOut, imIn);


    /* test for special conversions */

    if (strcmp(imIn->mode, "P") == 0)
	return frompalette(imOut, imIn, mode);

    if (strcmp(mode, "P") == 0)
	return topalette(imOut, imIn, NULL, dither);

    if (dither && strcmp(mode, "1") == 0)
	return tobilevel(imOut, imIn, dither);


    /* standard conversion machinery */

    convert = NULL;

    for (y = 0; converters[y].from; y++)
	if (!strcmp(imIn->mode, converters[y].from) &&
	    !strcmp(mode, converters[y].to)) {
	    convert = converters[y].convert;
	    break;
	}

    if (!convert)
	return (Imaging) ImagingError_Argument("Illegal conversion");

    imOut = ImagingNew2(mode, imOut, imIn);
    if (!imOut)
        return NULL;

    for (y = 0; y < imIn->ysize; y++)
	(*convert)((UINT8*) imOut->image[y], (UINT8*) imIn->image[y],
		   imIn->xsize);

    return imOut;
}

Imaging
ImagingConvert(Imaging imIn, const char *mode,
               ImagingPalette palette, int dither)
{
    return _convert(NULL, imIn, mode, palette, dither);
}

Imaging
ImagingConvert2(Imaging imOut, Imaging imIn)
{
    return _convert(imOut, imIn, imOut->mode, NULL, 0);
}
