/*
 * rd_hspice - analog waveform file reader for the transient-simulation
 * format (.tr*) written by the HSPICE simulator.   Turns out that it works
 * without change on sweep simulation output (.sw*) files also.
 *
 * Steve Tell, August 28, 1998.
 *
 * $Log: rd_hspice.c,v $
 * Revision 1.4  1998/09/17 22:39:12  tell
 * Attemted fixes for reading files generated by hspice decks without
 * ".options probe"  Works for .ac files, breaks the others.
 *
 * Revision 1.3  1998/09/17 18:24:36  tell
 * add backpointer from DataSet to DataFile
 *
 * Revision 1.2  1998/09/10 21:05:32  tell
 * added reader for binary hspice files
 * factored out common code into hs_process_header
 *
 * Revision 1.1  1998/08/31 20:56:37  tell
 * Initial revision
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <math.h>
#include <glib.h>
#include "reader.h"

/* given a string of ascii header information, set up the
 * DataFile structure appropriately.
 * Returns NULL on failure.
 */
static DataFile *
hs_process_header(int ndv, char *line, char *name)
{
	char *cp;
	int ivtype;
	int *dvtypes;
	char *signam;
	DataFile *df = NULL;
	int i;

	cp = strtok(line, " \t\n");
	if(!cp) {
		if(analog_read_debug)
			fprintf(stderr, "%s: initial vartype not found on header line.\n", name);

		return NULL;
	}
      
	dvtypes = g_new(int, ndv);
	for(i = 0; i < ndv; i++) { /* numbers indicate variable type */
		int hstype;

		cp = strtok(NULL, " \t\n");
		if(!cp) {
			if(analog_read_debug)
				fprintf(stderr, "%s: not enough vartypes on header line\n", name);
			return NULL;
		}
		if(!isdigit(cp[0])) {
			if(analog_read_debug)
				fprintf(stderr, "%s: bad vartype %d [%s] on header line\n", name, i, cp);

			return NULL;
		}
		hstype = atoi(cp);
		switch(hstype) {
		case 1:
		case 2:
		case 3:
			dvtypes[i] = VOLTAGE;
			break;
		case 15:
			dvtypes[i] = CURRENT;
			break;
		default:
			dvtypes[i] = UNKNOWN;
			break;
		}
	}
	signam = strtok(NULL, " \t\n"); /* independent variable, usually "TIME" */
	if(!signam) {
		if(analog_read_debug)
			fprintf(stderr, "%s: no IV name found on header line\n", name);
		goto bailout;
	}
	if(strcmp(signam, "TIME") == 0)
		ivtype = TIME;
	else if(strcmp(signam, "VOLTS") == 0)
		ivtype = VOLTAGE;
	else if(strcmp(signam, "HERTZ") == 0)
		ivtype = FREQUENCY;
	else
		ivtype = UNKNOWN;

	df = g_new0(DataFile, 1);
	df->filename = g_strdup(name);
	df->iv = g_new0(IVar, 1);
	an_init_dataset((DataSet *)df->iv, df, signam, ivtype);
	df->dv = g_new0(DVar*, ndv);
	df->ndv = ndv;
	
	for(i = 0; i < ndv; i++) {
		DVar *dv;
		if((signam = strtok(NULL, " \t\n")) == NULL) {
			if(analog_read_debug)
				fprintf(stderr, "%s: not enough DV names found on header line\n", name);
			goto bailout;
		}
		dv = g_new0(DVar, 1);
		dv->iv = df->iv;
		df->dv[i] = dv;

		an_init_dataset((DataSet *)&dv->d, df, signam, dvtypes[i]);
	}
	return df;
 bailout:
	if(dvtypes)
		g_free(dvtypes);
	if(df)
		an_free_datafile(df);

	return NULL;
}

DataFile *
hs_read_file_ascii(char *name, FILE *fp)
{
	DataFile *df = NULL;
	char *line = NULL;
	int nauto, nprobe;
	int ndv;
	int lineno = 0;
	int linesize = 1024;
	int nrows;
	double val;
	char lbuf[256];
	char nbuf[16];
	char *cp;
	int maxlines = 30;	/* should calculate based on # of columns */
	int i;
	int llen;

	if(fgets(lbuf, sizeof(lbuf), fp) == NULL)
		return NULL;
	lineno++;
	
	/* version of post format */
	if(strncmp(&lbuf[16], "9007", 4) != 0 
	   && strncmp(&lbuf[16], "9601", 4) != 0)
		return NULL;
	strncpy(nbuf, &lbuf[0], 4);
	nbuf[4] = 0;
	nauto = atoi(nbuf);
	strncpy(nbuf, &lbuf[4], 4);
	nbuf[4] = 0;
	nprobe = atoi(nbuf);

	ndv = nauto - 1 + nprobe;
	
	if(fgets(lbuf, sizeof(lbuf), fp) == NULL) /* date, time etc. */
		return NULL;
	lineno++;
	if(fgets(lbuf, sizeof(lbuf), fp) == NULL) /* always just a single '0' ?*/
		return NULL;
	lineno++;
	
	/* lines making up a fixed-field structure with variable-types and
	 * variable names.
	 * variable names can get split across lines! so we remove newlines,
	 * paste all of the lines together, and then deal with the
	 * whole header at once.
	 * A variable name of "$&%#" indicates the end!
	 */
	line = g_new0(char, linesize);
	do {
		if(fgets(lbuf, sizeof(lbuf), fp) == NULL)
			return NULL;
		lineno++;
		if((cp = strchr(lbuf, '\n')) != NULL)
			*cp = 0;
		if(llen + strlen(lbuf) > linesize) {
			linesize *= 2;
			line = g_realloc(line, linesize);
		}
		strcat(line, lbuf);

	} while(!strstr(line, "$&%#") && lineno < maxlines);
	if(lineno == maxlines) {
		if(analog_read_debug)
			fprintf(stderr, "%s:%d: end of hspice header not found\n", name,lineno);
		return NULL;
	}

	if(analog_read_debug) {
		fprintf(stderr, "expect %d columns\n", ndv+1);
	}

	df = hs_process_header(ndv, line, name);
	if(!df)
		goto bailout;

	nrows = 0;
	while(fscanf(fp, "%lg", &val) != EOF) {
		if(val >= (1e30 - DBL_EPSILON)) 
			break; /* hspice puts "infinity" at end */
		if(ds_blockno(nrows) >= df->iv->d.bpused) {
			an_expand_dset((DataSet *)df->iv);
			for(i = 0; i < df->ndv; i++)
				an_expand_dset((DataSet *)df->dv[i]);
		}

		an_set_point((DataSet *)&df->iv->d, nrows, val);
		df->iv->d.nvalues++;
		for(i = 0; i < df->ndv; i++) {
			val = 0;
			if(fscanf(fp, "%lg", &val) == NULL) {
				fprintf(stderr, "%s: partial last row (%d)\n", name, nrows);
				return df;
			}
			/* conjecture: the automatic columns are all complex 
			* Fix: proper complex-value handling throughout
			*/
			if(i < nauto-1) {
				double ival;
				if(fscanf(fp, "%lg", &ival) == NULL) {
					fprintf(stderr, "%s: partial last row (%d)\n", name, nrows);
					return df;
				}
				val = sqrt(val*val + ival*ival);
			}

			an_set_point((DataSet *)&df->dv[i]->d, nrows, val);
			df->dv[i]->d.nvalues++;
		}
		nrows++;
	}
	return df;

bailout:  /* TODO: free everything else that might have been allocated */
	if(line)
		g_free(line);
	if(df)
		an_free_datafile(df);
	return NULL;
}

/*
 * Read binary hspice file.
 */
DataFile *
hs_read_file_bin(char *name, FILE *fp)
{
	DataFile *df = NULL;
	char *ahdr = NULL;
	int ahdrsize;
	int datasize;
	int nauto, nprobe;
	int ndv;
	int nrows;
	char nbuf[16];
	float val;
	int i;
	
	unsigned int ibuf[4];
	int expected_vals, actual_vals;

	if(fread(ibuf, sizeof(int), 4, fp) != 4) {
		if(analog_read_debug)
			fprintf(stderr, "EOF reading block1 header\n");
		return NULL;
	}
	if(ibuf[0] != 4 || ibuf[2] != 4) {
		if(analog_read_debug)
			fprintf(stderr, "unexepected values in block1 header\n");
		return NULL;
	}
	ahdrsize = ibuf[3];
	ahdr = g_new(char, ahdrsize+1);
	if(fread(ahdr, sizeof(char), ahdrsize, fp) != ahdrsize) {
		if(analog_read_debug)
			fprintf(stderr, "EOF reading block1\n");
		goto bailout;
	}
	ahdr[ahdrsize] = '\0';
	if(fread(ibuf, sizeof(int), 1, fp) != 1) {
		if(analog_read_debug)
			fprintf(stderr, "EOF reading block1 trailer\n");
		goto bailout;
	}
	if(ibuf[0] != ahdrsize) {
		if(analog_read_debug)
			fprintf(stderr, "block1 trailer mismatch\n");
		goto bailout;
	}
	/* ahdr is an ascii header that describes the variables in
	 * much the same way that the first lines of the ascii format do,
	 * except that there are no newlines
	 */

	if(strncmp(&ahdr[16], "9007", 4) != 0 	/* version of post format */
	   && strncmp(&ahdr[16], "9601", 4) != 0)
		goto bailout;
	strncpy(nbuf, &ahdr[0], 4);
	nbuf[4] = 0;
	nauto = atoi(nbuf);	/* number of automaticly-included columns,
				   first one is independent variable */

	strncpy(nbuf, &ahdr[4], 4);
	nbuf[4] = 0;
	nprobe = atoi(nbuf);	/* number of user-requested columns */

	ndv = nauto - 1 + nprobe;

/*	printf("header: %s\n", ahdr);
	printf("na=%d ndv=%d\n", na, ndv);
	fflush(stdout);
	*/

	df = hs_process_header(ndv, &ahdr[256], name);
	if(!df)
		return NULL;
	
	if(fread(ibuf, sizeof(int), 4, fp) != 4) {
		if(analog_read_debug)
			fprintf(stderr, "EOF reading block2 header\n");
		goto bailout;
	}
	if(ibuf[0] != 4 || ibuf[2] != 4) {
		if(analog_read_debug)
			fprintf(stderr, "unexepected values in block2 header\n");
		goto bailout;
	}

	datasize = ibuf[3];
	expected_vals = datasize / sizeof(float);
	
	if(analog_read_debug) {
		fprintf(stderr, "expect %d columns, %d values; reading block2 at 0x%lx\n",
			ndv+1, expected_vals, ftell(fp));
	}
	actual_vals = 0;
	nrows = 0;
	do {
		/*
		if(analog_read_debug) {
			fprintf(stderr, "row=%d vals=%d offset=0x%lx\n", nrows, actual_vals, ftell(fp));
		}
		*/
		
		if(fread(&val, sizeof(float), 1, fp) != 1) {
			if(analog_read_debug)
				fprintf(stderr, "unexepected EOF in data\n");
			goto bailout;
		}
		actual_vals++;
		if(val >= (1e30 - DBL_EPSILON))
			break;
		if(ds_blockno(nrows) >= df->iv->d.bpused) {
			an_expand_dset((DataSet *)df->iv);
			for(i = 0; i < df->ndv; i++)
				an_expand_dset((DataSet *)df->dv[i]);
		}

		an_set_point((DataSet *)&df->iv->d, nrows, val);
		df->iv->d.nvalues++;
		for(i = 0; i < df->ndv; i++) {
			val = 0;
			if(fread(&val, sizeof(float), 1, fp) != 1) {
				if(analog_read_debug)
					fprintf(stderr, "unexepected EOF in data\n");
				goto bailout;
			}
			actual_vals++;
			if(i < nauto-1) {
				float ival;
				if(fread(&ival, sizeof(float), 1, fp) != 1) {
					if(analog_read_debug)
						fprintf(stderr, "unexepected EOF in data\n");
					goto bailout;
				}
				val = sqrt(val*val + ival*ival);
				actual_vals++;
			}
			an_set_point((DataSet *)&df->dv[i]->d, nrows, val);
			df->dv[i]->d.nvalues++;
		}
		nrows++;
	} while(actual_vals < expected_vals);

	if(analog_read_debug) {
		fprintf(stderr,
			"actual_vals=%d trying for block2 trailer at 0x%lx\n",
			actual_vals, ftell(fp));
	}
	if(fread(ibuf, sizeof(int), 1, fp) != 1) {
		if(analog_read_debug)
			fprintf(stderr, "EOF reading block2 trailer\n");
		goto bailout;
	}
	if(ibuf[0] != datasize) {
		if(analog_read_debug)
			fprintf(stderr, "block2 trailer mismatch\n");
		goto bailout;
	}
	
	return df;

 bailout:
	if(ahdr)
		g_free(ahdr);
	an_free_datafile(df);
	return NULL;
}


