// Static initializers and templates
// BZ_TEMPLATE_STATIC_INIT

#include <iostream.h>

// Forward declaration of MemoryBlockReference
template<class T_type> class MemoryBlockReference;

int freed = 0;

// Class MemoryBlock provides a reference-counted block of memory.  This block
// may be referred to by multiple vector, matrix and array objects.  The memory
// is automatically deallocated when the last referring object is destructed.
// MemoryBlock may be subclassed to provide special allocators.

template<class P_type>
class MemoryBlock {

    friend class MemoryBlockReference<P_type>;

public:
    typedef P_type T_type;

protected:
    MemoryBlock()
    {
        length_ = 0;
        data_ = 0;
        dataBlockAddress_ = 0;
        references_ = 0;
        ownData_ = 0;
    }

    explicit MemoryBlock(size_t items)
    {
        length_ = items;
        allocate(length_);
        references_ = 0;
        ownData_ = 1;
    }

    MemoryBlock(size_t length, T_type*  data)
    {
        length_ = length;
        data_ = data;
        dataBlockAddress_ = 0;
        references_ = 0;
        ownData_ = 0;
    }

    virtual ~MemoryBlock()
    {
        if ((dataBlockAddress_) && (ownData_))
        {
            delete [] dataBlockAddress_;
        }
    }

    void          addReference()
    { 
        ++references_; 
    }

    T_type*       data() 
    { 
        return data_; 
    }

    const T_type*  data()      const
    { 
        return data_; 
    }

    size_t        length()    const
    { 
        return length_; 
    }

    void          removeReference()
    {
        --references_;

        if (!references_)
            ++freed;
    }

    int references() const
    {
        return references_;
    }

protected:
    inline void allocate(int length);

private:   // Disabled member functions
    MemoryBlock(const MemoryBlock<T_type>&)
    { }

    void operator=(const MemoryBlock<T_type>&)
    { }

private:   // Data members
    T_type *  data_;
    T_type *  dataBlockAddress_;
    int     references_;
    size_t  length_;
    int     ownData_;
};

template<class P_type>
class NullMemoryBlock : public MemoryBlock<P_type> {
public:
    NullMemoryBlock()
    { 
        // This ensures that the delete operator will not be invoked
        // on an instance of NullMemoryBlock in removeReference().
        this->addReference();        
    }

    virtual ~NullMemoryBlock()  
    { }
};

template<class P_type>
class MemoryBlockReference {

public:
    typedef P_type T_type;

    MemoryBlockReference()
    {
        block_ = &nullBlock_;
        block_->addReference();
        data_ = 0;
    }

   ~MemoryBlockReference()
    {
        block_->removeReference();
    }

    int numReferences() const
    {
        return block_->references();
    }

protected:

    void newBlock(size_t items)
    {
        block_->removeReference();
        block_ = new MemoryBlock<T_type>(items);
        block_->addReference();
        data_ = block_->data();
    }

private:
    void operator=(const MemoryBlockReference<T_type>&)
    { }

private:
    MemoryBlock<T_type>* block_;
    static NullMemoryBlock<T_type> nullBlock_;
    
protected:
    T_type *  data_;
};


// Null memory block for each (template) instantiation of MemoryBlockReference
template<class P_type> 
NullMemoryBlock<P_type> MemoryBlockReference<P_type>::nullBlock_;

template<class P_type>
inline void MemoryBlock<P_type>::allocate(int length)
{
    data_ =  new T_type[length];
    dataBlockAddress_ = data_;
}

template<class T>
class Array : protected MemoryBlockReference<T> {
public:
    Array(int n)
    {
        MemoryBlockReference<T>::newBlock(n);
    }
};

Array<float> x(10);

int main()
{
    return freed;
}

