#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <locale.h>

/* getopt is contained in unistd If we have no unistd, 
   we can try to include getopt */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#elif HAVE_GETOPT_H
#include <getopt.h>
#endif

#include "mymain.h"
#include "colors.h"

#ifndef COMPRESSION
#define COMPRESSION 0
#endif

/* exit values */
#define CANNOT_READ_INPUT   1
#define CANNOT_WRITE_OUTPUT 2
#define PRINTED_USAGE       3
#define GZIP_NOT_FOUND      4

/* global variables */
FILE *actin;
FILE *actout;
config_type config;

int
MyMain(int argc, char *argv[])
{
  char *outfilename      = (char *) NULL;
  char *temp_ch_ptr      = (char *) NULL;
  int  i                 = 0;
  int  parsed_parameters = 0;
  int  return_code       = 0;

  (void) setlocale(LC_TIME, "C");
  /* initialize configuration with defaults */
  config.nocgi      = 0;
  config.width      = 80;
  config.title      = (char *) NULL;
  temp_ch_ptr       = strrchr(argv[0], '/');
  config.prog       = (temp_ch_ptr)? ++temp_ch_ptr : argv[0]; /* basename */
  temp_ch_ptr       = (char *) NULL;
  config.headfile   = (char *) NULL; 
  config.bottomfile = (char *) NULL; 

  /* initialize configuration with given parameters */
  parsed_parameters = ParseParameters(argc, argv);
  argc -= parsed_parameters;
  argv += parsed_parameters;

  /* do the job */
  if (0 == argc) 
    {
      actin  = stdin;
      actout = stdout;
      if (IsCGI())
	{ PrintCGIHeader(); }
      ConvertFile("");
    } 
  else 
    {
      for (i = 0; i < argc; i++) 
	{
	  actin = fopen (argv[i], "r");
	  if (!actin) 
	    {
	      fprintf (stderr, "%s: cannot read file '%s'\n", 
		       config.prog, argv[i]);
	      return_code = return_code < CANNOT_READ_INPUT ? 
		CANNOT_READ_INPUT : return_code;
	      continue;
	    }
	  outfilename = malloc (sizeof (char)*strlen (argv[i]) + 6);
	  strcat (strcpy (outfilename, argv[i]), ".html");
	  actout = fopen (outfilename, "w");
	  if (!actout) 
	    {
	      fprintf (stderr, "%s: cannot write file '%s'\n", 
		       config.prog, outfilename);
	      return_code = return_code < CANNOT_WRITE_OUTPUT ? 
		CANNOT_WRITE_OUTPUT : return_code;
	      continue;
	    }
	  ConvertFile(argv[i]);
	  
	  free (outfilename);
	  outfilename = (char *) NULL;
	  fclose (actin);
	  fclose (actout);
	}
    }
  return return_code;
}

int
Insert (FILE * outfile, char *filename)
{
  int c;
  FILE * infile;
 
  infile = fopen (filename, "r");
  if (NULL != infile)
    {
      c = fgetc (infile);
      while (EOF != c)
        { 
          fputc (c, outfile);
          c = fgetc (infile);
        }
      fclose (infile);
      return 1;
    }
  else
    { return 0; }
}

void 
PrintUsage() 
{
  fprintf (stderr,
           "usage: %s [options] [file_to_convert ...]\n", 
           config.prog);
  fprintf (stderr, "  -c       switch off CGI detection and HTTP header generation\n");
  fprintf (stderr, "  -s       suppress generation of HTML headers\n");
  fprintf (stderr, "  -n       number lines and label them\n");
  fprintf (stderr, "  -V       print version information\n");
  fprintf (stderr, "  -u       print this usage information\n");
  fprintf (stderr, "  -t title set HTML title\n");
  fprintf (stderr, "  -w width use width for output (default 80)\n");
  fprintf (stderr, "  -h file  insert file at the head of converted data\n");
  fprintf (stderr, "  -b file  insert file at the bottom of converted data\n");
  fprintf (stderr, "  --       interpret all following parameters as filenames\n");

  PrintConfig(stderr);
  exit(PRINTED_USAGE);
}

void 
PrintConfig(FILE * output)
{
  fprintf (output, "Current config is:\n");
  fprintf (output, "  (end of strings are marked with \"\\0\" and newline)\n");
  fprintf (output, "  COMPRESSION : %d\n", 
	   COMPRESSION);  
  fprintf (output, "  nocgi       : %s\n", 
	   config.nocgi ? "on" : "off");  
  fprintf (output, "  noheaders   : %s\n", 
	   config.noheaders ? "on" : "off");  
  fprintf (output, "  linelabeling: %s\n", 
	   config.linelabeling ? "on" : "off");  
  fprintf (output, "  width       : %d\n", 
	   config.width);  
  fprintf (output, "  title       : %s%s\n", 
	   config.title ? config.title : "(null)", 
	   config.title ? "\\0" : "");  
  fprintf (output, "  prog        : %s%s\n", 
	   config.prog ? config.prog : "(null)",
	   config.prog ? "\\0" : "");  
  fprintf (output, "  headfile    : %s%s\n", 
	   config.headfile ? config.headfile : "(null)",  
	   config.headfile ? "\\0" : "");  
  fprintf (output, "  bottomfile  : %s%s\n", 
	   config.bottomfile ? config.bottomfile : "(null)",
	   config.bottomfile ? "\\0" : "");  
}

int
ParseParameters(int argc,char *argv[])
{
  int ch = 0;

  while (-1 != (ch = getopt(argc, argv, "t:w:h:b:ncusV")))
    {
      switch(ch) 
	{
	case 't':
	  config.title = optarg;
	  break;
	case 'h':
	  config.headfile = optarg;
	  break;
	case 'b':
	  config.bottomfile = optarg;
	  break;
	case 'w':
	  if (1 != sscanf(optarg, "%d", &config.width))
	    { PrintUsage(); }
	  break;
	case 'n':
	  /* Switch on line numbering and labeling */
	  config.linelabeling = 1;
	  break;
        case 'c':
	  /* suppress generation of CGI header */
          config.nocgi = 1;
          break;
        case 's':
	  /* suppress generation of html headers */
          config.noheaders = 1;
          break;
	case 'u':
	  PrintUsage();
	  break;
	case 'V':
	  fprintf(stderr, "%s version %s%s\n",
		  PROJECT_NAME,
		  VERSION, 
                  COMPRESSION ? "" : " (no compression)");
	  exit(0);
	default:
	  PrintUsage();
	}
    }

  return optind;
}

int
IsCGI()
{
  return (getenv("PATH_TRANSLATED")
    && getenv("GATEWAY_INTERFACE") 
    && 0 == config.nocgi);
}

void
PrintCGIHeader()
{
  char       *pt = (char *) NULL;
  struct stat sb;
  char        gzipcmd[10];

  /* CGI */
  pt = getenv("PATH_TRANSLATED");
  if (!config.title)
    { config.title = pt; }
  actin = fopen (pt, "r");
  if (!actin) 
    {
      fprintf (actout, "Content-Type: text/html\n\n");
      fprintf (actout, "<HTML><HEAD><TITLE>error in CGI '%s'", config.prog);
      fprintf (actout, "</TITLE></HEAD>\n");
      fprintf (actout, "<BODY><H1>error in CGI program '%s': ", config.prog);
      fprintf (actout, "cannot read file '%s'</H1></BODY>\n", pt);
      fprintf (actout, "</HTML>\n");
      exit(1);
    }
  fprintf (actout, "Content-Type: text/html\n");
  if (fstat(fileno(actin), &sb) == 0) {
    /* report last modification date */
    char rfc1123date[32];
    
    if (strftime(rfc1123date, sizeof(rfc1123date),
		 "%a, %e %b %Y %T GMT", gmtime(&sb.st_mtime)))
      fprintf(actout, "Last-Modified: %.*s\n",
	      (int)sizeof(rfc1123date), rfc1123date);
  }
  if (COMPRESSION)
    {
      /* should we compress output with gzip? */
      do 
	{
	  if ((pt = getenv("REMOTE_ADDR")) 
	      && (!strncmp(pt, "127", 3)))
	    { break; } /* never compress to local client */
	  if (!(pt = getenv("HTTP_ACCEPT_ENCODING")))
	    { break; } /* no header Accept-Encoding: */
	  if (!strstr(pt, "gzip"))
	    { break; } /* browser doesn't understand gzip format */
	  if (strstr(pt, "x-gzip"))
	    { fprintf(actout, "Content-Encoding: x-gzip\n\n"); }
	  else /* some WWW browsers need this (namely ie) */
	    { fprintf(actout, "Content-Encoding: gzip\n\n"); }
	  fflush(actout);
	  sprintf(gzipcmd, "gzip -%d", COMPRESSION);
	  if (!(actout = (FILE *) popen(gzipcmd, "w")))
	    { exit(GZIP_NOT_FOUND); } /* cannot fork, gzip not found, ... */
	} 
      while (0);
    }
}

void 
ConvertFile(char * name)
{
  struct stat sb;

  if (0 == config.noheaders)
    {
      char * title = (char *) NULL;
      
      if (NULL == name)
	{ title = config.title ? config.title : "stdin"; }
      else
	{ title = config.title ? config.title : name; }

      fprintf (actout, "\n<HTML>\n<HEAD>\n");
      fprintf (actout, "<TITLE>%s</TITLE>\n", title);
      fprintf (actout, "<META NAME=\"generator\" CONTENT=\"%s %s\">\n",
	       PROJECT_NAME,
	       VERSION);
      if (fstat(fileno(actin), &sb) == 0) {
	  /* report last modification date */
	  char iso8601date[32];
    
	  if (strftime(iso8601date, sizeof(iso8601date),
		       "%Y-%m-%dT%H:%M:%S+00:00", gmtime(&sb.st_mtime)))
	      fprintf(actout, "<META NAME=\"date\" CONTENT=\"%.*s\">\n",
		      (int)sizeof(iso8601date), iso8601date);
      }
      fprintf (actout, "</HEAD>\n\n");
      fprintf (actout, "<BODY BGCOLOR=%s>\n", bgcolor);
    }

  Insert (actout, config.headfile);
  fprintf (actout, "<PRE WIDTH=\"%d\">", config.width);  
  StartNewYylex(actin, actout);
  
  fprintf (actout, "</PRE>\n");
  Insert (actout, config.bottomfile);
  if (0 == config.noheaders)
    { fprintf (actout, "</BODY>\n\n</HTML>\n"); }
}

void MyStringOutput(FILE * out, char * myString)
{
  static char        *previousColor = NULL;
  static weight_type previousWeight = NORMAL;
  char   ch = 0;
  
  if(NULL != myString)
    {
      while('\0' != myString[0])
	{
	  ch = *myString;
	  myString++;
	  if (0 == config.linelabeling)
	    { fputc(ch, out); }
	  else
	    {
	      /* Create labeled lines */	  
	      if ('\n' != ch)
		{ 
		  if (1 == config.needLabel)
		    { 
		      config.needLabel = 0;
		      fprintf(out, "<A NAME=\"line%d\">%3d: </A>", 
			      config.lineNumber, config.lineNumber);
		      ChangeFontTo(out, previousColor, previousWeight);
		    }
		  if ('\t' == ch)
		    { fprintf(out, "        "); }
		  else
		    { fputc(ch, out); }
		}
	      else
		{ 
		  if (0 == config.needLabel)
		    {
		      /* Do this only once for succeeding */
		      /* newlines */
		      previousColor  = config.currentColor;
		      previousWeight = config.currentWeight;
		    }
		  ChangeFontTo(out, NULL, NORMAL);
		  fputc(ch, out);
		  config.needLabel = 1;
		  config.lineNumber++;
		}
	    }
	}
    }
}

/* Change the font and the weight
 * If color is a NULL pointer the FONT will be closed if
 * one is opened. 
 * Generate the tags so that FONT is contained in an
 * STRONG if weight is BOLD.
 * Generate the tags in a flat structure (no nested fonts)
 */
void
ChangeFontTo(FILE *out, char *color, weight_type weight)
{
  switch(weight)
    {
    case NO_CHANGE:
      break;
    case BOLD:
      if (BOLD != config.currentWeight)
	{ 
	  if (NULL != config.currentColor)
	    {
	      MyStringOutput(out, "</FONT>");
	      config.currentColor = NULL;
	    }
	  MyStringOutput(out, "<STRONG>"); 
	  config.currentWeight = BOLD;
	}
      break;
    case NORMAL:
      if (NORMAL != config.currentWeight)
	{ 
	  if (NULL != config.currentColor)
	    {
	      MyStringOutput(out, "</FONT>");
	      config.currentColor = NULL;
	    }	  
	  MyStringOutput(out, "</STRONG>"); 
	  config.currentWeight = NORMAL;
	}
      break;
    default:
      fprintf (stderr, "internal error: bad weight\n");
    }

  if (color != config.currentColor)
    {
      if (NULL != config.currentColor)
	{
	  MyStringOutput(out, "</FONT>");
	  config.currentColor = NULL;
	}
      if (NULL != color)
	{
	  MyStringOutput(out, "<FONT COLOR=");
	  MyStringOutput(out, color);
	  MyStringOutput(out, ">");
	  config.currentColor = color;
	}
    }
  /* Check for consistency */
  if (color != config.currentColor)
    { fprintf(stderr, "internal error in ChangeFontTo\n"); }
}

/* Add a label a function */
/* Therefore search for the first opening parenthesis */
/* and use the word before as label name. */
void
AddLabelForFunction(FILE *out, char *text)
{
  char *start_ptr = NULL;
  char *end_ptr = NULL;

  end_ptr = strchr(text, '(');
  if (NULL != end_ptr)
    {
      end_ptr--;
      while ((' ' == *end_ptr)
	     || ('\t' == *end_ptr))
	{ end_ptr--; }
      start_ptr = end_ptr;
      while (((('A' <= *start_ptr) && ('Z' >= *start_ptr))
	      || (('a' <= *start_ptr) && ('z' >= *start_ptr))
	      || (('0' <= *start_ptr) && ('9' >= *start_ptr))
	      || ('_' == *start_ptr))
	     && (start_ptr != text))
	{ start_ptr--; }
      if (start_ptr != end_ptr)
	{ 
	  fprintf(out, "<A NAME=\"");
	  start_ptr++;
	  while (start_ptr != end_ptr)
	    { 
	      fputc(*start_ptr, out);
	      start_ptr++;
	    }
	  fputc(*start_ptr, out);
	  fprintf(out, "\"></A>");
	}
    }
}

/* Add a label for a class */
/* Therefore search for the word class and */
/* use the word after that as label name. */
void
AddLabelForClass(FILE *out, char *text)
{
  char *start_ptr = NULL;
  char *end_ptr = NULL;

  start_ptr = strstr(text, "class");
  if (NULL != start_ptr)
    {
      /* move the pointer behind the wor 'class' */
      start_ptr += 5;
      while ((' ' == *start_ptr)
	     || ('\t' == *start_ptr))
	{ start_ptr++; }
      end_ptr = start_ptr;
      while (((('A' <= *end_ptr) && ('Z' >= *end_ptr))
	      || (('a' <= *end_ptr) && ('z' >= *end_ptr))
	      || (('0' <= *end_ptr) && ('9' >= *end_ptr))
	      || ('_' == *end_ptr)))
	{ end_ptr++; }
      if (start_ptr != end_ptr)
	{ 
	  fprintf(out, "<A NAME=\"");
	  end_ptr--;
	  while (start_ptr != end_ptr)
	    { 
	      fputc(*start_ptr, out);
	      start_ptr++;
	    }
	  fputc(*start_ptr, out);
	  fprintf(out, "\"></A>");
	}
    }
}

