/* 
   Copyright  1998, 1999 Enbridge Pipelines Inc. 
   Copyright  1999 dave Carrigan
   All rights reserved.

   This module is free software; you can redistribute it and/or modify
   it under the same terms as Apache itself. This module is
   distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. The copyright holder of this
   module can not be held liable for any general, special, incidental
   or consequential damages arising out of the use of the module.

   $Id: auth_ldap.h,v 1.2 1999/08/11 03:17:03 dave Exp $

*/

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "multithread.h"

#include <lber.h>
#include <ldap.h>
#ifdef WITH_SSL
#include <ldap_ssl.h>
#endif

#include <stdarg.h>
#include <sys/types.h>
#include <time.h>

/* Backwards compatibility for Apache 1.2 */
#if APACHE_RELEASE < 1030000
#define auth_ldap_get_pw get_pw
#define ap_pcalloc pcalloc
#define ap_pstrdup pstrdup
#define ap_get_module_config get_module_config
#define ap_get_basic_auth_pw get_basic_auth_pw
#define ap_set_string_slot set_string_slot
#define ap_set_flag_slot set_flag_slot
#define ap_getword getword
#define ap_note_basic_auth_failure note_basic_auth_failure
#define ap_requires requires
#define ap_getword_conf getword_conf

#define APLOG_MARK __FILE__,__LINE__
#define APLOG_DEBUG 7
#define APLOG_WARNING 4
#define APLOG_ERR 3
#define APLOG_NOERRNO 0
#endif

/* Struct for holding cached compare operations */
typedef struct compare_node {
  char *attrib;
  char *value;
  time_t comparetime;
  struct compare_node *next;
} compare_node;

/* 
 * Struct for holding cached bind operations. The link to the compare cache
 * is here as well. This means that the only compares that get cached have 
 * first bound (i.e., it's user compares, not group compares). This implies
 * that group compares are never cached. Compares don't seem to be that
 * expensive, so caching of group compares is not yet implemented
 */
typedef struct cache_node {
  char *dn;
  char *pw;
  time_t bindtime;
  compare_node *compares;
  struct cache_node *next;
} cache_node;

/* This is the ldap operation cache */
typedef struct {
  cache_node **nodelist;
  int size;
  int maxentries;
} ldapcache;

/* 
 * We want the ldap connections to be long-lived; we'll need one
 * connection for each host/port combination. We keep these in a
 * global linked list, so they're around for as long as the apache
 * process is around.
 *
 * Note that the URLs ldap://ldap and ldap://ldap.airius.com should
 * represent the same connection, but will result in different
 * connections entries in the cache. Caveat emptor.
 */
#ifdef WITH_OPENLDAP
/*
 * Provide a definition for LDAPMemCache. Our definition just holds the ttl and 
 * size. This struct is filled in when we emulate ldap_memcache_init, and is used
 * when we emulate ldap_memcache_set.
 */
typedef struct {
  unsigned long ttl;
  unsigned long size;
} LDAPMemCache;
int ldap_memcache_init(unsigned long ttl, unsigned long size, char **baseDNs, 
		       void *thread_fns, LDAPMemCache **cachep);
int ldap_memcache_set(LDAP *, LDAPMemCache *);
int ldap_search_ext_s(LDAP *ldap, char *base, int scope, char *filter,
		      char **attrs, int attrsonly, void *servertrls, void *clientctrls,
		      void *timeout, int sizelimit, LDAPMessage **res);
void ldap_memfree(void *p);

/* The const_cast is used to get around the fact that some of OpenLDAP's prototypes
   have non-const parameters, while the same ones in Netscape's are const. If compiling
   with OpenLDAP, the const_cast casts away the constness, but won't under Netscape */
#define const_cast(x) ((char *)(x))
#else
#define const_cast(x) (x)
#endif /* WITH_OPENLDAP */

struct LDAPconnection {
  LDAP *ldap;
  LDAPMemCache *cache;
  ldapcache *ldapopcache;
  mutex *mtx;
  char *bounddn;
  char *host;
  int port;
  struct LDAPconnection *next;
};

/* auth_ldap global configuration */ 
typedef struct {
  long cache_ttl;
  long cache_size;
  long opcache_ttl;
  int opcache_size;
  int cache_compares;
  mutex *mtx;
  struct LDAPconnection *ldapconnections;
#ifdef WITH_SSL
  int have_certdb;
#endif
} auth_ldap_server_conf;

/* The actual LDAP auth struct. */
typedef struct {
  int auth_authoritative;	/* Is this auth method the one and only? */

  /* These parameters are all derived from the AuthLDAPURL directive */
  char *url;			/* String representation of the URL */
  char *host;			/* Name of the LDAP server (or space separated list) */
  int port;			/* Port of the LDAP server */
  char *basedn;			/* Base DN to do all searches from */
  char *attribute;		/* Attribute to search for */
  int scope;			/* Scope of the search */
  char *filter;			/* Filter to further limit the search  */

#ifdef AUTH_LDAP_FRONTPAGE_HACK
  int frontpage_hack;		/* Hack for frontpage support */
  char *frontpage_hack_pwfile;	/* Filename for frontpage auth checking */
#endif

  char *binddn;			/* DN to bind to server (can be NULL) */
  char *bindpw;			/* Password to bind to server (can be NULL) */
  int needbind;

  char *dn;			/* The saved dn from a successful search */
  int user_is_dn;		/* If true, connection->user is DN instead of userid */

  int have_ldap_url;		/* Set if we have found an LDAP url */
  struct LDAPconnection *ldc;	/* Pointer to an LDAP connection in the linked list */
#ifdef WITH_SSL
  int secure;			/* True if use SSL connection */
#endif
} auth_ldap_config_rec;

extern void *create_auth_ldap_dir_config(pool *p, char *d);
extern const char *parse_auth_ldap_url(cmd_parms *cmd, auth_ldap_config_rec *sec, 
 			       char *url);
extern const char *auth_ldap_set_cache_ttl(cmd_parms *cmd, void *dummy, char *ttl);
extern const char *auth_ldap_set_cache_size(cmd_parms *cmd, void *dummy, char *size);
extern const char *auth_ldap_set_opcache_ttl(cmd_parms *cmd, void *dummy, char *size);
extern const char *auth_ldap_set_opcache_size(cmd_parms *cmd, void *dummy, char *size);
extern const char *auth_ldap_set_compare_flag(cmd_parms *cmd, void *dummy, char *size);
#ifdef WITH_SSL
extern const char *auth_ldap_set_certdbpath(cmd_parms *cmd, void *dummy, char *path);
#endif
extern const char *auth_ldap_add_redundant(cmd_parms *cmd, 
 				   auth_ldap_config_rec *sec, char *f);
extern const char *auth_ldap_snarf_pwfile(cmd_parms *cmd, 
 				  auth_ldap_config_rec *sec, char *f);
extern void *create_auth_ldap_config(pool *p, server_rec *s);
extern const char *auth_ldap_version;

ldapcache *auth_ldap_new_cache(int maxentries);
int auth_ldap_authbind(const char *, request_rec *);
int auth_ldap_compare(const char *, const char *, const char *, request_rec *);
int auth_ldap_connect_to_server(request_rec *r);
void auth_ldap_free_connection(request_rec *r, int log);
void auth_ldap_log_reason(request_rec *r, const char *fmt, ...);
