/* Receives data from the EMWIN DM-296 modem line and multicasts it.

Written by Antonio Querubin, Capt
   	   199'th Weather Flight
           Hawaii Air National Guard
	   tony@hiang.af.mil
	   AH6BW

Revision history:
14 Sep 1999, version 1.0:  First production use.
21 Sep 1999, version 1.1:  Added some suggestions from Julian Cowley.
29 Sep 1999, version 1.2:  Debug toggler now uses SIGUSR1 instead of SIGINT.
			   Removed loopback disable.
 1 Oct 1999, version 1.3:  Forks off a child daemon (from Maitland 
                           Bottoms).
12 Oct 1999, version 1.4:  Added syslogging of sendto() failures instead
                           of just aborting.

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <termios.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <ctype.h>
#include <syslog.h>

#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define VERSION "1.4"

#undef DEBUG 1

/* emwin.mcast.net = 225.0.1.143 (IANA assigned) */
#define EMWINMCADDR "224.0.1.143"
#define EMWINPORT 1024
#define EMWINTTL 255
/* Note:  for testing purposes, use a different multicast group
address and/or TTL than the above. */

struct QBT {		/* http://iwin.nws.noaa.gov/emwin/winpro.htm */
  unsigned char preamble[6];
  unsigned char header[80];
  unsigned char data[1024];
  unsigned char trailer[6];
} QBT;

int debuglevel = 0;

void toggledebug(int sig) {	/* SIGUSR1 signal handler. */
  char logbuf[80];
  debuglevel ^= 1;
  sprintf(logbuf,"Debug level = %d",debuglevel);
  syslog(0,logbuf);
}

/* Opens the serial port at 9600 BAUD, no parity, 8 bits, 1 stop bit. */
FILE *openemwinstream(char *devicename) {
  int fd;
  struct termios ttyios;
  int modemlines;
  FILE *streamp;
  int off = 0;

  /* Open the serial line in non-blocking mode (ignore DCD). */
  if ((fd = open(devicename, O_RDONLY | O_NOCTTY | O_NONBLOCK, 0))
      == -1) {
    perror("openemwinstream, open serial line");
    exit(EXIT_FAILURE);
  }

  /* Initialize the line characteristics. */
  if (tcgetattr(fd, &ttyios)) {
    perror("openemwinstream, get attributes");
    exit(EXIT_FAILURE);
  }
  cfmakeraw(&ttyios);			/* Set raw mode. */
  if (cfsetspeed(&ttyios, B9600)) {	/* Use 9600 BAUD. */
    perror("openwemwinstream, set speed");
    exit(EXIT_FAILURE);
  }
  /* Set other attributes. */
  ttyios.c_cflag &= ~CSTOPB;		/* Use one stop bit. */
  ttyios.c_cflag |= CREAD | CLOCAL;	/* Enable reading. */
  ttyios.c_cc[VMIN] = 1;		/* Block for at least one character. */
  ttyios.c_cc[VTIME] = 0;
  if (tcsetattr(fd, TCSAFLUSH, &ttyios)) {	/* Set the new attributes.
*/
    perror("openemwinstream, set attributes");
    exit(EXIT_FAILURE);
  }

  /* Set the line back to blocking mode after setting CLOCAL. */
  if (ioctl(fd, FIONBIO, &off) < 0) {
    perror("openemwinstream, setting blocking mode");
    exit(EXIT_FAILURE);
  }

  /* The DTR line is automatically raised but we need to lower the RTS
     line to read anything from the EMWIN satellite modem. */
  sleep(1);
  modemlines = TIOCM_RTS; 
  if (ioctl(fd, TIOCMBIC, &modemlines)) {
    perror("openemwinstream, clear RTS line");
    exit(EXIT_FAILURE);
  }

  if ((streamp=fdopen(fd,"rb")) == NULL) {
    perror("openemwinstream, fdopen");
    exit(EXIT_FAILURE);
  }

  return streamp;
} /* openemwinstream */

/* Opens a multicast output socket. */
int openemwinsocketout(void) {
  struct sockaddr_in stLocal;
  int s, iTmp;

  /* get a datagram socket */
  s = socket(AF_INET, SOCK_DGRAM, 0);

  /* avoid EADDRINUSE error on bind() */
  iTmp = 1;
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&iTmp, sizeof(iTmp));

  /* name the socket */
  stLocal.sin_family = AF_INET;
  stLocal.sin_addr.s_addr = htonl(INADDR_ANY);
  stLocal.sin_port = htons(EMWINPORT);
  bind(s, (struct sockaddr*) &stLocal, sizeof(stLocal));

  /* set TTL to traverse up to multiple routers */
  iTmp = EMWINTTL;
  setsockopt(s, IPPROTO_IP, IP_MULTICAST_TTL, (char *)&iTmp, sizeof(iTmp));

  return s;
} /* openemwinsocketout */

int getqbtpacket(FILE *emwinstream, struct QBT *qbtpacket) {
  char qbtchar, qbtbuf[sizeof(QBT)];
  int nullcount = 0;
  char logbuf[80];

  memset(qbtpacket,'\0',sizeof(*qbtpacket));

  while(1) {

    switch(qbtchar=fgetc(emwinstream)) {
    case '\0' : /* Look for the null preamble. */
	nullcount++;
	break;
    case '/':  /* Possible start of a line header. */
	if (nullcount >= 6) {
	  if (debuglevel) {
            sprintf(logbuf,"Null count = %d",nullcount);
            syslog(0,logbuf);
	  }
	  ungetc(qbtchar, emwinstream);
	  if (fgets(qbtbuf,sizeof(qbtbuf),emwinstream) != NULL) {
	    if (debuglevel) {
              sprintf(logbuf,"%d chars in header",strlen(qbtbuf));
              syslog(0,logbuf);
	    }
	    if (strlen(qbtbuf) == sizeof(qbtpacket->header)) {
	      strncpy(qbtpacket->header,qbtbuf,sizeof(qbtpacket->header));
	      if (fread(qbtpacket->data, sizeof(qbtpacket->data),
		  1, emwinstream) == 1) return(0);
	    }
	  }
	}
        if (debuglevel) {
          syslog(0,"Resynchronizing");
        }
    default:
	nullcount = 0;	/* Restart the null train search. */
	break;
    }
  }
} /* qbtpacket */

int main(int argc, char *argv[]) {
  FILE *emwinstream;
  int emwinsocket;
  struct sockaddr_in socketst;
  struct QBT qbtpacket;
  int addr_size;
  char tempbuf[sizeof(QBT)];
  int i;
  pid_t pid;

  if (argc != 2) {
    printf("Usage:  %s devicename\n",argv[0]);
    exit(EXIT_FAILURE);
  }

  /* Fork a child process to run as a background daemon. */
  if((pid=fork()) < 0) {
    perror("fork");
    exit(EXIT_FAILURE);
  }
  else if (pid != 0) exit(EXIT_SUCCESS); /* Parent goes bye-bye */

  /* Child continues */

  /* Become the session leader. */
  if (setsid() < 0) {
    perror("setsid");
    exit(EXIT_FAILURE);
  }
  openlog("emwinmcs",LOG_PID,LOG_USER);
  syslog(LOG_INFO,"EMWIN Multicast Server daemon starting, version "
                  VERSION);

  signal(SIGUSR1, toggledebug);	/* Set trap for SIGUSR1. */

  emwinstream = openemwinstream(argv[1]);

  emwinsocket = openemwinsocketout();
  /* assign our destination address */
  socketst.sin_family = AF_INET;
  socketst.sin_addr.s_addr = inet_addr(EMWINMCADDR);
  socketst.sin_port = htons(EMWINPORT);

  /* Start input processing. */
  while (1) {

    getqbtpacket(emwinstream, &qbtpacket);
    if (debuglevel) {
	strncpy(tempbuf,qbtpacket.header,sizeof(qbtpacket.header));
	/* Truncate the trailing white-space at the end of
	the line so that it doesn't appear in the syslog. */
	for (i=sizeof(qbtpacket.header)-1; isspace(tempbuf[i]); i--)
          tempbuf[i] = '\0';
	syslog(0,tempbuf);
    }

    /* send to the multicast address */
    addr_size = sizeof(struct sockaddr_in);
    if (sendto(emwinsocket, &qbtpacket, sizeof(qbtpacket), 
        0, (struct sockaddr*)&socketst, addr_size) < 0) {
      perror("sendto() failed\n");
      sprintf(tempbuf,"sendto() failed, %s",strerror(errno)); 
      syslog(0,tempbuf);
    }
  }

  /*  Shouldn't get here but if we do */
  exit(EXIT_FAILURE);
 
}
