/* $Id: errno.c,v 1.5 1998/07/05 16:57:03 sverrehu Exp $ */
/**************************************************************************
 *
 *  FILE            errno.c
 *  MODULE OF       shhmsg - library for displaying messages.
 *
 *  DESCRIPTION     The functions in this file, are for displaying errno-
 *                  dependant error messages.
 *
 *  WRITTEN BY      Sverre H. Huseby <sverrehu@online.no>
 *
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>

#include "internal.h"
#include "shhmsg.h"

extern char *sys_errlist[];

/**************************************************************************
 *                                                                        *
 *                    P U B L I C    F U N C T I O N S                    *
 *                                                                        *
 **************************************************************************/

/*-------------------------------------------------------------------------
 *
 *  NAME          msgPerror
 *
 *  FUNCTION      Show an errno-dependant error message.
 *
 *  SYNOPSIS      #include "shhmsg.h"
 *                void msgPerror(const char *format, ...);
 *
 *  INPUT         format, ...
 *                        Arguments used as with printf(). If NULL,
 *                        the name of the program is printed.
 *
 *  DESCRIPTION   Prints the given text followed by ": " and the
 *                correct errormessage on the _msgErrorStream.
 *		  May add the program name before anything is printed.
 *
 *                If errno indicates "no error", nothing is printed.
 */
void
msgPerror(const char *format, ...)
{
    va_list ap;
    int en;

    if (!(en = errno))
        return;
    fflush(stdout);
    if (format && *format) {
	if (_msgShowNameAlways)
	    fprintf(_msgErrorStream, "%s: ", msgGetName());
	va_start(ap, format);
	vfprintf(_msgErrorStream, format, ap);
	va_end(ap);
    } else
	fprintf(_msgErrorStream, msgGetName());
    fprintf(_msgErrorStream, ": %s\n", sys_errlist[en]);
}



/*-------------------------------------------------------------------------
 *
 *  NAME          msgFatalPerror
 *
 *  FUNCTION      Show an errno-dependant error message and abort the program.
 *
 *  SYNOPSIS      #include "shhmsg.h"
 *                void msgFatalPerror(const char *format, ...);
 *
 *  INPUT         format, ...
 *                        Arguments used as with printf(). If NULL,
 *                        the name of the program is printed.
 *
 *  RETURNS       Nothing, never returns.
 *
 *  DESCRIPTION   Prints the given text followed by ": " and the
 *                correct errormessage on the _msgErrorStream, and then
 *                aborts the program.
 *		  May add the program name before anything is printed.
 *
 *                If errno indicates "no error", nothing is printed,
 *                and the program keeps running.
 */
void
msgFatalPerror(const char *format, ...)
{
    va_list ap;
    int en;

    if (!(en = errno))
        return;
    fflush(stdout);
    if (format && *format) {
	if (_msgShowNameAlways)
	    fprintf(_msgErrorStream, "%s: ", msgGetName());
	va_start(ap, format);
	vfprintf(_msgErrorStream, format, ap);
	va_end(ap);
    } else
	fprintf(_msgErrorStream, msgGetName());
    fprintf(_msgErrorStream, ": %s\n", sys_errlist[en]);
    exit(99);
}
