#include "Python.h"
#include <libIDL/IDL.h>

static PyObject *ErrorObject;

typedef struct {
	PyObject_HEAD
	PyObject	*x_attr;	/* Attributes dictionary */
	IDL_tree	tree;
} IDLtreeObject;

staticforward PyTypeObject IDLtree_Type;

#define IDLtreeObject_Check(v)	((v)->ob_type == &IDLtree_Type)

static IDLtreeObject *
IDLtree2Object(
	IDL_tree tree
) {
	IDLtreeObject *self;
	if (tree == NULL) {
		Py_INCREF(Py_None);
		return ((IDLtreeObject *) Py_None);
	}
	self = PyObject_NEW(IDLtreeObject, &IDLtree_Type);
	if (self == NULL)
		return NULL;
	self->x_attr = NULL;
	self->tree = tree;
	return self;
}

/* IDLtree methods */

static void
IDLtree_dealloc(self)
	IDLtreeObject *self;
{
	Py_XDECREF(self->x_attr);
	PyMem_DEL(self);
}

static PyObject *
IDLtree_free(self, args)
	IDLtreeObject *self;
	PyObject *args;
{
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	IDL_tree_free(self->tree);
	self->tree = NULL;
	Py_INCREF(Py_None);
	return (Py_None);
}

static PyObject *
IDLtree_get_type(self, args)
	IDLtreeObject *self;
	PyObject *args;
{
	char *type;
	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (self->tree == NULL) {
fprintf(stderr, "get_type on NULL IDL_tree\n");
		/* XXX set exception */
		return NULL;
	}
	switch(self->tree->_type) {
	case IDLN_NONE:
		type = "none";
		break;
	case IDLN_ANY:
		type = "any";
		break;
	case IDLN_LIST:
		type = "list";
		break;
	case IDLN_GENTREE:
		type = "gentree";
		break;
	case IDLN_INTEGER:
		type = "integer";
		break;
	case IDLN_STRING:
		type = "string";
		break;
	case IDLN_WIDE_STRING:
		type = "wide_string";
		break;
	case IDLN_CHAR:
		type = "char";
		break;
	case IDLN_WIDE_CHAR:
		type = "wide_char";
		break;
	case IDLN_FIXED:
		type = "fixed";
		break;
	case IDLN_FLOAT:
		type = "float";
		break;
	case IDLN_BOOLEAN:
		type = "boolean";
		break;
	case IDLN_IDENT:
		type = "ident";
		break;
	case IDLN_TYPE_DCL:
		type = "type_dcl";
		break;
	case IDLN_CONST_DCL:
		type = "const_dcl";
		break;
	case IDLN_EXCEPT_DCL:
		type = "except_dcl";
		break;
	case IDLN_ATTR_DCL:
		type = "attr_dcl";
		break;
	case IDLN_OP_DCL:
		type = "op_dcl";
		break;
	case IDLN_PARAM_DCL:
		type = "param_dcl";
		break;
	case IDLN_FORWARD_DCL:
		type = "forward_dcl";
		break;
	case IDLN_TYPE_INTEGER:
		type = "type_integer";
		break;
	case IDLN_TYPE_FLOAT:
		type = "type_float";
		break;
	case IDLN_TYPE_FIXED:
		type = "type_fixed";
		break;
	case IDLN_TYPE_CHAR:
		type = "type_char";
		break;
	case IDLN_TYPE_WIDE_CHAR:
		type = "type_wide_char";
		break;
	case IDLN_TYPE_STRING:
		type = "type_string";
		break;
	case IDLN_TYPE_WIDE_STRING:
		type = "type_wide_string";
		break;
	case IDLN_TYPE_BOOLEAN:
		type = "type_boolean";
		break;
	case IDLN_TYPE_OCTET:
		type = "type_octet";
		break;
	case IDLN_TYPE_ANY:
		type = "type_any";
		break;
	case IDLN_TYPE_OBJECT:
		type = "type_object";
		break;
	case IDLN_TYPE_TYPECODE:
		type = "type_typecode";
		break;
	case IDLN_TYPE_ENUM:
		type = "type_enum";
		break;
	case IDLN_TYPE_SEQUENCE:
		type = "type_sequence";
		break;
	case IDLN_TYPE_ARRAY:
		type = "type_array";
		break;
	case IDLN_TYPE_STRUCT:
		type = "type_struct";
		break;
	case IDLN_TYPE_UNION:
		type = "type_union";
		break;
	case IDLN_MEMBER:
		type = "member";
		break;
	case IDLN_NATIVE:
		type = "native";
		break;
	case IDLN_CASE_STMT:
		type = "case_stmt";
		break;
	case IDLN_INTERFACE:
		type = "interface";
		break;
	case IDLN_MODULE:
		type = "module";
		break;
	case IDLN_BINOP:
		type = "binop";
		break;
	case IDLN_UNARYOP:
		type = "unaryop";
		break;
	case IDLN_CODEFRAG:
		type = "codefrag";
		break;
	default:
		type = "<UNKNOWN IDL TYPE>";
		break;
	}
	return (Py_BuildValue("s", type));
}

static PyObject *
IDLtree_get_value(self, args)
	IDLtreeObject *self;
	PyObject *args;
{
	int x;
	char *tmpstr;
	PyObject *tmpob;
	IDL_tree tmptree;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	if (self->tree == NULL) {
fprintf(stderr, "get_value on NULL IDL_tree\n");
		/* XXX set exception */
		return NULL;
	}

	switch(self->tree->_type) {
	case IDLN_NONE:
		Py_INCREF(Py_None);
		return Py_None;
	case IDLN_ANY:
		Py_INCREF(Py_None);
		return Py_None;
	case IDLN_LIST:
		tmpob = PyList_New(0);
		if (tmpob == NULL)
			return NULL;
		tmptree = self->tree;
		while (tmptree) {
if (tmptree->u.idl_list.data == NULL) 
fprintf(stderr, "Creating NULL node in list\n");
			if (PyList_Append(tmpob,
				(PyObject *) IDLtree2Object(tmptree->u.idl_list.data)) < 0) {
				return NULL;
			}
			tmptree = tmptree->u.idl_list.next;
		}
		return tmpob;
	case IDLN_GENTREE:
		return (PyObject *) IDLtree2Object(self->tree->u.idl_gentree.data);
	case IDLN_INTEGER:
		/* XXX bogus */
		return Py_BuildValue("l", (long) self->tree->u.idl_integer.value);
	case IDLN_STRING:
		return Py_BuildValue("s", self->tree->u.idl_string.value);
	case IDLN_WIDE_STRING:
		/* XXX bogus */
		return Py_BuildValue("s", (char *) self->tree->u.idl_wide_string.value);
	case IDLN_CHAR:
		return Py_BuildValue("s", self->tree->u.idl_char.value);
	case IDLN_WIDE_CHAR:
		return Py_BuildValue("s", (char *) self->tree->u.idl_wide_char.value);
	case IDLN_FIXED:
		return Py_BuildValue("s", self->tree->u.idl_fixed.value);
	case IDLN_FLOAT:
		return Py_BuildValue("d", self->tree->u.idl_float.value);
	case IDLN_BOOLEAN:
		return Py_BuildValue("i", (int) self->tree->u.idl_boolean.value);
	case IDLN_IDENT:
		/* XXX add comments later */
		return Py_BuildValue("(ss)", self->tree->u.idl_ident.str,
			   self->tree->u.idl_ident.repo_id);
	case IDLN_TYPE_DCL:
		return Py_BuildValue("(NN)",
			IDLtree2Object(self->tree->u.idl_type_dcl.type_spec),
			IDLtree2Object(self->tree->u.idl_type_dcl.dcls));
	case IDLN_CONST_DCL:
		return Py_BuildValue("(NNN)",
			IDLtree2Object(self->tree->u.idl_const_dcl.const_type),
			IDLtree2Object(self->tree->u.idl_const_dcl.ident),
			IDLtree2Object(self->tree->u.idl_const_dcl.const_exp));
	case IDLN_EXCEPT_DCL:
		return Py_BuildValue("(NN)",
			IDLtree2Object(self->tree->u.idl_except_dcl.ident),
			IDLtree2Object(self->tree->u.idl_except_dcl.members));
	case IDLN_ATTR_DCL:
		return Py_BuildValue("(iNN)",
			(int) self->tree->u.idl_attr_dcl.f_readonly,
			IDLtree2Object(self->tree->u.idl_attr_dcl.param_type_spec),
			IDLtree2Object(self->tree->u.idl_attr_dcl.simple_declarations));
	case IDLN_OP_DCL:
		return Py_BuildValue("(iNNNNN)",
			(int) self->tree->u.idl_op_dcl.f_oneway,
			IDLtree2Object(self->tree->u.idl_op_dcl.op_type_spec),
			IDLtree2Object(self->tree->u.idl_op_dcl.ident),
			IDLtree2Object(self->tree->u.idl_op_dcl.parameter_dcls),
			IDLtree2Object(self->tree->u.idl_op_dcl.raises_expr),
			IDLtree2Object(self->tree->u.idl_op_dcl.context_expr));
	case IDLN_PARAM_DCL:
		switch (self->tree->u.idl_param_dcl.attr) {
		case IDL_PARAM_IN:
			tmpstr = "in";
			break;
		case IDL_PARAM_OUT:
			tmpstr = "out";
			break;
		case IDL_PARAM_INOUT:
			tmpstr = "inout";
			break;
		default:
			tmpstr = "<IDL ERROR>";
		}
		return Py_BuildValue("(sNN)", tmpstr,
			IDLtree2Object(self->tree->u.idl_param_dcl.param_type_spec),
			IDLtree2Object(self->tree->u.idl_param_dcl.simple_declarator));
	case IDLN_FORWARD_DCL:
		return (PyObject *) IDLtree2Object(self->tree->u.idl_forward_dcl.ident);
	case IDLN_TYPE_INTEGER:
		switch (self->tree->u.idl_type_integer.f_type) {
		case IDL_INTEGER_TYPE_SHORT:
			tmpstr = "short";
			break;
		case IDL_INTEGER_TYPE_LONG:
			tmpstr = "long";
			break;
		case IDL_INTEGER_TYPE_LONGLONG:
			tmpstr = "longlong";
			break;
		default:
			tmpstr = "<IDL ERROR>";
		}
		return Py_BuildValue("(is)",
			(int) self->tree->u.idl_type_integer.f_signed, tmpstr);
	case IDLN_TYPE_FLOAT:
		switch (self->tree->u.idl_type_float.f_type) {
		case IDL_FLOAT_TYPE_FLOAT:
			tmpstr = "float";
			break;
		case IDL_FLOAT_TYPE_DOUBLE:
			tmpstr = "double";
			break;
		case IDL_FLOAT_TYPE_LONGDOUBLE:
			tmpstr = "longdouble";
			break;
		default:
			tmpstr = "<IDL ERROR>";
		}
		return Py_BuildValue("s", tmpstr);
	case IDLN_TYPE_FIXED:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_type_fixed.positive_int_const),
			IDLtree2Object(self->tree->u.idl_type_fixed.integer_lit));
	case IDLN_TYPE_CHAR:
	case IDLN_TYPE_WIDE_CHAR:
		return Py_BuildValue("s", "Don't ask me, man");
	case IDLN_TYPE_STRING:
		return Py_BuildValue("N",
			IDLtree2Object(self->tree->u.idl_type_string.positive_int_const));
	case IDLN_TYPE_WIDE_STRING:
		return Py_BuildValue("N",
			IDLtree2Object(self->tree->u.idl_type_wide_string.positive_int_const));
	case IDLN_TYPE_BOOLEAN:
	case IDLN_TYPE_OCTET:
	case IDLN_TYPE_ANY:
	case IDLN_TYPE_OBJECT:
	case IDLN_TYPE_TYPECODE:
		return Py_BuildValue("s", "Don't ask me, man");
	case IDLN_TYPE_ENUM:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_type_enum.ident),
			IDLtree2Object(self->tree->u.idl_type_enum.enumerator_list));
	case IDLN_TYPE_SEQUENCE:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_type_sequence.simple_type_spec),
			IDLtree2Object(self->tree->u.idl_type_sequence.positive_int_const));
	case IDLN_TYPE_ARRAY:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_type_array.ident),
			IDLtree2Object(self->tree->u.idl_type_array.size_list));
	case IDLN_TYPE_STRUCT:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_type_struct.ident),
			IDLtree2Object(self->tree->u.idl_type_struct.member_list));
	case IDLN_TYPE_UNION:
		return Py_BuildValue("(NNN)", 
			IDLtree2Object(self->tree->u.idl_type_union.ident),
			IDLtree2Object(self->tree->u.idl_type_union.switch_type_spec),
			IDLtree2Object(self->tree->u.idl_type_union.switch_body));
	case IDLN_MEMBER:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_member.type_spec),
			IDLtree2Object(self->tree->u.idl_member.dcls));
	case IDLN_NATIVE:
		return Py_BuildValue("N",
			IDLtree2Object(self->tree->u.idl_native.ident));
	case IDLN_CASE_STMT:
		return Py_BuildValue("(NN)", 
			IDLtree2Object(self->tree->u.idl_case_stmt.labels),
			IDLtree2Object(self->tree->u.idl_case_stmt.element_spec));
	case IDLN_INTERFACE:
		return Py_BuildValue("(NNN)",
			IDLtree2Object(self->tree->u.idl_interface.ident),
			IDLtree2Object(self->tree->u.idl_interface.inheritance_spec),
			IDLtree2Object(self->tree->u.idl_interface.body));
	case IDLN_MODULE:
		return Py_BuildValue("(NN)",
			IDLtree2Object(self->tree->u.idl_module.ident),
			IDLtree2Object(self->tree->u.idl_module.definition_list));
	case IDLN_BINOP:
		switch (self->tree->u.idl_binop.op) {
		case IDL_BINOP_OR:
			tmpstr = "or";
			break;
		case IDL_BINOP_XOR:
			tmpstr = "xor";
			break;
		case IDL_BINOP_AND:
			tmpstr = "and";
			break;
		case IDL_BINOP_SHR:
			tmpstr = "shiftright";
			break;
		case IDL_BINOP_SHL:
			tmpstr = "shiftleft";
			break;
		case IDL_BINOP_ADD:
			tmpstr = "add";
			break;
		case IDL_BINOP_SUB:
			tmpstr = "subtract";
			break;
		case IDL_BINOP_MULT:
			tmpstr = "multiply";
			break;
		case IDL_BINOP_DIV:
			tmpstr = "divide";
			break;
		case IDL_BINOP_MOD:
			tmpstr = "modulo";
			break;
		default:
			tmpstr = "<IDL ERROR>";
		}
		return Py_BuildValue("(sNN)",
			tmpstr,
			IDLtree2Object(self->tree->u.idl_binop.left),
			IDLtree2Object(self->tree->u.idl_binop.right));
	case IDLN_UNARYOP:
		switch (self->tree->u.idl_unaryop.op) {
		case IDL_UNARYOP_PLUS:
			tmpstr = "plus";
			break;
		case IDL_UNARYOP_MINUS:
			tmpstr = "minus";
			break;
		case IDL_UNARYOP_COMPLEMENT:
			tmpstr = "complement";
			break;
		default:
			tmpstr = "<IDL ERROR>";
		}
	case IDLN_CODEFRAG:
	default:
fprintf(stderr, "get_value on NULL IDL_tree\n");
		/* XXX set exception */
		return NULL;
	}
}

static PyMethodDef IDLtree_methods[] = {
	{"free",	(PyCFunction)IDLtree_free,	1},
	{"get_type",	(PyCFunction)IDLtree_get_type,	1},
	{"get_value",	(PyCFunction)IDLtree_get_value,	1},
	{NULL,		NULL}		/* sentinel */
};

static PyObject *
IDLtree_getattr(self, name)
	IDLtreeObject *self;
	char *name;
{
	if (self->x_attr != NULL) {
		PyObject *v = PyDict_GetItemString(self->x_attr, name);
		if (v != NULL) {
			Py_INCREF(v);
			return v;
		}
	}
	return Py_FindMethod(IDLtree_methods, (PyObject *)self, name);
}

static int
IDLtree_setattr(self, name, v)
	IDLtreeObject *self;
	char *name;
	PyObject *v;
{
	if (self->x_attr == NULL) {
		self->x_attr = PyDict_New();
		if (self->x_attr == NULL)
			return -1;
	}
	if (v == NULL) {
		int rv = PyDict_DelItemString(self->x_attr, name);
		if (rv < 0)
			PyErr_SetString(PyExc_AttributeError,
			        "delete non-existing IDLtree attribute");
		return rv;
	}
	else
		return PyDict_SetItemString(self->x_attr, name, v);
}

statichere PyTypeObject IDLtree_Type = {
	/* The ob_type field must be initialized in the module init function
	 * to be portable to Windows without using C++. */
	PyObject_HEAD_INIT(NULL)
	0,			/*ob_size*/
	"IDLtree",			/*tp_name*/
	sizeof(IDLtreeObject),	/*tp_basicsize*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)IDLtree_dealloc, /*tp_dealloc*/
	0,			/*tp_print*/
	(getattrfunc)IDLtree_getattr, /*tp_getattr*/
	(setattrfunc)IDLtree_setattr, /*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
};
/* End IDLtree */

static PyObject *
libIDL_parse_filename(self, args)
	PyObject *self; /* Not used */
	PyObject *args;
{
	char *filename;
	IDL_tree tree;
	if (!PyArg_ParseTuple(args, "s", &filename))
		return NULL;
	if (IDL_parse_filename(filename, NULL, NULL, &tree, NULL, 0, IDL_ERROR)
		 != IDL_SUCCESS)
		return NULL;
	return ((PyObject *) IDLtree2Object(tree));
}


/* List of functions defined in the module */

static PyMethodDef libIDL_methods[] = {
	{"parse_filename",		libIDL_parse_filename,		1},
	{NULL,		NULL}		/* sentinel */
};

#ifndef DL_EXPORT
#define DL_EXPORT(x) x
#endif

DL_EXPORT(void)
initlibIDL()
{
	PyObject *m, *d;

	/* Initialize the type of the new type object here; doing it here
	 * is required for portability to Windows without requiring C++. */
	IDLtree_Type.ob_type = &PyType_Type;

	/* Create the module and add the functions */
	m = Py_InitModule("libIDL", libIDL_methods);

	/* Add some symbolic constants to the module */
	d = PyModule_GetDict(m);
	ErrorObject = PyErr_NewException("libIDL.error", NULL, NULL);
	PyDict_SetItemString(d, "error", ErrorObject);
}
