//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

//---------------------------------------------------------------------------

#ifndef VTIME_HH
#define VTIME_HH

#include <iostream.h>
#include "Types.hh"
#include "bool.hh"
#include <values.h>
#include <limits.h>

class UniversalInteger;
class UniversalReal;

struct VTime {
  friend inline ostream& operator<<(ostream& os, const VTime& vt) {
    os << (LONG) vt.time << "(" << (unsigned int) vt.delta << ") fs";
    return os;
  }

  friend inline istream& operator>>(istream& is, const VTime& vt) {
    is >> vt.time;
    is >> vt.delta;

    return is;
  }
  
public:
  
  LONG time;
  unsigned int delta;
  
  VTime() { time = 0LL; delta = 1;}
  VTime(const int t) { 
    time = t; 
    delta = 1; 
  }
  
  VTime(const unsigned int t) { time = t; delta = 1; }
  VTime(const double t) { time = (LONG) t; delta = 1; }
  VTime(const LONG t) { 
    time = t; 
    delta = 1; 
  }
  
  VTime(const LONG t, unsigned int d) { time = t; delta = d; };
  inline VTime(const VTime& t) {
    time = t.time;
    delta = t.delta;
  }

  static VTime getMax() {
    return VTime(9223372036854775807LL, 1);
  }

  static VTime getMin() {
    return VTime(0LL, 1);
  }

  inline VTime& operator=(const VTime& v) {
    time = v.time;
    delta = v.delta;
    return *this;
  }
  
  inline VTime operator+(const VTime& v) const {
    VTime result;
    result.time = time + v.time;
    if (v.time == 0) {
      result.delta = delta + 1;
    }
    else {
      result.delta = 1;
    }
    return result;
  }
  
  VTime operator-(const VTime& v) const {
    return VTime(time - v.time, 1);
  }
  
  VTime operator*(const VTime& v) const {
    return VTime(time * v.time, 1);
  }
  
  VTime& operator*=(const VTime& v) {
    time *= v.time;
    delta = 1;
    return *this;
  }
  
  bool operator==(const VTime& v) const {
    if ((time == v.time) && (delta == v.delta)) {
      return true;
    }
    return false;
  }
  
  bool operator!=(const VTime& v) const {
    if (time != v.time || (time == v.time && delta != v.delta)) {
      return true;
    }
    return false;
  }
  
  bool operator<=(const VTime& v) const {
    if ((time < v.time) || (time == v.time && delta <= v.delta)) {
      return true;
    }
    return false;
  }
  
  bool operator>=(const VTime& v) const {
    if ((time > v.time) || (time == v.time && delta >= v.delta)) {
      return true;
    }
    return false;
  }
  
  inline bool operator<(const VTime& v) const {
    if ((time < v.time) || (time == v.time && delta < v.delta)) {
      return true;
    }
    return false;
  }
  
  inline bool operator>(const VTime& v) const {
    if ((time > v.time) || (time == v.time && delta > v.delta)) {
      return true;
    }
    return false;
  }
  
  inline operator double() const {
    return (double)time;
  }
  
  int getSize() const { return sizeof(*this); }
  void print(ostream& os = cout) { os << *this; }
};

extern const VTime INVALIDTIME;
extern const VTime ZERO;
extern const VTime PINFINITY;
//static const VTime PINFINITY(INT_MAX, 1);

// extern bool savantEqual(const VTime&, const VTime&);
// extern bool savantNotEqual(const VTime&, const VTime&);
// extern bool savantLessThan(const VTime&, const VTime&);
// extern bool savantLessThanOrEqual(const VTime&, const VTime&);
// extern bool savantGreaterThan(const VTime&, const VTime&);
// extern bool savantGreaterThanOrEqual(const VTime&, const VTime&);

// extern VTime savantPlus(const VTime&);
// extern VTime savantMinus(const VTime&);
// extern VTime savantAbs(const VTime&);
// extern VTime savantPlus(const VTime&,const VTime&);
// extern VTime savantMinus(const VTime&,const VTime&);
// extern VTime savantMultiply(const VTime&, const UniversalInteger& );
// extern VTime savantMultiply(const VTime&, const UniversalReal&);
// extern VTime savantMultiply(const UniversalInteger& , const VTime&);
// extern VTime savantMultiply(const UniversalReal& , const VTime&);
// extern VTime savantDivide(const UniversalReal&, const VTime&);
// extern VTime savantDivide(const VTime&, const UniversalInteger&);
// extern VTime savantDivide(const VTime&, const UniversalReal&);

#endif
