// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//
//---------------------------------------------------------------------------

// This file contains the methods in VHDLKernel class that relate to signal
// attributes.  This file has been created to increase the readability of
// both VHDLKernel and SignalAttributes.

#include "VHDLKernel.hh"

void
VHDLKernel::updateAttribute(Attribute *attribute, 
			    const VHDLData &value) {
  switch(attribute->attrib) {
  case LAST_EVENT:
  case LAST_ACTIVE:
    ((PhysicalType *)attribute->value)->object->updateVal(UniversalLongLongInteger(((VHDLKernel_state *) state->current)->lVT.time));
    break;
  case LAST_VALUE:
    if(attribute->value->is_scalar_type() == true) {
      ((ScalarType *)attribute->value)->object->updateVal(value);
    } else {
      abort();
    }
    break;
  case QUIET:
  case EVENT:
  case ACTIVE:
  case STABLE:
    ((EnumerationType *)attribute->value)->object->updateVal(value);
    break;
  case TRANSACTION:
    ((EnumerationType *)attribute->value)->object->updateVal(*(VHDLData *)&(savantNot(*(EnumerationType *)attribute->value).object->readVal()));
    break;
  case DRIVING :
  case DRIVING_VALUE:
    cerr << "VHDLKernel::updateAttribute : Not yet implemented" << endl;
    break;
  default:
    cerr << "VHDLKernel::updateAttribute : (unknown attribute)" << endl;
    break;
  } // switch(attribute->attrib);
}

//     Attribute implementation: The attribute is called with a VHDLType and
// corresponding parameter as arguments.  This is the function defined below.
// This function finds the "kind" of the type using the "get_kind()" member
// function and correspondingly calls another function.  This function
// extracts the signal information from the type and calls yet another
// function which computes the attribute info.
//     This elaborate mechanism is to make the attribute evaluation
// transparent to scalar types, array types and record types.

// Note: The method "get_kind()" MUST NOT be overloaded by a derived type.
// This is to ensure that the type is recognized correctly.

EnumerationType
VHDLKernel::locateQuietAttribute(const VHDLType *sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);

  ASSERT(sig != NULL);
  ASSERT(time >= ZERO);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateQuietAttribute((const ScalarType*)sig, time);
    break;
  case ARRAY_TYPE:
    retval = locateQuietAttribute(((const ArrayType*)sig)->object, time);
    break;
  case RECORD_TYPE:
    retval = locateQuietAttribute((const RecordType*)sig, time);
    break;
  case VECTOR_BASE:
    retval = locateQuietAttribute((const VectorBase*)sig, time);
    break;
  default:
    cerr << "Unknown Type!! Asking for Quiet Attribute, Sssh, Keep QUIET ;-)"
	 << endl;
    break;
  }
  return retval;
}

EnumerationType
VHDLKernel::locateQuietAttribute(const ScalarType *sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);

  ASSERT(sig != NULL);
  ASSERT(time >= ZERO);

  retval = locateQuietAttribute((const SignalBase*)sig->object, time);

  return retval;
}			     

EnumerationType
VHDLKernel::locateQuietAttribute(const VectorBase *sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->numElems;

  retval = locateQuietAttribute((const VHDLType*)&(sig->get_element(0)),time);
  for(int i=1; i< noOfElements; i++) {
    tempval = locateQuietAttribute((const VHDLType*)&(sig->get_element(i)),time);
    retval = savantAnd(retval, tempval);
  }

  if (savantEqual(retval, SAVANT_BOOLEAN_TRUE) == SAVANT_BOOLEAN_TRUE)
    return SAVANT_BOOLEAN_TRUE;
  else
    return SAVANT_BOOLEAN_FALSE;
}

EnumerationType
VHDLKernel::locateQuietAttribute(const RecordType* sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->numberOfFields;

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateQuietAttribute((const VHDLType*)&(sig->get_field(i)),time);
    retval = savantAnd(retval, tempval);
  }
  return retval;
}

EnumerationType
VHDLKernel::locateQuietAttribute(const SignalBase *sig,
					const VTime time) {
  Attribute *attribute;
  Attribute *retval = NULL;
  UniversalLongLongInteger lastActiveTime;
  int i;

  ASSERT(sig != NULL);

  attribute = sig->attributeList;
  for (i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == LAST_ACTIVE) 
      retval = &attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == LAST_ACTIVE));
  lastActiveTime = ((ScalarType *)retval->value)->object->readVal();
 
  attribute = sig->attributeList;
  for (i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == QUIET) 
      retval = &attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == QUIET));

  if(time == ZERO) {
    if(savantLessThan(lastActiveTime, ((VHDLKernel_state *) state->current)->lVT.time)) {
      updateAttribute(retval, UNIVERSAL_TRUE);
    } else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  } else {
    if(savantLessThanOrEqual(lastActiveTime, 
			     ((VHDLKernel_state *) state->current)->lVT.time-time.time)){
      updateAttribute(retval, UNIVERSAL_TRUE);
    } else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  }
    
  return (*((EnumerationType*)retval->value));
}

EnumerationType
VHDLKernel::locateStableAttribute(const VHDLType *sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);
  ASSERT(sig != NULL);
  ASSERT(time >= ZERO);
  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateStableAttribute((const ScalarType*)sig, time);
    break;
  case ARRAY_TYPE:
    retval = locateStableAttribute(((const ArrayType*)sig)->object, time);
    break;
  case RECORD_TYPE:
    retval = locateStableAttribute((const RecordType*)sig, time);
    break;
  case VECTOR_BASE:
    retval = locateStableAttribute((const VectorBase*)sig, time);
    break;
  default:
    cerr << "Unknown Type!! Asking for Stable Attribute, Sssh, Keep QUIET ;-)"
	 << endl;
    break;
  }
  return retval;
}

EnumerationType
VHDLKernel::locateStableAttribute(const ScalarType *sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  ASSERT(sig != NULL);
  ASSERT(time >= ZERO);

  retval = locateStableAttribute((const SignalBase*)sig->object, time);

  return retval;
}			     

EnumerationType
VHDLKernel::locateStableAttribute(const VectorBase *sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->numElems;

  retval = locateStableAttribute((const VHDLType*)&(sig->get_element(0)),time);
  for(int i=0; i< noOfElements; i++) {
    tempval = locateStableAttribute((const VHDLType*)&(sig->get_element(i)),time);
    retval = savantAnd(retval, tempval);
  }
  return retval;
}

EnumerationType
VHDLKernel::locateStableAttribute(const RecordType* sig, const VTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->numberOfFields;

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateStableAttribute((const VHDLType*)&(sig->get_field(i)),time);
    retval = savantAnd(retval, tempval);
  }
  return retval;
}

EnumerationType
VHDLKernel::locateStableAttribute(const SignalBase *sig, 
					 const VTime time) {
  Attribute *attribute;
  Attribute *retval = NULL;
  UniversalLongLongInteger lastEventTime;
  int i;

  ASSERT(sig != NULL);

  attribute = sig->attributeList;
  for (i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == LAST_EVENT) 
      retval = &attribute[i];
  }

  ASSERT((retval != NULL && retval->attrib == LAST_EVENT));
  lastEventTime = ((ScalarType *)retval->value)->object->readVal();

  attribute = sig->attributeList;
  for (i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == STABLE) 
      retval = &attribute[i];
  }

  ASSERT((retval != NULL && retval->attrib == STABLE));
  UniversalLongLongInteger lvt(((VHDLKernel_state *) state->current)->lVT.time);
  UniversalLongLongInteger timeExpr(time.time);
  if(time == ZERO) {
    if(savantLessThan(lastEventTime, lvt)) {
      updateAttribute(retval, UNIVERSAL_TRUE);
    } else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  } else {
    if(savantLessThanOrEqual(lastEventTime, savantMinus(lvt, timeExpr))) {
      updateAttribute(retval, UNIVERSAL_TRUE);
    } else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  }

  return (*(EnumerationType *) retval->value);
}

EnumerationType 
VHDLKernel::locateEventAttribute(const VHDLType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateEventAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = 
      locateEventAttribute((const VectorBase*)(&((ArrayType*)sig)->get_array()));
    break;
  case RECORD_TYPE:
    retval = locateEventAttribute((const RecordType*)sig);
    break;
  case VECTOR_BASE: 
    retval = locateEventAttribute((const VectorBase*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for Event Attribute" << endl;
    break;
  }
  return retval;
}

EnumerationType
VHDLKernel::locateEventAttribute(const ScalarType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  retval = locateEventAttribute((SignalBase*)sig->object);

  return retval;
}			     

EnumerationType 
VHDLKernel::locateEventAttribute(const VectorBase *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  int noOfElements = sig->numElems;

  for(int i=0; i< noOfElements; i++) {
    tempval = locateEventAttribute((const VHDLType*)&(sig->get_element(i)));
    retval = savantOr(retval, tempval);
  }
  return retval;
}

EnumerationType 
VHDLKernel::locateEventAttribute(const RecordType* sig) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  int noOfElements = sig->numberOfFields;

  if(noOfElements >= 1) {
    retval = locateEventAttribute((const VHDLType*)&(sig->get_field(1)));
  }

  for(int i=2; i< noOfElements; i++) {
    tempval = locateEventAttribute((const VHDLType*)&(sig->get_field(i)));
    retval = savantOr(retval, tempval);
  }
  return retval;
}

EnumerationType 
VHDLKernel::locateEventAttribute(SignalBase *sig) {
  
  UniversalLongLongInteger lastEventTime;

    lastEventTime = sig->allAttributes.last_event;

    // taking the default stuff
    if(savantEqual(lastEventTime,UniversalLongLongInteger(((VHDLKernel_state *) state->current)->lVT)))    {
      sig->allAttributes.event = true;
      return SAVANT_BOOLEAN_TRUE;
    } 
    else 
    {
      sig->allAttributes.event = false;
      return SAVANT_BOOLEAN_FALSE;
    }    
}

EnumerationType *
VHDLKernel::locateTransactionAttribute(const VHDLType *sig) {
  
  EnumerationType *retval;
  ASSERT(sig != NULL);
  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateTransactionAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateTransactionAttribute(((const ArrayType*)sig)->object);
    break;
  case RECORD_TYPE:
    retval = locateTransactionAttribute((const RecordType*)sig);
    break;
  case VECTOR_BASE:
    retval = locateTransactionAttribute((const VectorBase*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for Transaction Attribute !"
	 << endl;
    break;
  }
  return retval;
}

EnumerationType*
VHDLKernel::locateTransactionAttribute(const ScalarType *sig) {

 EnumerationType *retval;

  ASSERT(sig != NULL);

  retval = locateTransactionAttribute((const SignalBase*)sig->object);

  return retval;
}			     

EnumerationType *
VHDLKernel::locateTransactionAttribute(const VectorBase *sig) {
  EnumerationType *retval;
  EnumerationType *tempval;
  int noOfElements = sig->numElems;

  for(int i=0; i< noOfElements; i++) {
    tempval = locateTransactionAttribute((const VHDLType*)&(sig->get_element(i)));
    *retval = savantAnd(*retval, *tempval);
    delete tempval;
  }
  return retval;
}

EnumerationType *
VHDLKernel::locateTransactionAttribute(const RecordType* sig) {
  EnumerationType *retval;
  EnumerationType *tempval;
  int noOfElements = sig->numberOfFields;

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateTransactionAttribute((const VHDLType*)&(sig->get_field(i)));
    *retval = savantAnd(*retval, *tempval);
    delete tempval;
  }
  return retval;
}

EnumerationType *
VHDLKernel::locateTransactionAttribute(const SignalBase *sig) {

  Attribute *attribute;
  Attribute *retval = NULL;
  
  attribute = sig->attributeList;
  for (int i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == TRANSACTION) 
      retval = &attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == TRANSACTION));
  return ((EnumerationType *)retval->value);
}  

// ***************** Active Attributes **********************************
 
EnumerationType
VHDLKernel::locateActiveAttribute(const VHDLType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateActiveAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateActiveAttribute(((const ArrayType*)sig)->object);
    break;
  case RECORD_TYPE:
    retval = locateActiveAttribute((const RecordType*)sig);
    break;
  case VECTOR_BASE:
    retval = locateActiveAttribute((const VectorBase*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for Active Attribute !"
	 << endl;
    break;
  }
  return retval;
}


EnumerationType
VHDLKernel::locateActiveAttribute(const ScalarType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  retval = locateActiveAttribute((const SignalBase*)sig->object);

  return retval;
}			     

EnumerationType
VHDLKernel::locateActiveAttribute(const VectorBase *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, SavantbooleanType_info);
  int noOfElements = sig->numElems;

  for(int i=0; i< noOfElements; i++) {
    tempval = locateActiveAttribute((const VHDLType*)&(sig->get_element(i)));
    retval = savantOr(retval, tempval);
  }
  return retval;
}


EnumerationType 
VHDLKernel::locateActiveAttribute(const RecordType* sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, SavantbooleanType_info);
  int noOfElements = sig->numberOfFields;

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateActiveAttribute((const VHDLType*)&(sig->get_field(i)));
    retval = savantOr(retval, tempval);
  }
  return retval;
}

EnumerationType
VHDLKernel::locateActiveAttribute(const SignalBase *sig) {
  Attribute *attribute;
  Attribute *retval = NULL;
  UniversalLongLongInteger lastActiveTime;
  int i;

  ASSERT(sig != NULL);

  attribute = sig->attributeList;
  for(i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == LAST_ACTIVE) 
      retval = &attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == LAST_ACTIVE));
  lastActiveTime = (UniversalLongLongInteger &) ((PhysicalType *)retval->value)->object->readVal();
 
  attribute = sig->attributeList;
  for(i = 0; i <sig->numAttributes; i++) {
    if(attribute[i].attrib == ACTIVE) 
      retval = &attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == ACTIVE));

  if(savantEqual(lastActiveTime,UniversalLongLongInteger(((VHDLKernel_state *) state->current)->lVT.time))){
    updateAttribute(retval, UNIVERSAL_TRUE);
    return (*(EnumerationType *) retval->value);
  } else {
    updateAttribute(retval, UNIVERSAL_FALSE);
    return (*(EnumerationType *) retval->value);
  }

}

// ***************** last_event Attributes **********************************
 
PhysicalType
VHDLKernel::locateLastEventAttribute(const VHDLType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateLastEventAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateLastEventAttribute(((const ArrayType*)sig)->object);
    break;
  case RECORD_TYPE:
    retval = locateLastEventAttribute((const RecordType*)sig);
    break;
  case VECTOR_BASE:
    retval = locateLastEventAttribute((const VectorBase*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for last_event Attribute !"
	 << endl;
    break;
  }
  return retval;
}


PhysicalType
VHDLKernel::locateLastEventAttribute(const ScalarType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);

  ASSERT(sig != NULL);

  retval = locateLastEventAttribute((const SignalBase*)sig->object);

  return retval;
}			     

PhysicalType
VHDLKernel::locateLastEventAttribute(const VectorBase *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  int noOfElements = sig->numElems;

  for(int i=0; i< noOfElements; i++) {
    tempval = locateLastEventAttribute((const VHDLType*)&(sig->get_element(i)));
    if(savantLessThan(tempval, retval) == SAVANT_BOOLEAN_TRUE) {
      retval = tempval;
    }
  }
  return retval;
}


PhysicalType
VHDLKernel::locateLastEventAttribute(const RecordType* sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  int noOfElements = sig->numberOfFields;

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateLastEventAttribute((const VHDLType*)&(sig->get_field(i)));
    if(savantLessThan(tempval, retval) == SAVANT_BOOLEAN_TRUE) {
      retval = tempval;
    }
  }
  return retval;
}

PhysicalType
VHDLKernel::locateLastEventAttribute(const SignalBase *sig) {
  UniversalLongLongInteger lastEventTime;
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);

  ASSERT(sig != NULL);

  lastEventTime = sig->allAttributes.last_event;
  lastEventTime = savantMinus(UniversalLongLongInteger(int(((VHDLKernel_state *) state->current)->lVT.time)),lastEventTime);

  retval = PhysicalType(ObjectBase::VARIABLE, lastEventTime, SavanttimeType_info);
  return retval;

}

// ***************** last_active Attributes **********************************
 
PhysicalType
VHDLKernel::locateLastActiveAttribute(const VHDLType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateLastActiveAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateLastActiveAttribute(((const ArrayType*)sig)->object);
    break;
  case RECORD_TYPE:
    retval = locateLastActiveAttribute((const RecordType*)sig);
    break;
  case VECTOR_BASE:
    retval = locateLastActiveAttribute((const VectorBase*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for last_active Attribute !"
	 << endl;
    break;
  }
  return retval;
}

PhysicalType
VHDLKernel::locateLastActiveAttribute(const ScalarType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);

  ASSERT(sig != NULL);

  retval = locateLastActiveAttribute((const SignalBase*)sig->object);

  return retval;
}			     

PhysicalType
VHDLKernel::locateLastActiveAttribute(const VectorBase *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  int noOfElements = sig->numElems;

  for(int i=0; i< noOfElements; i++) {
    tempval = locateLastActiveAttribute((const VHDLType*)&(sig->get_element(i)));
    if(savantLessThan(tempval, retval) == SAVANT_BOOLEAN_TRUE) {
      retval = tempval;
    }
  }
  return retval;
}


PhysicalType
VHDLKernel::locateLastActiveAttribute(const RecordType* sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);
  int noOfElements = sig->numberOfFields;

  for(int i=0; i< noOfElements; i++) {
    tempval = locateLastActiveAttribute((const VHDLType*)&(sig->get_field(i)));
    if(savantLessThan(tempval, retval) == SAVANT_BOOLEAN_TRUE) {
      retval = tempval;
    }
  }
  return retval;
}

PhysicalType
VHDLKernel::locateLastActiveAttribute(const SignalBase *sig) {
  Attribute *attribute;
  Attribute *attribval = NULL;
  UniversalLongLongInteger lastActiveTime;
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::MAX, SavanttimeType_info);

  ASSERT(sig != NULL);

  attribute = sig->attributeList;
  for (int i = 0; i < sig->numAttributes; i++) {
    if(attribute[i].attrib == LAST_ACTIVE) 
      attribval = &attribute[i];
  }

  ASSERT(attribval != NULL && attribval->attrib == LAST_ACTIVE);

  lastActiveTime = (UniversalLongLongInteger &)((PhysicalType *)attribval->value)->object->readVal();
  lastActiveTime = savantMinus(UniversalLongLongInteger(int(((VHDLKernel_state *) state->current)->lVT.time)), lastActiveTime);
  retval = PhysicalType(ObjectBase::VARIABLE, lastActiveTime, SavanttimeType_info);

  return retval;
}

// ***************** last_value Attributes **********************************
 
VHDLType *
VHDLKernel::locateLastValueAttribute(const VHDLType *sig) {
  VHDLType *retval;

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateLastValueAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateLastValueAttribute(((const ArrayType*)sig)->object);
    break;
  case RECORD_TYPE:
    retval = locateLastValueAttribute((const RecordType*)sig);
    break;
  case VECTOR_BASE:
    retval = locateLastValueAttribute((const VectorBase*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for last_value Attribute !"
	 << endl;
    break;
  }
  return retval;
}

VHDLType *
VHDLKernel::locateLastValueAttribute(const ScalarType *sig) {
  VHDLType *retval;
  static ScalarType* ptr = NULL;
  ObjectBase* objectPtr = NULL;
  ASSERT(sig != NULL);

  retval = locateLastValueAttribute((const SignalBase*)sig->object);
  if(retval == NULL) {
    switch(((SignalBase*)sig->object)->allAttributes.sigtype) {
    case VHDLData::UNIVERSAL_BOOLEAN:
    case VHDLData::UNIVERSAL_INTEGER:
      objectPtr = new Variable<UniversalInteger>;
      objectPtr->updateVal(UniversalInteger(((SignalBase*)sig->object)->allAttributes.last_value.inttype));
      break;
    case VHDLData::UNIVERSAL_REAL:
      objectPtr = new Variable<UniversalReal>;
      objectPtr->updateVal(UniversalReal(((SignalBase*)sig->object)->allAttributes.last_value.floattype));
      break;
    case VHDLData::UNIVERSAL_LONG_LONG_INTEGER:
      objectPtr = new Variable<UniversalLongLongInteger>;
      objectPtr->updateVal(UniversalLongLongInteger((LONG) ((SignalBase*)sig->object)->allAttributes.last_value.physicaltype));
      break;
    default:
      objectPtr = NULL;
      break;
    }
    if(ptr != NULL) {
      delete ptr;
    }
    ptr = new ScalarType(ObjectBase::VARIABLE);
    ptr->object = objectPtr;
    retval = ptr;
  }
  return retval;
}			     

VHDLType *
VHDLKernel::locateLastValueAttribute(const VectorBase * /* sig */) {
  VHDLType *retval = NULL ;
  //VHDLType *tempval;
  //int noOfElements = sig->numElems;
  //for(int i=0; i< noOfElements; i++) {
  //  tempval = locateLastValueAttribute((VHDLType*)&(sig->get_element(i)));
  //}

  cerr << "VHDLKernel<State>::locateLastValueAttribute() : This will not work" 
       << endl; 
  abort();

  return retval;
}


VHDLType *
VHDLKernel::locateLastValueAttribute(const RecordType* /* sig */) {
  VHDLType *retval = NULL;
//   VHDLType *tempval;
//   int noOfElements = sig->numberOfFields;

//   retval = locateLastValueAttribute((VHDLType*)&(sig->get_field(0)));
//   for(int i=1; i< noOfElements; i++) {
//     tempval = locateLastValueAttribute((VHDLType*)&(sig->get_field(i)));
//     *retval = savantMinus(*retval, *tempval);
//     delete tempval;
//   }

  cerr << "VHDLKernel<State>::locateLastValueAttribute():This will not work" 
       << endl; 
  abort();
  return retval;
}

// Caution: Does not return a new'ed pointer.  Retruns the same poointer
// as in the attribute.

VHDLType *
VHDLKernel::locateLastValueAttribute(const SignalBase *sig) {
  Attribute *attribute;
  Attribute *attribval = NULL;

  ASSERT(sig != NULL);

  attribute = sig->attributeList;
  for (int i = 0; i < sig->numAttributes; i++) {
    if(attribute[i].attrib == LAST_VALUE)
      attribval = &attribute[i];
  }

  //  ASSERT(attribval != NULL && attribval->attrib == LAST_VALUE);
  if(attribval != NULL && attribval->attrib == LAST_VALUE) {
    return attribval->value;
  }
  else {
    return NULL;
  }
}
