/*
 * RT-Linux scheduler compatibility functions
 *
 * Written by Michael Barabanov
 * Copyright (C) VJY Associates LLC, 1998,1999
 * Released under the terms of the GNU  General Public License
 *
 */

#include <rtl_conf.h>
#ifdef CONFIG_RTL_USE_V1_API

#include <linux/errno.h>
#include <linux/malloc.h>
#include <asm/system.h>
#include <asm/segment.h>

#include <rtl_time.h>
#include <rtl_sync.h>
#include <rtl_sched.h>
#include <rtl_tqueue.h>

struct rtl_compat_struct {
	void (*fn)(int data);
	int data;
};

static void *rtl_compat_start_routine(void *compat)
{
	void (*fn)(int data) = ((struct rtl_compat_struct *) compat) -> fn;
	int data = ((struct rtl_compat_struct *) compat) -> data;
	pthread_wait_np();
	fn(data);
	return NULL;
}

int rt_task_init (RT_TASK *t, void (*fn)(int data), int data, int stack_size, int priority)
{
	int ret;
	pthread_attr_t attr;
	rtl_sched_param param;
	struct rtl_compat_struct compat; /* it's ok to use this structure in the thread
					   as the priority of the newly created thread is higher
					   than that of Linux */
	schedule_t *s;

	s = LOCAL_SCHED;
	compat.fn = fn;
	compat.data = data;
	pthread_attr_init (&attr);
	pthread_attr_setstacksize(&attr, stack_size);
	param.sched_priority = sched_get_priority_max(0) - priority; /* note the priority scheme is inverted to correspond to the POSIX one */
	pthread_attr_setschedparam(&attr, &param);
	ret = pthread_create(t, &attr, rtl_compat_start_routine, &compat);
	return -ret;
}


#ifdef __SMP__
static void rt_task_suspend_wrapper(void *data)
{
	pthread_t thread = (pthread_t) (data);
	thread->state = RTL_THREAD_DORMANT;
}
#endif

int rt_task_suspend (RT_TASK *task)
{
	pthread_t thread = *task;
#ifdef __SMP__
	DECLARE_CPUID(cpu_id);
	struct tq_struct tq;
	if (thread->cpu != cpu_id) {
		tq . next = 0;
		tq . sync = 0;
		tq . data = thread;
		tq . routine = rt_task_suspend_wrapper;
		rtl_queue_task (&tq, &RTL_TQ(thread->cpu));
		rtl_reschedule (thread->cpu);
		rtl_tq_sync (&tq);
	} else
#endif 
	{
		rtl_irqstate_t flags;
		rtl_no_interrupts(flags);
		thread->state = RTL_THREAD_DORMANT;
		rtl_schedule();
		rtl_restore_interrupts(flags);
	}
	return 0;
}


#ifdef __SMP__
static void rt_task_wakeup_wrapper(void *data)
{
	pthread_t thread = (pthread_t) (data);
	thread->state = RTL_THREAD_READY;
}
#endif


int rt_task_wakeup (RT_TASK *task)
{
	pthread_t thread = *task;
#ifdef __SMP__
	DECLARE_CPUID(cpu_id);
	struct tq_struct tq;
	if (thread->cpu != cpu_id) {
		tq . next = 0;
		tq . sync = 0;
		tq . data = thread;
		tq . routine = rt_task_wakeup_wrapper;
		rtl_queue_task (&tq, &RTL_TQ(thread->cpu));
		rtl_reschedule (thread->cpu);
		rtl_tq_sync (&tq);
	} else
#endif 
	{
		rtl_irqstate_t flags;
		rtl_no_interrupts(flags);
		thread->state = RTL_THREAD_READY;
		rtl_schedule();
		rtl_restore_interrupts(flags);
	}
	return 0;
}


#ifdef __SMP__
struct rt_task_make_periodic_param {
	pthread_t *threadptr;
	RTIME start_time;
	RTIME period;
};

static void rt_task_make_periodic_wrapper (void *data)
{
	struct rt_task_make_periodic_param *param = (struct rt_task_make_periodic_param *) data;
	(*(param->threadptr))->resume_time = param->start_time;
	(*(param->threadptr))->period = param->period;
	(*(param->threadptr))->state = RTL_THREAD_DELAYED;
}

#endif

int rt_task_make_periodic (RT_TASK *task, RTIME start_time, RTIME period)
{
	pthread_t thread = *task;
#ifdef __SMP__
	DECLARE_CPUID(cpu_id);
	struct tq_struct tq;
	struct rt_task_make_periodic_param param = {task, start_time, period};
	if (thread->cpu != cpu_id) {
		tq . next = 0;
		tq . sync = 0;
		tq . data = &param;
		tq . routine = rt_task_make_periodic_wrapper;
		rtl_queue_task (&tq, &RTL_TQ(thread->cpu));
		rtl_reschedule (thread->cpu);
		rtl_tq_sync (&tq);
	} else

#endif
	{
		
		thread->resume_time = start_time;
		thread->period = period;
		thread->state = RTL_THREAD_DELAYED;
		rtl_schedule();
	}
	return 0;
}

#endif /* CONFIG_RTL_USE_V1_API */
