/*
  cpp.ll

  Copyright (c) 1996 Roland Wunderling, Malte Zoeckler
  Copyright (c) 1998 Michael Meeks
  Copyright (c) 1999 Dragos Acostachioaie

  This file is part of DOC++.

  DOC++ is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License as published by the Free Software Foundation; either
  version 2 of the license, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

%{

#include <assert.h>
#include <ctype.h>
#include <iostream.h>
#include <stdarg.h>
#include <stdio.h>

#include "doc.h"

static const char*	inputString ;
static int		inputPosition ;
static int		lastCContext ;
static int		protection   = PUBL ;
static int		curlyBracketCount = 0 ;
static int		innerCurlyCount = 0 ;
static int              skipCurlyStart = 0;
static int		roundBracketCount = 0 ;
static int		skipReturn   = 0 ;
static int		sharpCount   = 0 ;
static Entry*		current_root = 0 ;
static Entry*		global_root  = 0 ;
static Entry*		current      = 0 ;
static int		current_done = 0 ;
static Entry*		last         = 0 ; // For handling trailing comments

McDArray<namespace_entry *> namespace_table;

static int		yyLineNr = 0 ;
static char		yyFileName[264] ;

static void	msg( const char* str, const char* str2=0 )
{
#ifdef DEBUG
    if( verb )
    {
	printf("%s(%d): %s", yyFileName, yyLineNr, str);
	if( str2 )
	    printf("`%s'", str2);
	printf( "\n" ) ;
    }
#endif
}

static void lineCount()
{
    const char *c;
    for(c = yytext; *c; ++c)
	yyLineNr += (*c == '\n');
}

static void addType(Entry *current)
{
    current->type = current->name;
    current->name.clear();
    current->type += current->args;
    current->args.clear();
    current->startLine = yyLineNr;
}

static char nonewline(char c)
{
    return (c == '\n') ? ' ' : c;
}

#undef YY_INPUT
#define	YY_INPUT(buf, result, max_size) result = yyread(buf, max_size);

static int yyread(char *buf, int max_size)
{
    int c = 0;
    while(c < max_size && inputString[inputPosition])
	{
	*buf = inputString[inputPosition++];
	c++;
	buf++;
	}
    return c;
}

%}

%x	Cxx_Memo
%x	SubDoc
%x	SubDocComment
%x	SubDocCppComment
%x	Namespace
%x	UsingNamespace
%x	Doc
%x	VerbDoc
%x	Define
%x	DefineEnd
%x	DefineEnded

%x	ClassName
%x	Bases

%x	NextSemi
%x	FindMembers
%x	FindMembersSuffix
%x	GrabSuffixMemo
%x	FindMembersName
%x	Function
%x	Operator
%x	Throws

%x	Array
%x	Round
%x	Curly
%x	SkipCurly
%x	SkipInits
%x	SkipCPP
%x	SkipSemiOrCurly
%x	SkipSemiAndCurly
%x	Sharp

%x	Comment
%x	SkipComment
%x	SkipCxxComment

%%

<*>\x06[^\x06]*\x06			{ 
                                        int i;
					for(i = 0; yytext[i + 1] != 6; i++)
					    yyFileName[i] = yytext[i + 1];
					yyFileName[i] = 0;
					}

"'"[{}]"'"				{
					}

^[ \t]*"#i".*   			{ // kill `#ifdef', `#ifndef'
					}

^[ \t]*"#e".* 				{ // kill `#else', `#endif'
					}

^[ \t]*"////".* 			{ // kill `////' comments
					}

^[ \t]*"$"[ \t]*"$"			{ // kill RCS keywords
					BEGIN(SkipSemiOrCurly);
					}

<NextSemi>"{"				{ // Array inits
					  skipReturn=NextSemi ;
					  skipCurlyStart = curlyBracketCount;
					  BEGIN( SkipCurly ) ;
					}

<NextSemi>[;,]				{
					BEGIN(FindMembersSuffix);
					}

<FindMembers>[ \t]*"public"[ \t\n]*":"[ \t\n]*	{
					current->protection = protection = PUBL;
					lineCount();
					}

<FindMembers>[ \t]*"protected"[ \t\n]*":"[ \t\n]*	{
					current->protection = protection = PROT;
					lineCount();
					}

<FindMembers>[ \t]*"private"[ \t\n]*":"[ \t\n]*	{
					current->protection = protection = PRIV;
					lineCount();
					}

<FindMembers>[ \t]*"namespace"[ \t\n]+	{
					current->section = NAMESPACE_SEC;
					current->type = "namespace";
					lineCount();
					BEGIN(Namespace);
					}

<FindMembers>[ \t]*"using"[ \t\n]+"namespace"[ \t\n]+	{
					lineCount();
					BEGIN(UsingNamespace);
					}

<FindMembers>[ \t]*"typedef"[ \t\n]+"class"[ \t\n]+	{
					current->section = TYPEDEF_SEC;
					current->type = "typedef class";
					current->name.clear();
					lineCount();
					}

<FindMembers>[ \t]*"typedef"[ \t\n]+"struct"[ \t\n]+	{
					current->section = TYPEDEF_SEC;
					current->type = "typedef struct";
					current->name.clear();
					lineCount();
					}

<FindMembers>[ \t]*"typedef"[ \t\n]+"enum"[ \t\n]+	{
					current->section = TYPEDEF_SEC;
					current->type = "typedef enum";
					current->name.clear();
					lineCount();
					}

<FindMembers>[ \t]*"typedef"[ \t\n]+"union"[ \t\n]+	{
					current->section = TYPEDEF_SEC;
					current->type = "typedef union";
					current->name.clear();
					lineCount();
					}

<FindMembers>[ \t]*"typedef"[ \t\n]+	{
					current->section = TYPEDEF_SEC;
					current->type = "typedef";
					current->name.clear();
					lineCount();
					}

<FindMembers>[ \t]*"class"[ \t\n]+	{
					current->section = CLASS_SEC;
					addType( current ) ;
					current->type = "class";
					lineCount();
					BEGIN(ClassName);
					}

<FindMembers>[ \t]*"struct"[ \t\n]+	{
					current->section = UNION_SEC;
					addType(current);
					current->type += "struct";
					lineCount();
					BEGIN(ClassName);
					}

<FindMembers>[ \t]*"enum"[ \t\n]+	{
					current->section = UNION_SEC;
					addType(current);
					current->type += "enum";
					lineCount();
					BEGIN(ClassName);
					}

<FindMembers>[ \t]*"union"[ \t\n]+	{
					current->section = UNION_SEC;
					addType(current);
					current->type += "union";
					lineCount();
					BEGIN(ClassName);
					}

<FindMembers>[ \t]*"inline"[ \t\n]+	{
					}

<FindMembers>"operator"/[^a-z_A-Z0-9]	{
					addType(current);
					current->name = yytext;
					BEGIN(Operator);
					}

<Operator>[ \t\n]*"()"?[^(]*		{
					current->name += yytext;
					BEGIN(FindMembersName);
					}

<FindMembers,FindMembersName>[ \t]*":"[ \t]*[0-9]+[ \t]*/";"
					{ // kill obscure bit-width stuff
					}

<FindMembers>::				{ // Must append to name but if previous
					  // text is current classname discard it.
					  if ( (strcmp (current->name, current_root->name) == 0) )
					    current->name.clear() ;
					  else
				            current->name+= yytext ;
					  BEGIN( FindMembersName );
					}

<FindMembers>[a-z_A-Z~.0-9]+		{ // Normal name
					addType(current);
					current->name = yytext;
					}

<FindMembers>"}"			{
					if(current_root->section == NAMESPACE_SEC)
					  current_root = current_root->parent;
					}

<FindMembersName>"operator"/[^a-z_A-Z0-9] {
 					  current->name += yytext ;
 					  BEGIN( Operator ) ;
 					}

<FindMembersName>[a-z_A-Z~.0-9]+	{
					current->name += yytext;
					}

<FindMembersName>.			{ yyless(0) ;
					  BEGIN( FindMembers );
 					}

<FindMembers>^[ \t]*"#"			{ BEGIN( SkipCPP ) ; }
<SkipCPP>.
<SkipCPP>"\\"[ \t]*"\n"			{ yyLineNr++ ; }
<SkipCPP>\n				{ yyLineNr++ ;
					  BEGIN( FindMembers ) ;
					}

<FindMembers>^[ \t]*"#"[ \t]*define[ \t]+ {
					current->type = "#define" ;
					BEGIN( Define ) ;
					}

<Define>[a-z_A-Z:.0-9]+			{
					current->name = yytext;
					BEGIN(DefineEnd);
					}

<DefineEnd>[ \t]			{ BEGIN( DefineEnd ) ; }
<DefineEnd>"("[^)]*")"			{ current->args = yytext ;
					  BEGIN( DefineEnded ) ;
					}
<DefineEnd,DefineEnded>"\\"[ \t]*"\n"	{ yyLineNr++ ; }
<DefineEnd,DefineEnded>"\n"		{ if(( current_done == onlyDocs ) && (!noDefines))
					  {
					    current->section = MACRO_SEC ;
					    msg("found macro ", current->name);
					    current->file = yyFileName ;
					    current_root->addSubEntry( current ) ;
					    last = current ;
					    current = new Entry ;
					    current->protection = protection;
					  }
					  else
					  {
					    current->name.clear() ;
					    current->type.clear() ;
					  }
					  yyLineNr++ ;
					  current_done = 0;
					  BEGIN( FindMembers ) ;
					}

<FindMembers>[*&]+			{
					current->name += yytext;
					}

<FindMembers>[;=,]			{
					BEGIN(FindMembersSuffix);
					if(current_done == onlyDocs)
					    {
					    msg("found variable ", current->name);
					    if(current->section != TYPEDEF_SEC)
						current->section = VARIABLE_SEC;
					    current->file = yyFileName;
					    current_root->addSubEntry(current);
				    	    last = current;
					    current = new Entry;
					    // So `int a, b' gives `int a, int b'
					    if(*yytext == ',')
						current->type = (char *)last->type;
					    current->protection = protection;
					    }
					 else
					    {
					    current->section = EMPTY_SEC;
					    current->name.clear();
					    current->type.clear();
					    current->args.clear();
					    }
					current_done = 0;
					if(*yytext == '=')
					    BEGIN(NextSemi);
					}

<FindMembers>"["			{ current->args += yytext ;
					  sharpCount=1;
					  BEGIN( Array ) ;
					}
<Array>"]"				{ current->args += *yytext ;
					  if (--sharpCount<=0)
	                                     BEGIN( FindMembers );
					}
<Array>"["				{ current->args += *yytext ;
					  sharpCount++;	
					}
<Array>.				{ current->args += *yytext ; }

<FindMembers>"<"			{ addType( current ) ;
					  current->type += yytext ;
					  sharpCount=1;
					  BEGIN( Sharp ) ;
					}
<Sharp>">"				{ current->type += *yytext ;
					  if (--sharpCount<=0)
	                                     BEGIN( FindMembers ) ;
					}
<Sharp>"<"				{ current->type += *yytext ;
					  sharpCount++;	
					}
<Sharp>.				{ current->type += *yytext ; }

<Curly>[^\n{}"/]*			{ current->program += yytext ; }
<Curly>"//".*				{ current->program += yytext ; }
<Curly>\"[^\n"]*]\"			{ current->program += yytext ; }
<Curly>"/*"\**[ \t]*			{ current->program += yytext ;
					  BEGIN( Comment ) ;
					}
<Curly>"/*"\**[ \t]*\n			{ current->program += yytext ;
					  ++yyLineNr ;
					  BEGIN( Comment ) ;
					}

<Curly>"{"				{
					current->program += yytext;
					++curlyBracketCount;
					++innerCurlyCount;
					}

<Curly>"}"				{
					if(curlyBracketCount > 0)
					    --curlyBracketCount;
					if(innerCurlyCount > 0)
					    {
					    // do we have any ``using namespace'' at this level?
					    // if so, remove them
					    for(int i = 0; i < namespace_table.size(); i++)
						if(namespace_table[i]->innerCurlyCount == innerCurlyCount)
						    {
						    delete namespace_table[i]->name;
						    namespace_table.remove(i);
						    }
					    current->program += yytext;
					    --innerCurlyCount;
					    }
					else
					    {
					    if((current_done == onlyDocs) &&
						(current->name.length() > 0))
				    		{
						current_root->addSubEntry(current);
						current->file = yyFileName;
						last = current;
						current = new Entry;
						current->protection = protection;
						}
					    else
						{
						current->name.clear();
						current->type.clear();
						current->args.clear();
						current->section = EMPTY_SEC;
						}
					    current_done = 0;
					    BEGIN(NextSemi);
					    }
					}

<Curly>\n				{ current->program += yytext ;
					  yyLineNr++ ;
					}
<Curly>.				{ current->program += yytext ; }

<FindMembers>"("			{ 
					  current->args = yytext ;
					  BEGIN( Round ) ;
					}
<Round>"("				{ current->args += *yytext ;
					  ++roundBracketCount ;
					}
<Round>")"				{ current->args += *yytext ;
					  if( roundBracketCount )
					    --roundBracketCount ;
					  else
					    BEGIN( Function ) ;
					}
<Round>[ \t\n]*","[ \t\n]*		{ lineCount() ; current->args += ", " ; }
<Round>[ \t\n]+				{ lineCount() ; current->args += ' ' ; }
<Round>.				{ current->args += *yytext ; }

<Function>[ \t]*"const"[ \t\n]*		{ current->args += " const " ;
					  lineCount() ;
					}

<Function>[ \t]*"throw"[ \t\n]*"("	{ current->args += " throw(" ;
					  lineCount() ;
					  BEGIN( Round ) ;
					}

<Function>"("				{
					current->type += current->name;
					current->name = (char *)current->args;
					current->args = yytext;
					BEGIN(Round);
					}

<Function>[":"";""{"]			{ msg("found method ", current->name);
					  if( current_done == onlyDocs )
					  {
					    current->section = FUNCTION_SEC ;
					    current->file = yyFileName ;
					    current_root->addSubEntry(current);
					    last = current ;
					    current = new Entry ;
					    current->protection = protection ;
					  }
					  else
					  {
					    current->name.clear() ;
					    current->type.clear() ;
					    current->args.clear() ;
					  }
					  current_done = 0;
					  if( *yytext == '{' ) {
					      skipReturn=SkipSemiAndCurly ;
					      skipCurlyStart = curlyBracketCount;
					      BEGIN( SkipCurly ) ;
					  }
					  else if( *yytext == ':' )
					      BEGIN( SkipInits ) ;
					  else
					      BEGIN( FindMembersSuffix ) ;
					}
<Function>.				{ }
<FindMembersSuffix>\n			{ yyless(0) ;
                                          BEGIN( FindMembers ) ; }

<FindMembersSuffix>.*"//"[ \t]*		  { if (commentExtn)
				 	    {
				            msg ("appended note to ", last->name) ;
				            last->memo+=" ";
				            BEGIN( GrabSuffixMemo ) ;
					    }
					    else
					    {
					      yyless(0) ;
 					      BEGIN( FindMembers );
					    }
					  }
<FindMembersSuffix>(.*"///"[ \t]*)|(.*"/**"[ \t]*) {
					  if (QuantelExtn)
				          {
				            msg ("appended note to ", last->name) ;
				            last->memo+=" ";
				            BEGIN( GrabSuffixMemo ) ;
					  }
					  else
					  {
					    yyless(0) ;
					    BEGIN( FindMembers ) ;
					  }
					}
<FindMembersSuffix>.			{ yyless(0) ;
					  BEGIN( FindMembers ) ;
					}
				     
<GrabSuffixMemo>.			{ last->memo += nonewline(*yytext); }
<GrabSuffixMemo>"*/"|(\n)	       	{ lineCount() ;
                                            BEGIN( FindMembers ) ;
  					}

<SkipSemiOrCurly>;			{ BEGIN( FindMembers ) ; }
<SkipSemiOrCurly>"{"			{ skipReturn = FindMembers ;
					  skipCurlyStart = curlyBracketCount;
					  BEGIN( SkipCurly ) ;
					}

<SkipCurly>"{"				{
					++curlyBracketCount;
					}

<SkipCurly>"}"				{
					if(curlyBracketCount > skipCurlyStart)
					  --curlyBracketCount;
					 else
					  BEGIN(skipReturn);
					}

<SkipInits>"{"				{ skipReturn = SkipSemiAndCurly ;
					  skipCurlyStart = curlyBracketCount;
					  BEGIN( SkipCurly ) ;
					}
<SkipInits>";"				{ BEGIN( FindMembers ) ; }
<SkipSemiAndCurly>[ \t\n]
<SkipSemiAndCurly>("///")|("/**") {
					    yyless(0) ;
					    BEGIN( FindMembers ) ;
					}
<SkipSemiAndCurly>.			{ // IE. a semicolon ( or anything else )
					  if (*yytext != ';')
					    yyless(0) ;
					  BEGIN( FindMembers ) ;
					}

<ClassName>";"				{ // Remove forward class definition
					current->section = EMPTY_SEC;
					current->type.clear();
					current->name.clear();
					current->args.clear();
					BEGIN(FindMembersSuffix);
					}

<Bases>";"				{ 
					  if( current_done == onlyDocs)
					  {
					    current->section = VARIABLE_SEC ;
					    current_root->addSubEntry(current);
					    current->file = yyFileName ;
					    last = current ;
					    current = new Entry ;
					    current->protection = protection ;
					  }
					  else
					  {
					    current->section = EMPTY_SEC ;
					    current->type.clear() ;
					    current->name.clear() ;
					    current->args.clear() ;
					  }
					  current_done = 0;
					  BEGIN( FindMembersSuffix ) ;
					}

<ClassName>[a-z_A-Z0-9\[\]*&]+		{
					if(current->name.length() > 0 && current->section == UNION_SEC)
					  {
					  yyless(0);
					  BEGIN(FindMembers);
					  }
					 else
					  {
					  current->type += current->name;
					  current->name = yytext;
					  }
					}

<ClassName>[ \t]*":"[ \t]*		{ current->args = ":" ;
					  BEGIN( Bases ) ;
					}

<Bases,ClassName>[ \t]*"{"[ \t]*	{
					current->file = yyFileName;
					current->startLine = yyLineNr;
					msg("found class ",
					  (const char *)current->name);
					++curlyBracketCount;
					innerCurlyCount = 0;
					BEGIN(Curly);
					}

<Bases>[a-z_A-Z*.<>0-9:]+		{
					current->extends.append(
					  new McString(yytext));
					current->args += ' ';
					current->args += yytext;
					}

<Bases>","				{ current->args += ',' ; }

<Comment>\n				{ current->program += yytext ;
					  yyLineNr++ ;
					}
<Comment>.				{ current->program += yytext ; }
<Comment>.*"*/"				{ current->program += yytext ;
					  BEGIN( Curly ) ;
					}

<FindMembers>([ \t\n]*"///"[ \t]*)|([ \t\n]*"//{{{"[ \t]*) {
					lineCount();
					if(current->doc.length() > 0 ||
					  current->memo.length() > 0)
					  {
					  msg("found commented entry");
					  current->file = yyFileName;
					  current_root->addSubEntry(current);
					  last = current;
					  current = new Entry;
					  }
					current_done = onlyDocs;
					BEGIN(Cxx_Memo);
					}

<Cxx_Memo>.*				{ current->memo += yytext ; }
<Cxx_Memo>\n				{ yyLineNr++ ;
					  BEGIN( FindMembers ) ;
					}

<FindMembers>[ \t\n]*"/*""*"+"/"
<FindMembers>[ \t\n]*"/***"		{ lastCContext = YY_START ;
					  lineCount () ;
					  BEGIN( SkipComment ) ;
					}

<FindMembers>[ \t\n]*("/**"|"/*{{{")[ \t]* { 
					lineCount();
        				if(current->doc.length() > 0 ||
					  current->memo.length() > 0)
					  {
					  msg("found commented entry");
					  current_root->addSubEntry(current);
					  current->file = yyFileName;
					  last = current;
				          current = new Entry;
					  }
					current_done = onlyDocs;
					BEGIN(Doc);
					}

<VerbDoc,Doc>\n[ \t]*"*"+"/"		{ lineCount() ; BEGIN( FindMembers ) ; }

<Namespace>[a-z_A-Z0-9]*		{
					Entry *find = current_root->sub;
					for(; find; find = find->next)
					  if(find->section == NAMESPACE_SEC)
					    if(strcmp(yytext, find->name) == 0)
					      break;
					if(find)
					  current_root = find;
					 else
                                          {
					  current->name = yytext;
					  msg("found namespace ", yytext);
					  current->file = yyFileName;
					  current_root->addSubEntry(current);
					  current_root = current;
					  last = current;
					  current = new Entry;
					  }
					}

<Namespace>"{"				{
					++curlyBracketCount;
					BEGIN(FindMembers);
					}

<UsingNamespace>[a-z_A-Z0-9]*		{
					Entry *find = current_root->sub;
					for(; find; find = find->next)
					    if(find->section == NAMESPACE_SEC &&
						(strcmp(yytext, find->name) == 0))
						break;
					if(find)
					    {
					    // add the namespace to the lookup
					    // table, which will tell us where
					    // to look when a identifier isn't
					    // known
					    namespace_entry *n = new namespace_entry;
					    n->name = new char[strlen(yytext) + 1];
					    strcpy(n->name, yytext);
					    n->innerCurlyCount = innerCurlyCount;
					    namespace_table.append(n);
#ifdef DEBUG
					    printf("`%s' appended to lookup table (innerCurlyCount = %d)\n",
						yytext, innerCurlyCount);
#endif
					    }
					else
					    fprintf(stderr, "Warning: unknown namespace `%s', ignoring `using' keyword\n", yytext);
					}

<UsingNamespace>;			{
					BEGIN(FindMembers);
					}

<Doc>"\\begin{verbatim}"		{ current->doc += yytext ;
					  BEGIN( VerbDoc ) ;
					}
<VerbDoc>\n				{ current->doc += '\n' ;
					  yyLineNr++ ;
					}
<VerbDoc>"\\end{verbatim}"		{ current->doc += yytext ;
					  BEGIN( Doc ) ;
					}
<Doc>\n[ \t]*"*"*[ \t]*			{ current->doc += '\n' ;
					  yyLineNr++ ;
					}
<VerbDoc,Doc>.				{ current->doc += *yytext ; }
<VerbDoc,Doc>"//"			{ current->doc += yytext ; }
<VerbDoc,Doc>"/*"			{ current->doc += yytext ; }
<VerbDoc,Doc>"*/"			{ BEGIN( FindMembers ) ; }

<FindMembers>("//@{".*\n)|("/*@{"[^*]*\*+"/")	{ lineCount() ;
					  current->file = yyFileName ;
					  current->startLine = yyLineNr ;
					  innerCurlyCount = 0;
					  BEGIN( SubDoc ) ;
					}
<SubDoc>"/*"				{ current->program += yytext ;
					  BEGIN( SubDocComment ) ;
					}
<SubDoc>"//"				{ current->program += yytext ;
					  BEGIN( SubDocCppComment ) ;
					}
<SubDoc>.				{ current->program += *yytext ; }
<SubDoc>\n				{ current->program += *yytext ;
					  ++yyLineNr ;
					}
<SubDoc>("//@{".*\n)|("/*@{"[^*]*\*+"/")	{
					lineCount();
					current->program += yytext;
					++curlyBracketCount;
					++innerCurlyCount;
					}

<SubDoc>("//@}".*\n)|("/*@}"[^*]*\*+"/") {
					lineCount();
					--curlyBracketCount;
					if(innerCurlyCount)
					  {
					  current->program += yytext;
					  --innerCurlyCount;
					  }
					 else
					  {
					  msg("found explicit subentry");
					  current_root->addSubEntry(current);
					  current_done = 0;
					  current = new Entry;
					  current->protection = protection;
					  BEGIN(FindMembers);
					  }
					}

<SubDocComment,SubDocCppComment>"/*"	{ current->program += yytext ; }
<SubDocComment,SubDocCppComment>"//"	{ current->program += yytext ; }
<SubDocComment>.			{ current->program += yytext ; }
<SubDocComment>\n			{ current->program += yytext ;
					  ++yyLineNr ;
					}
<SubDocComment>"*/"			{ current->program += yytext ;
					  BEGIN( SubDoc ) ;
					}

<SubDocCppComment>.			{ current->program += yytext ; }
<SubDocCppComment>\n			{ current->program += yytext ;
					  ++yyLineNr ;
					  BEGIN( SubDoc ) ;
					}

<SkipComment>"//"
<SkipComment>[ \t]*"*/"			{ BEGIN( lastCContext ) ; }
<*>"//"					{ lastCContext = YY_START ;
					  BEGIN( SkipCxxComment ) ;
					}
<SkipCxxComment>.*\n			{ yyLineNr++ ;
					  BEGIN( lastCContext ) ;
					}

<*>.
<*>\n					{ yyLineNr++ ; }
<*>"/*"					{ lastCContext = YY_START ;
					  BEGIN( SkipComment ) ;
					}
%%

void callcppYYlex()
{
    cppYYlex() ;
    if( (current->name.length()  ||  current->program.length()  ||
	 current->memo.length()  ||  current->doc.length() )
    &&  current_done == onlyDocs )
    {
	if( current->section == EMPTY_SEC )
	    current->section = VARIABLE_SEC ;
	current_root->addSubEntry( current ) ;
	current_done = 0;
	last = current ;
	current->file = yyFileName ;
	current = new Entry ;
	current->protection = protection ;
    }
}

void	parseCppClasses(Entry* rt)
{
    extern void parseDoc(Entry *rt);
    if( rt == 0 )
	return ;
    for( Entry *cr = rt->sub ; cr ; cr = cr->next )
    {
#ifdef DEBUG
	if (verb)
	{
	    printf("Scanning `%s %s':\n", (const char *)cr->type,
	      (const char *)cr->name);
	    cr->dump(stdout) ;
	}
#endif
	if(cr->program.length())
	{
#ifdef DEBUG
	    if (verb)
		printf("Scanning substuff of `%s %s':\n",
		       (const char *)cr->type,
		       (const char *)cr->name);
#endif
	    inputString   = cr->program ;
	    inputPosition = 0 ;
	    cppYYrestart( cppYYin ) ;
	    BEGIN( FindMembers ) ;
	    current_root = cr ;
	    strcpy( yyFileName, cr->file ) ;
	    yyLineNr = cr->startLine ;
	    current->type.clear() ;
	    current->name.clear() ;
	    current->args.clear() ;
	    current->section = EMPTY_SEC ;
	    if( cr->section == CLASS_SEC )
		current->protection = protection = PRIV ;
	    else
		current->protection = protection = PUBL ;
	    callcppYYlex() ;
	    cr->program.clear() ;
	}
        parseCppClasses( cr ) ;
    }
    parseDoc( rt ) ;
}

void	parseCpp(Entry* rt)
{
    assert( rt ) ;

    current_root = rt ;
    global_root  = rt ;
    protection   = PUBL ;
    current      = new Entry ;
    last	 = current ;

    inputString   = rt->program ;
    inputPosition = 0 ;
    cppYYrestart( cppYYin ) ;
    BEGIN( FindMembers ) ;
    callcppYYlex() ;
    rt->program.clear() ;
    parseCppClasses( rt ) ;

    delete current ;
}

extern "C" {
  int cppYYwrap()
    {
    return 1;
    }
};
